/*!
******************************************************************************

	@file	ohci.h

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#ifndef __USB_HOST_OHCI_H__
#define __USB_HOST_OHCI_H__

#include "usb.h"

#pragma pack(1)

#define OHCI_HOST_MAX	16
#define OHCI_PORT_MAX	15

// Host Controller Operational Registers
typedef volatile struct {
	u32		HcRevision;
	u32		HcControl;
	u32		HcCommandStatus;
	u32		HcInterruptStatus;
	u32		HcInterruptEnable;
	u32		HcInterruptDisable;
	u32		HcHCCA;
	u32		HcPeriodCurrentED;
	u32		HcControlHeadED;
	u32		HcControlCurrentED;
	u32		HcBulkHeadED;
	u32		HcBulkCurrentED;
	u32		HcDoneHead;
	u32		HcFmInterval;
	u32		HcFmRemaining;
	u32		HcFmNumber;
	u32		HcPeriodicStart;
	u32		HcLSThreshold;
	u32		HcRhDescriptorA;
	u32		HcRhDescriptorB;
	u32		HcRhStatus;
	u32		HcRhPortStatus[1];
	u8		reserved[168];
	u32		HceControl;
	u32		HceInput;
	u32		HceOutput;
	u32		HceStatus;
} OHCI_REGS;

#define OHCI_NULL_ED_PTR		0x00000001	// bit0=1: Halted

// HcControl
#define OHCI_REG_CTRL_CBSR		0x00000003	// ControlBulkServiceRatio
#define OHCI_REG_CTRL_PLE		0x00000004	// PeriodicListEnable
#define OHCI_REG_CTRL_IE		0x00000008	// IsochronousEnable
#define OHCI_REG_CTRL_CLE		0x00000010	// ControlListEnable
#define OHCI_REG_CTRL_BLE		0x00000020	// BulkListEnable
#define OHCI_REG_CTRL_HCFS		0x000000C0	// HostControllerFunctionalState
#define OHCI_REG_CTRL_IR		0x00000100	// InterruptRouting
#define OHCI_REG_CTRL_RWC		0x00000200	// RemoteWakeupConnected
#define OHCI_REG_CTRL_RWE		0x00000400	// RemoteWakeupEnable

#define OHCI_STATE_MASK			OHCI_REG_CTRL_HCFS
#define OHCI_STATE(r)			(ohci_read32( &(r)->HcControl ) & OHCI_STATE_MASK)
#define OHCI_STATE_RESET		0x00000000	// UsbReset
#define OHCI_STATE_RESUME		0x00000040	// UsbResume
#define OHCI_STATE_OPERATIONAL	0x00000080	// UsbOperational
#define OHCI_STATE_SUSPEND		0x000000C0	// UsbSuspend

// HcCommandStatus
#define OHCI_REG_CMDST_HCR		0x00000001	// HostControllerReset
#define OHCI_REG_CMDST_CLF		0x00000002	// ControlListFilled
#define OHCI_REG_CMDST_BLF		0x00000004	// BulkListFilled
#define OHCI_REG_CMDST_OCR		0x00000008	// OwnershipChangeRequest
#define OHCI_REG_CMDST_SOC		0x00030000	// SchedulingOverrunCount

// HcInterruptStatus
// HcInterruptEnable
// HcInterruptDisable
#define OHCI_REG_INT_SO			0x00000001	// SchedulingOverrun
#define OHCI_REG_INT_WDH		0x00000002	// WritebackDoneHead
#define OHCI_REG_INT_SF			0x00000004	// StartofFrame
#define OHCI_REG_INT_RD			0x00000008	// ResumeDetected
#define OHCI_REG_INT_UE			0x00000010	// UnrecoverableError
#define OHCI_REG_INT_FNO		0x00000020	// FrameNumberOverflow
#define OHCI_REG_INT_RHSC		0x00000040	// RootHubStatusChange
#define OHCI_REG_INT_OC			0x40000000	// OwnershipChange
#define OHCI_REG_INT_MIE		0x80000000	// MasterInterruptEnable
#define OHCI_REG_INT_ALL		0x4000007F

// HcFmInterval
#define OHCI_FRAME_INTERVAL(d)	((d) & 0x00003FFF)

// HcRhDescriptorA
#define OHCI_REG_RHDA_NDP		0x000000FF	// NumberDownstreamPorts
#define OHCI_REG_RHDA_PSM		0x00000100	// PowerSwitchingMode
#define OHCI_REG_RHDA_NPS		0x00000200	// NoPowerSwitching
#define OHCI_REG_RHDA_DT		0x00000400	// DeviceType
#define OHCI_REG_RHDA_OCPM		0x00000800	// OverCurrentProtectionMode
#define OHCI_REG_RHDA_NOCP		0x00001000	// NoOverCurrentProtection
#define OHCI_REG_RHDA_POTPGT	0xFF000000	// PowerOnToPowerGoodTime

#define OHCI_RH_PORT_NUM(d)		((d) & OHCI_REG_RHDA_NDP)
#define OHCI_RH_POTPGT(d)		((d) >> 24)

// HcRhStatus
#define OHCI_REG_RHST_LPS		0x00000001	// LocalPowerStatus
#define OHCI_REG_RHST_OCI		0x00000002	// OverCurrentIndicator
#define OHCI_REG_RHST_DRWE		0x00008000	// DeviceRemoteWakeupEnable
#define OHCI_REG_RHST_LPSC		0x00010000	// LocalPowerStatusChange
#define OHCI_REG_RHST_OCIC		0x00020000	// OverCurrentInidicatorChange
#define OHCI_REG_RHST_CRWE		0x80000000	// ClearRemoteWakeupEnable

// HcRhPortStatus
#define OHCI_REG_PRTST_CCS		0x00000001	// CurrentConnectStatus
#define OHCI_REG_PRTST_PES		0x00000002	// PortEnableStatus
#define OHCI_REG_PRTST_PSS		0x00000004	// PortSuspendStatus
#define OHCI_REG_PRTST_POCI		0x00000008	// PortOverCurrentIndicator
#define OHCI_REG_PRTST_PRS		0x00000010	// PortResetStatus
#define OHCI_REG_PRTST_PPS		0x00000100	// PortPowerStatus
#define OHCI_REG_PRTST_LSDA		0x00000200	// LowSpeedDeviceAttached
#define OHCI_REG_PRTST_CSC		0x00010000	// ConnectStatusChange
#define OHCI_REG_PRTST_PESC		0x00020000	// PortEnableStatusChange
#define OHCI_REG_PRTST_PSSC		0x00040000	// PortSuspendStatusChange
#define OHCI_REG_PRTST_OCIC		0x00080000	// PortOverCurrentIndicatorChange
#define OHCI_REG_PRTST_PRSC		0x00100000	// PortResetStatusChange

// Host Controller Communications Area
typedef volatile struct {
	u32		HccaInterruptTable[32];
	u16		HccaFrameNumber;
	u16		HccaPad1;
	u32		HccaDoneHead;
	u8		rsvd[120];
} OHCI_HCCA;

/*
// Host Controller Endpoint Descriptor
typedef volatile struct {
	u32		Flags;
	u32		TailP;
	u32		HeadP;
	u32		NextED;
} OHCI_HCED;

#define OHCI_ED_IS_HALTED(ed)	(ohci_read32( &(ed)->HeadP ) & 0x01)
#define OHCI_ED_CARRY(ed)		((ohci_read32( &(ed)->HeadP ) & 0x02)>>1)
*/
#define OHCI_ED_IS_HALTED(ed)	(ohci_read32( &(ed)->ohci.head ) & 0x01)

#define OHCI_ED_DIR_FROM_TD		0
#define OHCI_ED_DIR_OUT			1
#define OHCI_ED_DIR_IN			2

/*
// Host Controller Transfer Descriptor
typedef union {
	// General Transfer Descriptor (ED.Format=0)
	volatile struct {
		u32		Flags;
		u32		CurBufPtr;
		u32		NextTD;
		u32		BufferEnd;
	} gen;

	// Ishochronous Transfer Descriptor (ED.Format=1)
	volatile struct {
		u32		Flags;
		u32		BufferPage0;
		u32		NextTD;
		u32		BufferEnd;
		union {
			u16		Offset[8];
			u16		PacketStatusWord[8];
		};
	} iso;
} OHCI_HCTD;
*/
#define OHCI_TD_PID_SETUP			0
#define OHCI_TD_PID_OUT				1
#define OHCI_TD_PID_IN				2

#define OHCI_TD_TOGGLE_AUTO			0
#define OHCI_TD_DATA0				2
#define OHCI_TD_DATA1				3

#define OHCI_TD_CC_NO_ERROR			0
#define OHCI_TD_CC_CRC				1
#define OHCI_TD_CC_BIT_STUFFING		2
#define OHCI_TD_CC_TOGGLE_MISMATCH	3
#define OHCI_TD_CC_STALL			4
#define OHCI_TD_CC_DEV_NOT_RESP		5
#define OHCI_TD_CC_PID_CHK_FAIL		6
#define OHCI_TD_CC_UNEXPECTED_PID	7
#define OHCI_TD_CC_DATA_OVERRUN		8
#define OHCI_TD_CC_DATA_UNDERRUN	9
#define OHCI_TD_CC_BUF_OVERRUN		12
#define OHCI_TD_CC_BUF_UNDERRUN		13
#define OHCI_TD_CC_NOT_ACCESSED		15

typedef struct {
	union {
		u32		flags;
		union {
			struct {
				u32		rsvd			: 18;
				u32		buf_round		: 1;
				u32		dir				: 2;
				u32		delay_int		: 3;
				u32		toggle			: 2;
				u32		error_cnt		: 2;
				u32		cond_code		: 4;
			} gen;
			struct {
				u32		start_frame		: 16;
				u32		rsvd1			: 5;
				u32		delay_int		: 3;
				u32		frame_cnt		: 3;
				u32		reserved2		: 1;
				u32		cond_code		: 4;
			} iso;
		} f;
	};
	USB_HCTD *	hctd;
	void *		buf;
	size_t		buf_len;
} OHCI_TD;

typedef struct _ohci_td_entry {
	struct _ohci_td_entry *	prev;
	struct _ohci_td_entry *	next;
	OHCI_TD *				td;
} OHCI_TD_ENTRY;

typedef struct {
	union {
		u32		flags;
		struct {
			u32		func_addr		: 7;
			u32		endpoint		: 4;
			u32		dir				: 2;
			u32		speed			: 1;
			u32		skip			: 1;
			u32		format			: 1;
			u32		max_pkt_size	: 11;
			u32		rsvd			: 5;
		} f;
	};
	USB_HCQH *		hcqh;
	OHCI_TD_ENTRY *	td_list;
} OHCI_ED;

typedef struct _ohci_ed_entry {
	struct _ohci_ed_entry *	prev;
	struct _ohci_ed_entry *	next;
	OHCI_ED *				ed;
} OHCI_ED_ENTRY;

typedef struct {
	u8				id;
	u8				ports;
	u8				power_wait;
	bool			legacy_support;
	USB_DEVICE *	dev;
	OHCI_REGS *		regs;
	OHCI_HCCA *		hcca;
	OHCI_ED_ENTRY *	ctrl_ed_list;
	OHCI_ED_ENTRY *	bulk_ed_list;
	OHCI_ED *		int_ed[32];
} OHCI_HCD;

#pragma pack()

// ohci.cpp
extern bool ohci_init( void );
extern bool ohci_probe( PCI_DEVICE *dev );
extern bool ohci_start( void );
extern u16  ohci_read16( volatile u16 *p );
extern u32  ohci_read32( volatile u32 *p );
extern void ohci_write16( volatile u16 *p, u16 data );
extern void ohci_write32( volatile u32 *p, u32 data );

// ohci_hub.cpp
extern bool ohci_init_hub( OHCI_HCD *hcd );

// ohci_q.cpp
extern bool ohci_alloc_queue( USB_DEVICE *dev, u8 ep, size_t buf_len );
extern bool ohci_ctrl_xfer( USB_DEVICE *dev, u8 req_type, u8 request, u16 value, u16 index,
							void *buf, size_t buf_len, int timeout );
extern bool ohci_bulk_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );
extern bool ohci_interrupt_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );

#endif // !__USB_HOST_OHCI_H__
