/*!
******************************************************************************

	@file	ohci_q.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "pci.h"
#include "ohci.h"
#include "timer.h"
#include "printf.h"

static OHCI_ED_ENTRY ohci_ed_entry[USB_DEVICE_MAX][USB_HCQH_MAX];
static OHCI_TD_ENTRY ohci_td_entry[USB_DEVICE_MAX][USB_HCTD_MAX];
static OHCI_ED		 ohci_ed[USB_DEVICE_MAX][USB_HCQH_MAX];
static OHCI_TD		 ohci_td[USB_DEVICE_MAX][USB_HCTD_MAX];

static bool ohci_start_ctrl_xfer( USB_DEVICE *dev, USB_REQUEST *req );
static bool ohci_start_bulk_xfer( USB_DEVICE *dev, u8 ep, size_t buf_len );
static bool ohci_start_interrupt_xfer( USB_DEVICE *dev, u8 ep, size_t buf_len );

static void ohci_print_ctrl_xfer_result( OHCI_HCD *, USB_HCQH *, USB_HCTD *, USB_HCTD *, USB_HCTD * );
static void ohci_print_bulk_xfer_result( OHCI_HCD *, USB_HCQH *, USB_HCTD * );
static void ohci_print_interrupt_xfer_result( OHCI_HCD *, USB_HCQH *, USB_HCTD * );

bool ohci_alloc_queue( USB_DEVICE *dev, u8 ep, size_t buf_len )
{
	(void)buf_len;

	const u8 func_addr = dev->func_addr;
	if ( func_addr >= USB_DEVICE_MAX )
		return false;

	OHCI_ED_ENTRY *ed_entry = &ohci_ed_entry[func_addr][ep];
	OHCI_ED		  *ed		= &ohci_ed[func_addr][ep];
	memset( ed, 0, sizeof(OHCI_ED) );
	ed->hcqh		   = &((USB_HCQH *)(USB_HCQHBUF + USB_HCQH_ALIGN * USB_HCQH_MAX * func_addr))[ep];
	ed->f.func_addr	   = func_addr;
	ed->f.endpoint	   = ep;
	ed->f.speed		   = dev->low_speed? 1:0;
	ed->f.max_pkt_size = dev->ep[ep].max_pkt_size;
	ed->f.skip		   = 1;
	ed_entry->ed   = ed;
	ed_entry->prev = NULL;
	ed_entry->next = NULL;

	OHCI_TD_ENTRY *td_entry = ohci_td_entry[func_addr];
	OHCI_TD *td;
	USB_HCTD *hctd = (USB_HCTD *)(USB_HCTDBUF + USB_HCTD_ALIGN * USB_HCTD_MAX * func_addr);
	if ( ep == 0 )
	{	// コントロール転送用
		// TD(NULL)
		td = &ohci_td[func_addr][USB_HCTD_NULL];
		memset( td, 0, sizeof(OHCI_TD) );
		td->hctd = &hctd[USB_HCTD_NULL];
		td_entry[USB_HCTD_NULL].td = td;
		// TD(SETUP)
		td = &ohci_td[func_addr][USB_HCTD_CTRL_SETUP];
		memset( td, 0, sizeof(OHCI_TD) );
		td->hctd = &hctd[USB_HCTD_CTRL_SETUP];
		td_entry[USB_HCTD_CTRL_SETUP].td = td;
		// TD(DATA)
		td = &ohci_td[func_addr][USB_HCTD_CTRL_DATA];
		memset( td, 0, sizeof(OHCI_TD) );
		td->hctd = &hctd[USB_HCTD_CTRL_DATA];
		td_entry[USB_HCTD_CTRL_DATA].td = td;
		// TD(STATUS)
		td = &ohci_td[func_addr][USB_HCTD_CTRL_STATUS];
		memset( td, 0, sizeof(OHCI_TD) );
		td->hctd = &hctd[USB_HCTD_CTRL_STATUS];
		td_entry[USB_HCTD_CTRL_STATUS].td = td;
		ed->td_list	= &td_entry[USB_HCTD_CTRL_SETUP];
	}
	else
	{	// バルク転送/インタラプト転送用
		td = &ohci_td[func_addr][ep];
		memset( td, 0, sizeof(OHCI_TD) );
		td->hctd = &hctd[ep];
		td->f.gen.toggle = OHCI_TD_DATA0;
		if ( dev->ep[ep].addr & 0x80 ) {
			td->f.gen.buf_round = 1;
			td->f.gen.dir = OHCI_TD_PID_IN;
		}
		else {
			td->f.gen.buf_round = 0;
			td->f.gen.dir = OHCI_TD_PID_OUT;
		}
		td_entry[ep].td = td;
		ed->td_list	= &td_entry[ep];
	}

	return true;
}

static bool ohci_build_ed( OHCI_ED_ENTRY *ed_entry )
{
	if ( ed_entry == NULL )
		return false;

	OHCI_ED *ed = ed_entry->ed;
	if ( ed == NULL )
		return false;
	OHCI_ED_ENTRY *ed_next = ed_entry->next;
	if ( ed_next != NULL ) {
		if ( ed_next->ed == NULL )
			return false;
		if ( ed_next->ed->hcqh == NULL )
			return false;
	}

	OHCI_TD_ENTRY *td_head = ed->td_list;
	if ( td_head == NULL )
		return false;
	OHCI_TD_ENTRY *td_tail = td_head->prev;
	if ( td_tail == NULL )
		return false;
	if ( (td_head->td == NULL) || (td_tail->td == NULL) )
		return false;
	if ( (td_head->td->hctd == NULL) || (td_tail->td->hctd == NULL) )
		return false;

	USB_HCQH *hcqh = ed->hcqh;
	if ( hcqh == NULL )
		return false;
	hcqh->ohci.flags = ed->flags;
	hcqh->ohci.head  = virt_to_phys( (void *)td_head->td->hctd );
	hcqh->ohci.tail  = virt_to_phys( (void *)td_tail->td->hctd );
	if ( ed_next == NULL )
		hcqh->ohci.next = OHCI_NULL_ED_PTR;
	else
		hcqh->ohci.next = virt_to_phys( (void *)ed_next->ed->hcqh );

	return true;
}

static bool ohci_build_gen_td( OHCI_TD_ENTRY *td_entry, void *buf, size_t buf_len )
{
	if ( td_entry == NULL )
		return false;

	OHCI_TD *td = td_entry->td;
	OHCI_TD_ENTRY *td_next = td_entry->next;
	if ( (td == NULL) || (td_next == NULL) )
		return false;
	if ( td_next->td == NULL )
		return false;
	if ( td_next->td->hctd == NULL )
		return false;

	USB_HCTD *hctd = td->hctd;
	if ( hctd == NULL )
		return false;
	hctd->ohci.gen.flags = td->flags;
	hctd->ohci.gen.next  = virt_to_phys( (void *)td_next->td->hctd );
	if ( (buf == NULL) || (buf_len == 0) ) {
		hctd->ohci.gen.buf_ptr = 0;
		hctd->ohci.gen.buf_end = 0;
	}
	else {
		const u32 buf_pa = virt_to_phys( buf );
		hctd->ohci.gen.buf_ptr = buf_pa;
		hctd->ohci.gen.buf_end = buf_pa + buf_len - 1;
	}

	return true;
}

bool ohci_ctrl_xfer( USB_DEVICE *dev, u8 req_type, u8 request,
					 u16 value, u16 index, void *buf, size_t buf_len, int timeout )
{
	OHCI_HCD *hcd = (OHCI_HCD *)dev->hcd_priv;

	const u8 func_addr = dev->func_addr;
	OHCI_ED *ed		   = &ohci_ed[func_addr][0];
	OHCI_TD *td_setup  = &ohci_td[func_addr][USB_HCTD_CTRL_SETUP];
	OHCI_TD *td_data   = &ohci_td[func_addr][USB_HCTD_CTRL_DATA];
	OHCI_TD *td_status = &ohci_td[func_addr][USB_HCTD_CTRL_STATUS];

	if ( timeout < 0 )
		timeout = 0x7FFFFFFF;

	USB_REQUEST *req = (USB_REQUEST *)dev->ep[0].buf;
	req->bmRequestType = req_type;
	req->bRequest	   = request;
	req->wValue		   = value;
	req->wIndex		   = index;
	req->wLength	   = buf_len;

	// ED/TDを準備する
	ed->f.skip = 0;
	td_setup ->f.gen.toggle	   = OHCI_TD_DATA0;
	td_data  ->f.gen.toggle	   = OHCI_TD_DATA1;
	td_status->f.gen.toggle	   = OHCI_TD_DATA1;
	td_setup ->f.gen.cond_code = OHCI_TD_CC_NOT_ACCESSED;
	td_data  ->f.gen.cond_code = OHCI_TD_CC_NOT_ACCESSED;
	td_status->f.gen.cond_code = OHCI_TD_CC_NOT_ACCESSED;
	if ( req_type & USB_REQ_DIR_IN ) {
		td_data  ->f.gen.buf_round = 1;
		td_setup ->f.gen.dir = OHCI_TD_PID_SETUP;
		td_data  ->f.gen.dir = OHCI_TD_PID_IN;
		td_status->f.gen.dir = OHCI_TD_PID_OUT;
	}
	else if ( buf_len > 0 ) {
		memcpy( req+1, buf, buf_len );
		td_data  ->f.gen.buf_round = 0;
		td_setup ->f.gen.dir = OHCI_TD_PID_SETUP;
		td_data  ->f.gen.dir = OHCI_TD_PID_OUT;
		td_status->f.gen.dir = OHCI_TD_PID_IN;
	}
	else {
		td_setup ->f.gen.dir = OHCI_TD_PID_SETUP;
		td_status->f.gen.dir = OHCI_TD_PID_IN;
	}

	// コントロール転送を開始し、終了するまで待つ
	if ( !ohci_start_ctrl_xfer( dev, req ) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): ohci_start_ctrl_xfer() failed.\n", hcd->id );
		return false;
	}
	USB_HCQH *hcqh = ed->hcqh;
	USB_HCTD *hctd_status = td_status->hctd;
	while ( timeout > 0 ) {
		if ( OHCI_ED_IS_HALTED( hcqh ) )
			break;
		if ( ohci_read32( &hcqh->ohci.tail ) == (ohci_read32( &hcqh->ohci.head ) & 0xFFFFFFF0) )
			break;
		timer_usleep( 1000 );
		timeout--;
	}
	td_status->flags = ohci_read32( &hctd_status->ohci.gen.flags );

	if ( (timeout <= 0) || (td_status->f.gen.cond_code != OHCI_TD_CC_NO_ERROR) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): Control Transfer(s) failed.\n", hcd->id );
		ohci_print_ctrl_xfer_result( hcd, hcqh, td_setup->hctd, td_data->hctd, hctd_status );
		return false;
	}

	if ( req_type & USB_REQ_DIR_IN )
		memcpy( buf, req+1, buf_len );

	return true;
}

bool ohci_bulk_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	OHCI_HCD *hcd = (OHCI_HCD *)dev->hcd_priv;

	const u8 func_addr = dev->func_addr;
	OHCI_ED *ed = &ohci_ed[func_addr][ep];
	OHCI_TD *td = &ohci_td[func_addr][ep];

	if ( timeout < 0 )
		timeout = 0x7FFFFFFF;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !(endp->addr & 0x80) )
		memcpy( endp->buf, buf, buf_len );

	// ED/TDを準備する
	ed->f.skip = 0;
	td->f.gen.cond_code	= OHCI_TD_CC_NOT_ACCESSED;

	// バルク転送を開始し、終了するまで待つ
	if ( !ohci_start_bulk_xfer( dev, ep, buf_len ) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): ohci_start_bulk_xfer() failed.\n", hcd->id );
		return false;
	}
	USB_HCQH *hcqh = ed->hcqh;
	USB_HCTD *hctd = td->hctd;
	while ( timeout > 0 ) {
		if ( OHCI_ED_IS_HALTED( hcqh ) )
			break;
		if ( ohci_read32( &hcqh->ohci.tail ) == (ohci_read32( &hcqh->ohci.head ) & 0xFFFFFFF0) )
			break;
		timer_usleep( 1000 );
		timeout--;
	}
	td->flags = ohci_read32( &hctd->ohci.gen.flags );

	if ( (timeout <= 0) || (td->f.gen.cond_code != OHCI_TD_CC_NO_ERROR) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): Bulk Transfer(s) failed.\n", hcd->id );
		ohci_print_bulk_xfer_result( hcd, hcqh, hctd );
		return false;
	}

	if ( endp->addr & 0x80 )
		memcpy( buf, endp->buf, buf_len );

//	timer_sleep( dev->low_speed? dev->ep[0].interval:dev->ep[0].interval>>3 );
//	timer_usleep( 1000 );

	return true;
}

bool ohci_interrupt_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	OHCI_HCD *hcd = (OHCI_HCD *)dev->hcd_priv;

	const u8 func_addr = dev->func_addr;
	OHCI_ED *ed = &ohci_ed[func_addr][ep];
	OHCI_TD *td = &ohci_td[func_addr][ep];

	if ( timeout < 0 )
		timeout = 0x7FFFFFFF;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !(endp->addr & 0x80) )
		memcpy( endp->buf, buf, buf_len );

	// ED/TDを準備する
	ed->f.skip = 0;
	td->f.gen.cond_code	= OHCI_TD_CC_NOT_ACCESSED;

	// インタラプト転送を開始し、終了するまで待つ
	if ( !ohci_start_interrupt_xfer( dev, ep, buf_len ) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): ohci_start_interrupt_xfer() failed.\n", hcd->id );
		return false;
	}
	USB_HCQH *hcqh = ed->hcqh;
	USB_HCTD *hctd = td->hctd;
	while ( timeout > 0 ) {
		if ( OHCI_ED_IS_HALTED( hcqh ) )
			break;
		if ( ohci_read32( &hcqh->ohci.tail ) == (ohci_read32( &hcqh->ohci.head ) & 0xFFFFFFF0) )
			break;
		timer_usleep( 1000 );
		timeout--;
	}
	td->flags = ohci_read32( &hctd->ohci.gen.flags );

	if ( (timeout <= 0) || (td->f.gen.cond_code != OHCI_TD_CC_NO_ERROR) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): Interrupt Transfer(s) failed.\n", hcd->id );
		ohci_print_interrupt_xfer_result( hcd, hcqh, hctd );
		return false;
	}

	if ( endp->addr & 0x80 )
		memcpy( buf, endp->buf, buf_len );

	//	timer_sleep( dev->low_speed? dev->ep[0].interval:dev->ep[0].interval>>3 );
//	timer_usleep( 1000 );

	return true;
}

static bool ohci_start_ctrl_xfer( USB_DEVICE *dev, USB_REQUEST *req )
{
	OHCI_HCD  *hcd  = (OHCI_HCD *)dev->hcd_priv;
	OHCI_HCCA *hcca = hcd->hcca;
	OHCI_REGS *regs = hcd->regs;

	const u8 func_addr = dev->func_addr;

	// TDを準備する
	OHCI_TD_ENTRY *td_entry_null   = &ohci_td_entry[func_addr][USB_HCTD_NULL];
	OHCI_TD_ENTRY *td_entry_setup  = &ohci_td_entry[func_addr][USB_HCTD_CTRL_SETUP];
	OHCI_TD_ENTRY *td_entry_data   = &ohci_td_entry[func_addr][USB_HCTD_CTRL_DATA];
	OHCI_TD_ENTRY *td_entry_status = &ohci_td_entry[func_addr][USB_HCTD_CTRL_STATUS];
	td_entry_setup->prev = td_entry_null;
	if ( req->wLength > 0 )
	{	// SETUP->DATA(IN/OUT)->STATUS
		td_entry_setup ->next = td_entry_data;
		td_entry_data  ->prev = td_entry_setup;
		td_entry_data  ->next = td_entry_status;
		td_entry_status->prev = td_entry_data;
	}
	else
	{	// SETUP->STATUS(IN)
		td_entry_setup ->next = td_entry_status;
		td_entry_status->prev = td_entry_setup;
	}
	td_entry_status->next = td_entry_null;
	td_entry_null  ->prev = td_entry_status;
	td_entry_null  ->next = td_entry_setup;
	if ( !ohci_build_gen_td( td_entry_setup, req, sizeof(USB_REQUEST) ) )
		return false;
	if ( !ohci_build_gen_td( td_entry_data, req+1, req->wLength ) )
		return false;
	if ( !ohci_build_gen_td( td_entry_status, NULL, 0 ) )
		return false;

	// EDを準備する
	OHCI_ED_ENTRY *ed_entry = &ohci_ed_entry[func_addr][0];
	if ( !ohci_build_ed( ed_entry ) )
		return false;

	// コントロール転送を中断する
	volatile u32 ctrl_reg = ohci_read32( &regs->HcControl );
	ohci_write32( &regs->HcControl, ctrl_reg & ~OHCI_REG_CTRL_CLE );

	// レジスタを設定する
	const u32 ed_pa = virt_to_phys( (void *)ed_entry->ed->hcqh );
	ohci_write32( &regs->HcDoneHead,		 0 );
	ohci_write32( &hcca->HccaDoneHead,		 0 );
	ohci_write32( &regs->HcControlHeadED,	 ed_pa );
	ohci_write32( &regs->HcControlCurrentED, ed_pa );
	ohci_write32( &regs->HcCommandStatus,	 OHCI_REG_CMDST_CLF );

	// コントロール転送を再開する
	ohci_write32( &regs->HcControl, ctrl_reg | OHCI_REG_CTRL_CLE );

	return true;
}

static bool ohci_start_bulk_xfer( USB_DEVICE *dev, u8 ep, size_t buf_len )
{
	OHCI_HCD  *hcd  = (OHCI_HCD *)dev->hcd_priv;
	OHCI_HCCA *hcca = hcd->hcca;
	OHCI_REGS *regs = hcd->regs;

	const u8 func_addr = dev->func_addr;

	// TDを準備する
	OHCI_TD_ENTRY *td_entry_bulk = &ohci_td_entry[func_addr][ep];
	OHCI_TD_ENTRY *td_entry_null = &ohci_td_entry[func_addr][USB_HCTD_NULL];
	td_entry_bulk->prev = td_entry_null;
	td_entry_bulk->next = td_entry_null;
	td_entry_null->prev = td_entry_bulk;
	td_entry_null->next = td_entry_bulk;
	if ( !ohci_build_gen_td( td_entry_bulk, dev->ep[ep].buf, buf_len ) )
		return false;

	// EDを準備する
	OHCI_ED_ENTRY *ed_entry = &ohci_ed_entry[func_addr][ep];
	if ( !ohci_build_ed( ed_entry ) )
		return false;

	// バルク転送を中断する
	volatile u32 ctrl_reg = ohci_read32( &regs->HcControl );
	ohci_write32( &regs->HcControl, ctrl_reg & ~OHCI_REG_CTRL_BLE );

	// レジスタを設定する
	const u32 ed_pa = virt_to_phys( (void *)ed_entry->ed->hcqh );
	ohci_write32( &regs->HcDoneHead,	  0 );
	ohci_write32( &hcca->HccaDoneHead,	  0 );
	ohci_write32( &regs->HcBulkHeadED,	  ed_pa );
	ohci_write32( &regs->HcBulkCurrentED, ed_pa );
	ohci_write32( &regs->HcCommandStatus, OHCI_REG_CMDST_BLF );

	// バルク転送を再開する
	ohci_write32( &regs->HcControl, ctrl_reg | OHCI_REG_CTRL_BLE );

	return true;
}

static bool ohci_start_interrupt_xfer( USB_DEVICE *dev, u8 ep, size_t buf_len )
{
	OHCI_HCD  *hcd  = (OHCI_HCD *)dev->hcd_priv;
	OHCI_HCCA *hcca = hcd->hcca;
	OHCI_REGS *regs = hcd->regs;

	const u8 func_addr = dev->func_addr;

	// TDを準備する
	OHCI_TD_ENTRY *td_entry_int  = &ohci_td_entry[func_addr][ep];
	OHCI_TD_ENTRY *td_entry_null = &ohci_td_entry[func_addr][USB_HCTD_NULL];
	td_entry_int->prev  = td_entry_null;
	td_entry_int->next  = td_entry_null;
	td_entry_null->prev = td_entry_int;
	td_entry_null->next = td_entry_int;
	if ( !ohci_build_gen_td( td_entry_int, dev->ep[ep].buf, buf_len ) )
		return false;

	// EDを準備する
	OHCI_ED_ENTRY *ed_entry = &ohci_ed_entry[func_addr][ep];
	if ( !ohci_build_ed( ed_entry ) )
		return false;

	// インタラプト転送を中断する
	volatile u32 ctrl_reg = ohci_read32( &regs->HcControl );
	ohci_write32( &regs->HcControl, ctrl_reg & ~OHCI_REG_CTRL_PLE );

	// レジスタを設定する
	const u32 ed_pa = virt_to_phys( (void *)ed_entry->ed->hcqh );
	ohci_write32( &regs->HcDoneHead,   0 );
	ohci_write32( &hcca->HccaDoneHead, 0 );
	for ( int i=0; i<32; i+=16 )
		ohci_write32( &hcca->HccaInterruptTable[i], ed_pa );

	// インタラプト転送を再開する
	ohci_write32( &regs->HcControl, ctrl_reg | OHCI_REG_CTRL_PLE );

	return true;
}

static void ohci_print_regs( OHCI_HCD *hcd )
{
	OHCI_HCCA *hcca = hcd->hcca;
	OHCI_REGS *regs = hcd->regs;

	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcControl ............ %08x\n",
				hcd->id, ohci_read32( &regs->HcControl ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcCommandStatus ...... %08x\n",
				hcd->id, ohci_read32( &regs->HcCommandStatus ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcInterruptStatus .... %08x\n",
				hcd->id, ohci_read32( &regs->HcInterruptStatus ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhStatus ........... %08x\n",
				hcd->id, ohci_read32( &regs->HcRhStatus ) );
	for ( int i=0; i<hcd->ports; i++ )
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhPortStatus[%d] .... %08x\n",
					hcd->id, i, ohci_read32( &regs->HcRhPortStatus[i] ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcDoneHead ........... %08x\n",
				hcd->id, ohci_read32( &regs->HcDoneHead ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HccaDoneHead ......... %08x\n",
				hcd->id, ohci_read32( &hcca->HccaDoneHead ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcControlCurrentED ... %08x\n",
				hcd->id, ohci_read32( &regs->HcControlCurrentED ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcBulkCurrentED ...... %08x\n",
				hcd->id, ohci_read32( &regs->HcBulkCurrentED ) );
}

static void ohci_print_ctrl_xfer_result( OHCI_HCD *hcd, USB_HCQH *hcqh, USB_HCTD *hctd_setup,
										 USB_HCTD *hctd_data, USB_HCTD *hctd_status )
{
	ohci_print_regs( hcd );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [ED] Flags=%08x, TailP=%08x, HeadP=%08x, NextED=%08x\n",
				hcd->id, hcqh->ohci.flags, hcqh->ohci.tail, hcqh->ohci.head, hcqh->ohci.next );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [TD:S ] Flags=%08x, BufPtr=%08x, NextTD=%08x, BufferEnd=%08x\n",
				hcd->id, hctd_setup->ohci.gen.flags, hctd_setup->ohci.gen.buf_ptr,
				hctd_setup->ohci.gen.next, hctd_setup->ohci.gen.buf_end );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [TD:D ] Flags=%08x, BufPtr=%08x, NextTD=%08x, BufferEnd=%08x\n",
				hcd->id, hctd_data->ohci.gen.flags, hctd_data->ohci.gen.buf_ptr,
				hctd_data->ohci.gen.next, hctd_data->ohci.gen.buf_end );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [TD:ST] Flags=%08x, BufPtr=%08x, NextTD=%08x, BufferEnd=%08x\n",
				hcd->id, hctd_status->ohci.gen.flags, hctd_status->ohci.gen.buf_ptr,
				hctd_status->ohci.gen.next, hctd_status->ohci.gen.buf_end );
}

static void ohci_print_bulk_xfer_result( OHCI_HCD *hcd, USB_HCQH *hcqh, USB_HCTD *hctd )
{
	ohci_print_regs( hcd );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [ED] Flags=%08x, TailP=%08x, HeadP=%08x, NextED=%08x\n",
				hcd->id, hcqh->ohci.flags, hcqh->ohci.tail, hcqh->ohci.head, hcqh->ohci.next );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [TD] Flags=%08x, BufPtr=%08x, NextTD=%08x, BufferEnd=%08x\n",
				hcd->id, hctd->ohci.gen.flags, hctd->ohci.gen.buf_ptr,
				hctd->ohci.gen.next, hctd->ohci.gen.buf_end );
}

static void ohci_print_interrupt_xfer_result( OHCI_HCD *hcd, USB_HCQH *hcqh, USB_HCTD *hctd )
{
	ohci_print_regs( hcd );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [ED] Flags=%08x, TailP=%08x, HeadP=%08x, NextED=%08x\n",
				hcd->id, hcqh->ohci.flags, hcqh->ohci.tail, hcqh->ohci.head, hcqh->ohci.next );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [TD] Flags=%08x, BufPtr=%08x, NextTD=%08x, BufferEnd=%08x\n",
				hcd->id, hctd->ohci.gen.flags, hctd->ohci.gen.buf_ptr,
				hctd->ohci.gen.next, hctd->ohci.gen.buf_end );
}
