/*!
******************************************************************************

	@file	vbe_ddc.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "vbe.h"
#include "vbe_ddc.h"
#include "printf.h"

static u8 vbe_ddc_level;
static VBE_DDC_EDID vbe_ddc_edid;

static bool vbe_ddc_report_capabilities( u8, u8 * );
static bool vbe_ddc_read_edid( u8, u8, VBE_DDC_EDID * );

static void vbe_ddc_print_edid( const VBE_DDC_EDID * );

bool vbe_ddc_init( void )
{
	vbe_ddc_level = 0;
	if ( !vbe_ddc_report_capabilities( 0, &vbe_ddc_level ) )
		return false;
	vmm_printf( VMM_DEBUG, "vbe-ddc: DDC1 %s, DDC2 %s, Screen %s during data transfer\n",
				(vbe_ddc_level & 0x01)? "supported":"not supported",
				(vbe_ddc_level & 0x02)? "supported":"not supported",
				(vbe_ddc_level & 0x04)? "blanked":"not blanked" );

	if ( !vbe_ddc_read_edid( 0, 0, &vbe_ddc_edid ) )
		return false;
	vbe_ddc_print_edid( &vbe_ddc_edid );

	return true;
}

static bool vbe_ddc_report_capabilities( u8 unit, u8 *ddc_level )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax = 0x4F15;	// VBE/DDC Services
	regs.ebx = 0x0000;	// Report DDC Capabilities
	regs.ecx = unit;

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe-ddc: Function 00h failed. (status=%04x)\n", status );
		return false;
	}
	*ddc_level = (u8)regs.ebx;

	return true;
}

static bool vbe_ddc_read_edid( u8 unit, u8 block_num, VBE_DDC_EDID *edid )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax = 0x4F15;	// VBE/DDC Services
	regs.ebx = 0x0001;	// Read EDID
	regs.ecx = unit;
	regs.edx = block_num;
	regs.es_base = edid;
	regs.es_size = sizeof(VBE_DDC_EDID);

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe-ddc: Function 01h failed. (status=%04x)\n", status );
		return false;
	}

	return true;
}

static void vbe_ddc_print_edid( const VBE_DDC_EDID *edid )
{
	static const char *video_iface[] = {
		"undefined",	"DVI",			"HDMI-a",		"HDMI-b",
		"MDDI",			"DisplayPort",	"reserved",		"reserved",
		"reserved",		"reserved",		"reserved",		"reserverd",
		"reserved",		"reserved",		"reserved",		"reserved"
	};
	static const char *video_color[] = {
		"",	", 6 bits color",	", 8 bits color",	", 10 bits color",
			", 12 bits color",	", 14 bits color",	", 16 bits color", ""
	};
	static const char *video_level[] = {
		"0.700:0.300", "0.714:0.286", "1.000:0.400", "0.700:0.000"
	};
	static const char *display_color_fmt[] = {
		"RGB 4:4:4", "RGB 4:4:4 & YCrCb 4:4:4",
		"RGB 4:4:4 & YCrCb 4:2:2", "RGB 4:4:4 & YCrCb 4:4:4 & YCrCb 4:2:2"
	};
	static const char *display_color_type[] = {
		"monochrome", "RGB", "non-RGB", "undefined color type"
	};

	vmm_printf( VMM_DEBUG, "[EDID]\n" );
	vmm_printf( VMM_DEBUG, "Header ....................... %016llx\n", edid->header );
	vmm_printf( VMM_DEBUG, "ID Manufacturer Name ......... %04x\n", edid->manufacturer_id );
	vmm_printf( VMM_DEBUG, "ID Product Code .............. %04x\n", edid->product_code );
	vmm_printf( VMM_DEBUG, "ID Serial Number ............. %08x\n", edid->serial_number );
	vmm_printf( VMM_DEBUG, "Week of Manufacture .......... %02x (%d)\n", edid->week, edid->week );
	vmm_printf( VMM_DEBUG, "Year of Manufacture .......... %02x (%d)\n", edid->year, edid->year + 1990 );
	vmm_printf( VMM_DEBUG, "Version Number ............... %02x\n", edid->version );
	vmm_printf( VMM_DEBUG, "Revision Number .............. %02x\n", edid->revision );
	if ( edid->video_input_def & 0x80 ) {
		vmm_printf( VMM_DEBUG, "Video Input Definition ....... %02x (Digital(%s)%s)\n",
					edid->video_input_def,
					video_iface[edid->video_input_def & 0x0F],
					video_color[(edid->video_input_def >> 4) & 0x07] );
	}
	else {
		vmm_printf( VMM_DEBUG, "Video Input Definition ....... %02x (Analog(%s)%s%s%s%s%s)\n",
					edid->video_input_def,
					video_level[(edid->video_input_def >> 5) & 0x03],
					(edid->video_input_def & 0x10)? ", blank-to-blank setup":"",
					(edid->video_input_def & 0x08)? ", separate syncs":"",
					(edid->video_input_def & 0x04)? ", composite sync":"",
					(edid->video_input_def & 0x02)? ", sync on green":"",
					(edid->video_input_def & 0x01)? ", serration vsync":"" );
	}
	vmm_printf( VMM_DEBUG, "Max. Horizontal Image Size ... %02x (%d cm)\n",
				edid->max_horz_img_size, edid->max_horz_img_size );
	vmm_printf( VMM_DEBUG, "Max. Vertical Image Size ..... %02x (%d cm)\n",
				edid->max_vert_img_size, edid->max_vert_img_size );
	vmm_printf( VMM_DEBUG, "Display Gamma ................ %02x (%d.%d)\n",
				edid->gamma, edid->gamma / 100 + 1, edid->gamma % 100 );
	if ( edid->video_input_def & 0x80 ) {
		vmm_printf( VMM_DEBUG, "Feature Support .............. %02x (%s%s%s%s%s%s%s)\n",
					edid->features, display_color_fmt[(edid->features >> 3) & 0x03],
					(edid->features & 0x80)? ", standby":"",
					(edid->features & 0x40)? ", suspend":"",
					(edid->features & 0x20)? ", active-off":"",
					(edid->features & 0x04)? ", sRGB":"",
					(edid->features & 0x02)? ", preferred timing":"",
					(edid->features & 0x01)? ", default GTF supported":"" );
	}
	else {
		vmm_printf( VMM_DEBUG, "Feature Support .............. %02x (%s%s%s%s%s%s%s)\n",
					edid->features, display_color_type[(edid->features >> 3) & 0x03],
					(edid->features & 0x80)? ", standby":"",
					(edid->features & 0x40)? ", suspend":"",
					(edid->features & 0x20)? ", active-off":"",
					(edid->features & 0x04)? ", sRGB":"",
					(edid->features & 0x02)? ", preferred timing":"",
					(edid->features & 0x01)? ", default GTF supported":"" );
	}
	vmm_printf( VMM_DEBUG, "Color Characteristics ........ R:[%03x,%03x], G:[%03x,%03x], B:[%03x,%03x], W:[%03x,%03x]\n",
				(edid->red_x_h   << 2) | edid->red_x_l,   (edid->red_y_h   << 2) | edid->red_y_l,
				(edid->green_x_h << 2) | edid->green_x_l, (edid->green_y_h << 2) | edid->green_y_l,
				(edid->blue_x_h  << 2) | edid->blue_x_l,  (edid->blue_y_h  << 2) | edid->blue_y_l,
				(edid->white_x_h << 2) | edid->white_x_l, (edid->white_y_h << 2) | edid->white_y_l );
	vmm_printf( VMM_DEBUG, "Established Timings .......... %02x,%02x,%02x\n",
				edid->est_timing[0], edid->est_timing[1], edid->est_timing[2] );
	vmm_printf( VMM_DEBUG, "   720 x  400 @ 70Hz ......... %s\n",
				(edid->est_timing[0] & 0x80)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   720 x  400 @ 88Hz ......... %s\n",
				(edid->est_timing[0] & 0x40)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   640 x  480 @ 60Hz ......... %s\n",
				(edid->est_timing[0] & 0x20)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   640 x  480 @ 67Hz ......... %s\n",
				(edid->est_timing[0] & 0x10)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   640 x  480 @ 72Hz ......... %s\n",
				(edid->est_timing[0] & 0x00)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   640 x  480 @ 75Hz ......... %s\n",
				(edid->est_timing[0] & 0x04)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   800 x  600 @ 56Hz ......... %s\n",
				(edid->est_timing[0] & 0x02)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   800 x  600 @ 60Hz ......... %s\n",
				(edid->est_timing[0] & 0x01)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   800 x  600 @ 72Hz ......... %s\n",
				(edid->est_timing[1] & 0x80)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   800 x  600 @ 75Hz ......... %s\n",
				(edid->est_timing[1] & 0x40)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "   832 x  624 @ 75Hz ......... %s\n",
				(edid->est_timing[1] & 0x20)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1024 x  768 @ 87Hz ......... %s\n",
				(edid->est_timing[1] & 0x10)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1024 x  768 @ 60Hz ......... %s\n",
				(edid->est_timing[1] & 0x08)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1024 x  768 @ 70Hz ......... %s\n",
				(edid->est_timing[1] & 0x04)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1024 x  768 @ 75Hz ......... %s\n",
				(edid->est_timing[1] & 0x02)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1280 x 1024 @ 75Hz ......... %s\n",
				(edid->est_timing[1] & 0x01)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "  1152 x  870 @ 75Hz ......... %s\n",
				(edid->est_timing[2] & 0x80)? "supported":"not supported" );
	vmm_printf( VMM_DEBUG, "Standard Timings ............. %04x,%04x,%04x,%04x,%04x,%04x,%04x,%04x\n",
				edid->std_timing[0], edid->std_timing[1], edid->std_timing[2], edid->std_timing[3],
				edid->std_timing[4], edid->std_timing[5], edid->std_timing[6], edid->std_timing[7] );
	vmm_printf( VMM_DEBUG, "Preferred Timing Mode ........ %02x", edid->pref_timing[0] );
	for ( int i=1; i<18; i++ )
		vmm_printf( VMM_DEBUG, ",%02x", edid->pref_timing[i] );
	vmm_printf( VMM_DEBUG, "\n" );
	for ( int n=0; n<3; n++ ) {
		if ( (edid->detail_info[n][0] == 0x00) && (edid->detail_info[n][1] == 0x00) ) {
			vmm_printf( VMM_DEBUG, "Display Descriptor ........... %02x", edid->detail_info[n][0] );
			for ( int i=1; i<18; i++ )
				vmm_printf( VMM_DEBUG, ",%02x", edid->detail_info[n][i] );
			vmm_printf( VMM_DEBUG, "\n" );
		}
		else {
			vmm_printf( VMM_DEBUG, "Detail Timing Descriptor ..... %02x", edid->detail_info[n][0] );
			for ( int i=1; i<18; i++ )
				vmm_printf( VMM_DEBUG, ",%02x", edid->detail_info[n][i] );
			vmm_printf( VMM_DEBUG, "\n" );
		}
	}
	vmm_printf( VMM_DEBUG, "Extension Block Count ........ %02x\n", edid->ext_block_cnt );
	vmm_printf( VMM_DEBUG, "Checksum ..................... %02x\n", edid->checksum );
}
