/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

/**
 * メール本文を生成する機能を提供します。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public interface MailBodyFactory {

	/**
	 * 指定のファイルの内容を本文とするメール本文を作成して返します。
	 * 
	 * @param file
	 *            ファイル
	 * @return メール本文
	 * 
	 * @throws IOException
	 *             対象のファイルオブジェクトがファイルとして読み取りできない場合
	 * @throws NullPointerException
	 *             fileが{@code null}の場合
	 */
	MailBody createBinaryMailBody(File file) throws IOException;

	/**
	 * 指定の入力ストリームの内容を本文とするメール本文を作成して返します。<br>
	 * 引数の入力ストリームはクローズされません。
	 * 
	 * @param in
	 *            入力ストリーム
	 * @param contentType
	 *            コンテントタイプ
	 * @return メール本文
	 * 
	 * @throws IOException
	 *             inの操作によって例外がスローされた場合
	 * @throws NullPointerException
	 *             inが{@code null}の場合
	 */
	MailBody createBinaryMailBody(InputStream in, String contentType)
			throws IOException;

	/**
	 * 指定のリソースを示すURLでメール本文を生成して返します。
	 * 
	 * @param url
	 *            URL
	 * @return メール本文
	 * 
	 * @throws NullPointerException
	 *             urlが{@code null}の場合
	 */
	MailBody createBinaryMailBody(URL url);

	/**
	 * 指定のHTMLを本文とするメール本文を生成して返します。
	 * 
	 * @param html
	 *            HTML
	 * @return メール本文
	 * 
	 * @throws NullPointerException
	 *             htmlが{@code null}の場合
	 */
	MailBody createHTMLMailBody(String html);

	/**
	 * 指定のHTMLを本文とするメール本文を生成して返します。<br>
	 * 文字コードは実装によって正しく適用されるとは限りません。
	 * 
	 * @param html
	 *            HTML
	 * @param charset
	 *            文字コード
	 * @return メール本文
	 * 
	 * @throws NullPointerException
	 *             htmlが{@code null}の場合
	 */
	MailBody createHTMLMailBody(String html, String charset);

	/**
	 * 指定の文字列を本文とするメール本文を生成して返します。
	 * 
	 * @param content
	 *            本文
	 * @return メール本文
	 * 
	 * @throws NullPointerException
	 *             contentが{@code null}の場合
	 */
	MailBody createPlainTextMailBody(String content);
}
