/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.IOException;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.commons.lang.StringUtils;
import com.wankuma.mail.Configuration;
import com.wankuma.mail.Mail;
import com.wankuma.mail.MailAddress;
import com.wankuma.mail.MailException;
import com.wankuma.mail.MailSender;

/**
 * <a href="http://java.sun.com/products/javamail/">Java Mail</a>によるSMTP接続機能を利用した{@link MailSender}の基底クラスです。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public abstract class AbstractJavaMailMailSender extends
		AbstractJavaMailService implements MailSender {

	private Session session;

	private Transport transport;

	/**
	 * 指定の接続プロパティでインスタンスを構築します。
	 * 
	 * @param properties
	 *            プロパティ
	 */
	public AbstractJavaMailMailSender(Properties properties) {
		super(properties);

	}

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailService#connect(com.wankuma.mail.Configuration,
	 *      java.util.Properties)
	 */
	@Override
	protected void connect(Configuration configuration, Properties properties)
			throws MailException {
		String user = configuration.getUser();
		String password = configuration.getPassword();
		if (!StringUtils.isEmpty(user) || !StringUtils.isEmpty(password)) {
			properties.put("mail.smtp.auth", "true");
		}

		properties.put("mail.smtp.connectiontimeout", String
				.valueOf(configuration.getConnectionTimeout()));
		properties.put("mail.smtp.timeout", String.valueOf(configuration
				.getTimeout()));

		if (configuration.isDebug()) {
			properties.put("mail.debug", Boolean.TRUE.toString());
		}

		this.session = Session.getDefaultInstance(properties);

		try {
			this.transport = this.session.getTransport("smtp");
		} catch (NoSuchProviderException e) {
			throw new MailException(e);
		}
		try {
			this.transport.connect(user, password);
		} catch (MessagingException e) {
			throw new MailException(e);
		}
	}

	/**
	 * 指定のメールをJavaMailのメッセージに変換して返します。
	 * 
	 * @param session
	 *            セッション
	 * @param mail
	 *            メール
	 * @return メッセージ
	 * 
	 * @throws MessagingException
	 *             メッセージの変換に失敗した場合
	 * @throws IOException
	 *             メールの入出力処理に失敗した場合
	 */
	protected abstract Message convertMessage(Session session, Mail mail)
			throws MessagingException, IOException;

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailService#disconnect()
	 */
	@Override
	protected void disconnect() throws MailException {
		try {
			this.transport.close();
			this.transport = null;
			this.session = null;
		} catch (MessagingException e) {
			throw new MailException(e);
		}
	}

	/**
	 * @see com.wankuma.mail.MailSender#send(com.wankuma.mail.Mail[])
	 */
	@Override
	public void send(Mail... mails) throws MailException {
		Assertion.notNullValues("mails", mails);
		this.assertConnected();

		if (mails.length == 0) {
			return;
		}

		for (Mail mail : mails) {
			Message message;
			try {
				message = this.convertMessage(this.session, mail);
			} catch (IOException e) {
				throw new MailException(e);
			} catch (MessagingException e) {
				throw new MailException(e);
			}

			String[] envelopeTo = mail.getEnvelopeTo();
			Address[] addresses;
			if (envelopeTo.length == 0) {
				try {
					addresses = message.getAllRecipients();
				} catch (MessagingException e) {
					throw new MailException(e);
				}
			} else {
				addresses = new Address[envelopeTo.length];
				for (int i = 0; i < envelopeTo.length; i++) {
					try {
						addresses[i] = new InternetAddress(envelopeTo[i]);
					} catch (MessagingException e) {
						throw new MailException(e);
					}
				}

			}

			try {
				String returnPath = mail.getReturnPath();
				if (returnPath == null) {
					MailAddress[] from = mail.getFrom();
					if (from != null && from.length > 0) {
						returnPath = from[0].getAddress();
					}
				}
				if (returnPath != null) {
					this.session.getProperties().put("mail.smtp.from",
							returnPath);
				}
				this.transport.sendMessage(message, addresses);
			} catch (MessagingException e) {
				throw new MailException(e);
			}
		}
	}
}
