/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileTypeMap;
import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Message.RecipientType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import com.wankuma.commons.lang.StringUtils;
import com.wankuma.mail.AttachmentFile;
import com.wankuma.mail.Header;
import com.wankuma.mail.Importance;
import com.wankuma.mail.Mail;
import com.wankuma.mail.MailAddress;
import com.wankuma.mail.MailBody;
import com.wankuma.mail.MessageIdGenerator;

/**
 * 日本語処理に対応した{@link com.wankuma.mail.MailSender}の実装です。<br>
 * 内部では<a href="http://java.sun.com/products/javamail/">Java Mail</a>を利用しています。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
class JapaneseMailSender extends AbstractJavaMailMailSender {

	private static class AttachmentFileDataSource implements DataSource {

		private AttachmentFile attachmentFile;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param attachmentFile
		 *            添付ファイル
		 */
		AttachmentFileDataSource(AttachmentFile attachmentFile) {
			this.attachmentFile = attachmentFile;
		}

		/**
		 * @see javax.activation.DataSource#getContentType()
		 */
		@Override
		public String getContentType() {
			return FileTypeMap.getDefaultFileTypeMap().getContentType(
					this.attachmentFile.getFileName());
		}

		/**
		 * @see javax.activation.DataSource#getInputStream()
		 */
		@Override
		public InputStream getInputStream() throws IOException {
			return this.attachmentFile.getInputStream();
		}

		/**
		 * @see javax.activation.DataSource#getName()
		 */
		@Override
		public String getName() {
			return null;
		}

		/**
		 * @see javax.activation.DataSource#getOutputStream()
		 */
		@Override
		public OutputStream getOutputStream() throws IOException {
			return null;
		}
	}

	private static class MailBodyDataSource implements DataSource {

		private MailBody mailBody;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param mailBody
		 *            メール本文
		 */
		MailBodyDataSource(MailBody mailBody) {
			this.mailBody = mailBody;
		}

		/**
		 * @see javax.activation.DataSource#getContentType()
		 */
		@Override
		public String getContentType() {
			return this.mailBody.getContentType();
		}

		/**
		 * @see javax.activation.DataSource#getInputStream()
		 */
		@Override
		public InputStream getInputStream() throws IOException {
			return this.mailBody.getInputStream();
		}

		/**
		 * @see javax.activation.DataSource#getName()
		 */
		@Override
		public String getName() {
			return null;
		}

		/**
		 * @see javax.activation.DataSource#getOutputStream()
		 */
		@Override
		public OutputStream getOutputStream() throws IOException {
			return null;
		}
	}

	private static final class MimeMessageImpl extends MimeMessage {

		private final MessageIdGenerator id;

		/**
		 * インスタンスを構築します。
		 * 
		 * @param session
		 *            セッション
		 * @param id
		 *            メッセージIDジェネレータ
		 */
		MimeMessageImpl(Session session, MessageIdGenerator id) {
			super(session);

			this.id = id;
		}

		/**
		 * @see javax.mail.internet.MimeMessage#getMessageID()
		 */
		@Override
		public String getMessageID() throws MessagingException {
			return this.id.getMessageId();
		}

		/**
		 * @see javax.mail.internet.MimeMessage#updateMessageID()
		 */
		@Override
		protected void updateMessageID() throws MessagingException {
			this.setHeader("Message-ID", this.id.getMessageId());
		}
	}

	/**
	 * 指定の接続プロパティでインスタンスを構築します。
	 * 
	 * @param properties
	 *            プロパティ
	 */
	JapaneseMailSender(Properties properties) {
		super(properties);
	}

	private Address convert(MailAddress address) throws MessagingException,
			IOException {
		if (address == null) {
			return null;
		}
		String addr = address.getAddress();
		String personal = address.getPersonal();

		if (StringUtils.isEmpty(personal)) {
			return new InternetAddress(addr);
		}

		personal = EncodeUtils.encode(personal);
		return new InternetAddress(addr, personal);
	}

	private Address[] convert(MailAddress[] addresses)
			throws MessagingException, IOException {
		Address[] results = new Address[addresses.length];
		for (int i = 0; i < results.length; i++) {
			results[i] = this.convert(addresses[i]);
		}

		return results;
	}

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailMailSender#convertMessage(javax.mail.Session,
	 *      com.wankuma.mail.Mail)
	 */
	@Override
	protected Message convertMessage(Session session, Mail mail)
			throws MessagingException, IOException {
		// message id
		final MessageIdGenerator id = mail.getMessageIdGenerator();
		MimeMessage message;
		if (id == null) {
			message = new MimeMessage(session);
		} else {
			message = new MimeMessageImpl(session, id);
		}

		// importance
		Importance importance = mail.getImportance();
		if (importance != null) {
			message.setHeader("Importance", importance.getImportance());
			message.setHeader("X-Priority", String.valueOf(importance
					.getXPriority()));
		}

		// recipients
		for (MailAddress address : mail.getTo()) {
			message.addRecipient(RecipientType.TO, this.convert(address));
		}
		for (MailAddress address : mail.getCc()) {
			message.addRecipient(RecipientType.CC, this.convert(address));
		}
		for (MailAddress address : mail.getBcc()) {
			message.addRecipient(RecipientType.BCC, this.convert(address));
		}

		// from
		MailAddress[] from = mail.getFrom();
		if (from != null) {
			message.addFrom(this.convert(from));
		}

		// reply to
		message.setReplyTo(this.convert(mail.getReplayTo()));

		// subject
		String subject = mail.getSubject();
		if (subject != null) {
			message.setSubject(EncodeUtils.encode(subject));
		}

		// date
		Date sentDate = mail.getSentDate();
		if (sentDate == null) {
			sentDate = new Date();
		}
		message.setSentDate(sentDate);

		// header
		for (Header header : mail.getHeaders()) {
			message.setHeader(header.getName(), header.getValue());
		}

		// body & attachments
		MailBody body = mail.getBody();

		MimeMultipart textMultipart = new MimeMultipart("alternative");
		MimeBodyPart bodyPart = new MimeBodyPart();
		if (body == null) {
			body = new JapaneseMailBodyFactory().createPlainTextMailBody("");
		}
		bodyPart.setDataHandler(new DataHandler(new MailBodyDataSource(body)));
		for (Header header : body.getHeaders()) {
			bodyPart.setHeader(header.getName(), header.getValue());
		}
		textMultipart.addBodyPart(bodyPart);

		MimeMultipart multipart = new MimeMultipart();
		MimeBodyPart temp = new MimeBodyPart();
		temp.setContent(textMultipart);
		multipart.addBodyPart(temp);

		for (AttachmentFile attachmentFile : mail.getAttachmentFiles()) {
			MimeBodyPart attachment = new MimeBodyPart();
			attachment.setFileName(EncodeUtils.encode(attachmentFile
					.getFileName()));
			attachment.setDataHandler(new DataHandler(
					new AttachmentFileDataSource(attachmentFile)));
			multipart.addBodyPart(attachment);
		}

		message.setContent(multipart);

		return message;
	}
}
