/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.IOException;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import javax.mail.FetchProfile;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.Session;
import javax.mail.Store;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.commons.lang.StringUtils;
import com.wankuma.mail.Configuration;
import com.wankuma.mail.MailException;
import com.wankuma.mail.MailReceiveHandler;
import com.wankuma.mail.MailReceiver;
import com.wankuma.mail.MailReceiverProtocol;
import com.wankuma.mail.ReceivedMailInfo;

/**
 * <a href="http://java.sun.com/products/javamail/">Java Mail</a>によるPOP3及びIMAP接続機能を利用した{@link MailReceiver}の基底クラスです。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public abstract class AbstractJavaMailMailReceiver extends
		AbstractJavaMailService implements MailReceiver {

	private MailReceiverProtocol protocol;

	private Session session;

	private Store store;

	/**
	 * 指定の接続プロパティでインスタンスを構築します。
	 * 
	 * @param protocol
	 *            プロトコル
	 * @param properties
	 *            プロパティ
	 */
	public AbstractJavaMailMailReceiver(MailReceiverProtocol protocol,
			Properties properties) {
		super(properties);

		Assertion.notNull("protocol", protocol);
		this.protocol = protocol;
	}

	/**
	 * @see com.wankuma.mail.MailReceiver#authOnly(com.wankuma.mail.Configuration)
	 */
	@Override
	public void authOnly(Configuration configuration) throws MailException {
		this.open(configuration);
		this.close();
	}

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailService#connect(com.wankuma.mail.Configuration,
	 *      java.util.Properties)
	 */
	@Override
	protected void connect(Configuration configuration, Properties properties)
			throws MailException {
		String user = configuration.getUser();
		String password = configuration.getPassword();
		String p = this.protocol == MailReceiverProtocol.IMAP ? "imap" : "pop3";

		if (!StringUtils.isEmpty(user) || !StringUtils.isEmpty(password)) {
			properties.put("mail." + p + ".auth", "true");
		}

		properties.put("mail." + p + ".connectiontimeout", String
				.valueOf(configuration.getConnectionTimeout()));
		properties.put("mail." + p + ".timeout", String.valueOf(configuration
				.getTimeout()));

		if (configuration.isDebug()) {
			properties.put("mail.debug", Boolean.TRUE.toString());
		}

		this.session = Session.getDefaultInstance(properties);

		try {
			this.store = this.session.getStore(p);
		} catch (NoSuchProviderException e) {
			throw new MailException(e);
		}
		try {
			this.store.connect(user, password);
		} catch (MessagingException e) {
			throw new MailException(e);
		}
	}

	/**
	 * 指定のメッセージを受信メールへ変換します。
	 * 
	 * @param message
	 *            メッセージ
	 * @return 受信メール
	 * 
	 * @throws IOException
	 *             受信ファイルの入出力処理に失敗した場合
	 * @throws MessagingException
	 *             メッセージの処理に失敗した場合
	 */
	protected abstract ReceivedMailInfo convertMessage(Message message)
			throws IOException, MessagingException;

	/**
	 * @see com.wankuma.mail.javamail.AbstractJavaMailService#disconnect()
	 */
	@Override
	protected void disconnect() throws MailException {
		this.session = null;

		try {
			this.store.close();
		} catch (MessagingException e) {
			throw new MailException(e);
		}
		this.store = null;
	}

	/**
	 * @see com.wankuma.mail.MailReceiver#getFolderNames()
	 */
	@Override
	public String[] getFolderNames() throws MailException {
		this.assertConnected();

		Set<String> names = new HashSet<String>();
		try {
			this.getFolderNames0(names, this.store.getDefaultFolder());
		} catch (MessagingException e) {
			throw new MailException(e);
		}

		return names.toArray(new String[names.size()]);
	}

	private void getFolderNames0(Set<String> names, Folder folder)
			throws MessagingException {
		int type = folder.getType();
		if ((type ^ Folder.HOLDS_FOLDERS) != 0) {
			return;
		}
		for (Folder child : folder.list()) {
			names.add(child.getFullName());
			this.getFolderNames0(names, child);
		}
	}

	/**
	 * プロトコルを返します。
	 * 
	 * @return プロトコル
	 */
	protected MailReceiverProtocol getProtocol() {
		return this.protocol;
	}

	/**
	 * @see com.wankuma.mail.MailReceiver#receive(com.wankuma.mail.MailReceiveHandler)
	 */
	@Override
	public void receive(MailReceiveHandler handler) throws MailException {
		this.receive(handler, null);
	}

	/**
	 * @see com.wankuma.mail.MailReceiver#receive(com.wankuma.mail.MailReceiveHandler,
	 *      java.lang.String)
	 */
	@Override
	public void receive(MailReceiveHandler handler, String folderName)
			throws MailException {
		Assertion.notNull("handler", handler);
		this.assertConnected();

		String folderName0 = folderName == null ? MailReceiver.DEFAULT_FOLDER_NAME
				: folderName;

		Folder folder;
		try {
			folder = this.store.getFolder(folderName0);
		} catch (MessagingException e) {
			throw new MailException(e);
		}
		try {
			if (!folder.exists()) {
				return;
			}
		} catch (MessagingException e) {
			throw new MailException(e);
		}
		try {
			folder.open(Folder.READ_WRITE);
		} catch (MessagingException e) {
			throw new MailException(e);
		}

		try {
			FetchProfile profile = new FetchProfile();
			profile.add("Subject");
			profile.add("Date");
			profile.add("From");

			Message[] messages = folder.getMessages();
			folder.fetch(messages, profile);

			for (Message message : messages) {
				ReceivedMailInfo mail = this.convertMessage(message);
				handler.doReceive(mail);
			}
		} catch (IOException e) {
			throw new MailException(e);
		} catch (MessagingException e) {
			throw new MailException(e);
		} finally {
			try {
				folder.close(true);
			} catch (MessagingException e) {
				throw new MailException(e);
			}
		}
	}
}
