/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.util.Properties;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.mail.MailReceiver;
import com.wankuma.mail.MailReceiverFactory;
import com.wankuma.mail.MailReceiverProtocol;

/**
 * 日本語処理に対応したメール送信が可能な{@link MailReceiverFactory}の実装です。<br>
 * このファクトリによって生成されるインスタンスは、<a
 * href="http://java.sun.com/products/javamail/">Java Mail</a>を利用しています。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public class JapaneseMailReceiverFactory implements MailReceiverFactory {

	/**
	 * @see com.wankuma.mail.MailReceiverFactory#getMailReceiver(com.wankuma.mail.MailReceiverProtocol,
	 *      java.util.Properties)
	 */
	@Override
	public MailReceiver getMailReceiver(MailReceiverProtocol protocol,
			Properties properties) {
		Assertion.notNull("properties", properties);

		return new JapaneseMailReceiver(protocol, properties);
	}

	/**
	 * @see com.wankuma.mail.MailReceiverFactory#getMailReceiver(com.wankuma.mail.MailReceiverProtocol,
	 *      java.lang.String)
	 */
	@Override
	public MailReceiver getMailReceiver(MailReceiverProtocol protocol,
			String host) {
		return this.getMailReceiver(protocol, host, -1, false);
	}

	/**
	 * @see com.wankuma.mail.MailReceiverFactory#getMailReceiver(com.wankuma.mail.MailReceiverProtocol,
	 *      java.lang.String, boolean)
	 */
	@Override
	public MailReceiver getMailReceiver(MailReceiverProtocol protocol,
			String host, boolean ssl) {
		return this.getMailReceiver(protocol, host, -1, ssl);
	}

	/**
	 * @see com.wankuma.mail.MailReceiverFactory#getMailReceiver(com.wankuma.mail.MailReceiverProtocol,
	 *      java.lang.String, int)
	 */
	@Override
	public MailReceiver getMailReceiver(MailReceiverProtocol protocol,
			String host, int port) {
		return this.getMailReceiver(protocol, host, port, false);
	}

	/**
	 * @see com.wankuma.mail.MailReceiverFactory#getMailReceiver(com.wankuma.mail.MailReceiverProtocol,
	 *      java.lang.String, int, boolean)
	 */
	@Override
	public MailReceiver getMailReceiver(MailReceiverProtocol protocol,
			String host, int port, boolean ssl) {
		Assertion.notNull("protocol", protocol);
		Assertion.notNull("host", host);

		int port0 = port;
		if (port0 < 0 || port0 > 65535) {
			if (protocol == MailReceiverProtocol.IMAP) {
				port0 = ssl ? MailReceiverFactory.DEFAULT_IMAP_OVER_SSL_PORT
						: MailReceiverFactory.DEFAULT_IMAP_PORT;
			} else {
				port0 = ssl ? MailReceiverFactory.DEFAULT_POP3_OVER_SSL_PORT
						: MailReceiverFactory.DEFAULT_POP3_PORT;
			}
		}

		Properties properties = new Properties();
		String p = protocol == MailReceiverProtocol.IMAP ? "imap" : "pop3";
		properties.setProperty("mail." + p + ".host", host);
		properties.setProperty("mail." + p + ".port", Integer.toString(port0));

		if (ssl) {
			properties.setProperty("mail." + p + ".socketFactory.class",
					javax.net.ssl.SSLSocketFactory.class.getName());
			properties.setProperty("mail." + p + ".socketFactory.fallback",
					Boolean.FALSE.toString());
			properties.setProperty("mail." + p + ".socketFactory.port", Integer
					.toString(port0));
		}

		if (protocol == MailReceiverProtocol.APOP) {
			properties.setProperty("mail.pop3.apop.enable", Boolean.TRUE
					.toString());
		}

		return this.getMailReceiver(protocol, properties);
	}
}
