/*
 * Copyright (C) 2006-2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.Iterator;
import java.util.Stack;
import javax.inject.Inject;
import org.infodb.wax.core.BraceArray.BraceItem;
import org.infodb.wax.core.html.A;
import org.infodb.wax.core.html.AbstractBody;
import org.infodb.wax.core.html.Attr;
import org.infodb.wax.core.html.H;
import org.infodb.wax.core.html.LI;
import org.infodb.wax.core.html.OL;
import org.infodb.wax.core.html.UL;
import org.xml.sax.SAXException;

public class WikiProcessor {
    private Stack<AbstractBody> stack;
    
    private WikiBridge bridge;
    private PluginManager pluginManager;
    private AbstractBody body;

    @Inject
    public WikiProcessor(WikiBridge bridge, PluginManager pluginManager) {
        this.bridge = bridge;
        this.pluginManager = pluginManager;
        stack = new Stack<>();
    }
    /**
     * Wikiで表現された文字列を解析します。
     * 文字列は改行コード等含む全データを引数に渡す必要があります。
     */
    public void parse(String src, AbstractBody body) throws IOException, SAXException {
        parse(new BufferedReader(new StringReader(src)), body);
    }
    public void parse(BufferedReader reader, AbstractBody body) throws IOException, SAXException {
        this.body = body;
        String line = reader.readLine();
        while (line != null) {
            if (checkIndividualMarks(line) == false) {
                checkBlockMarks(line);
            }
            line = reader.readLine();
        }
        resetStack();
    }
    private void resetStack() throws IOException {
        while (stack.empty() == false) {
            stack.pop().end();
        }
    }
    private static interface Listener {
        AbstractBody call() throws IOException;
        Class getType();
    }
    private void judgeIndentLevel(int level, Listener listener) throws IOException {
        while(stack.size() != level) {
            if(stack.size() < level) {
                stack.push(listener.call());
            } else if(stack.size() > level) {
                stack.pop().end();
            }
        }
        if(!stack.peek().getClass().equals(listener.getType())) {
            stack.pop().end();
            stack.push(listener.call());
        }
    }
    private boolean checkIndividualMarks(String line) throws IOException {
        if (line.equals("")) {
            resetStack();
            body.br();
            return true;
        }
        if (line.startsWith("----")) {
            resetStack();
            body.hr();
            return true;
        }
        if (line.startsWith("#")) {
            resetStack();
            String name = line.substring(1).split(" ")[0];
            try {
                pluginManager.view(name, bridge, body);
            }
            catch(PluginException e) {
                body.text(e.getMessage());
            }
            return true;
        }
        return false;
    }

    private void checkBlockMarks(String line) throws IOException {
        int count;
        count = parseBlockMarksCount(line, '*');
        if (count > 0) {
            resetStack();
            H h = body.h(count);
            checkBraces(line.substring(count));
            h.end();
            return;
        }
        count = parseBlockMarksCount(line, '+');
        if (count > 0) {
            judgeIndentLevel(count, new Listener() {
                @Override
                public AbstractBody call() throws IOException {
                    return body.ol();
                }
                @Override
                public Class getType() {
                    return OL.class;
                }
            });
            LI li = ((OL)stack.peek()).li();
            checkBraces(line.substring(count));
            li.end();
            return;
        }
        count = parseBlockMarksCount(line, '-');
        if (count > 0) {
            judgeIndentLevel(count, new Listener() {
                @Override
                public AbstractBody call() throws IOException {
                    return body.ul();
                }
                @Override
                public Class getType() {
                    return UL.class;
                }
            });
            LI li = ((UL)stack.peek()).li();
            checkBraces(line.substring(count));
            li.end();
            return;
        }
        checkBraces(line);
        if (line.endsWith("~")) {
            body.br();
        }
    }

    private int parseBlockMarksCount(String line, char ch) throws IOException {
        int count = 0;
        for (int i = 0; i < line.length(); i++) {
            if (line.charAt(i) == ch) {
                count++;
                if (count > 3) {
                    body.text("###Error");
                    count = 0;
                    break;
                }
            } else {
                break;
            }
        }
        return count;
    }
    
    private void checkBraces(String line) throws IOException {
        if (line == null) {
            return;
        }
        if (line.equals("")) {
            return;
        }
        line = line.replaceAll("'''(.+?)'''", "<i>$1</i>");
        line = line.replaceAll("''(.+?)''", "<strong>$1</strong>");
        line = line.replaceAll("%%(.+?)%%", "<s>$1</s>");
        
        BraceArray list = new BraceArray();
        findBrace(list, BraceType.ANCHOR, line);

        int from = 0;
        int to = line.length();
        Iterator<BraceItem> iter = list.iterator();
        if (iter.hasNext() == false) { // 特にブレース等無かった場合
            body.text(line);
            return;
        }
        while (iter.hasNext()) {
            BraceItem item = iter.next();
            body.text(line, from, item.getFrom()); // ブレースまでの文字を出力

            anchor(line.substring(item.getFrom() + item.getBraceType().getFromLength(), item.getTo()));
            from = item.getTo() + item.getBraceType().getToLength();
        }
        if (from < to) {
            body.text(line, from, to); // 残りを出力する。
        }
    }
    private void findBrace(BraceArray list, BraceType type, String line) {
        int i = 0;
        int j = 0;
        while (true) {
            i = line.indexOf(type.getFrom(), i);
            if (i != -1) {
                if (j < i) {
                    j = i;
                }
                int k = line.indexOf(type.getTo(), j);
                if (k != -1) {
                    list.add(type, i, k);
                    j = k + type.getTo().length();
                } else {
                    break;
                }
            } else {
                break;
            }
        }
    }
    private void anchor(String src) throws IOException {
        int index = src.indexOf(">");
        if (index != -1) {
            A a = body.a(Attr.href(src.substring(index + 1)));
            a.text(src, 0, index + 1);
            a.end();
        } else {
            A a = body.a(Attr.href(src));
            a.text(src);
            a.end();
        }
    }
}
