/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.List;
import java.util.UUID;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.io.Buffer;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.infodb.wax.core.ActionParams;
import org.infodb.wax.core.HeaderMenu;
import org.infodb.wax.core.PluginException;
import org.infodb.wax.core.PluginManager;
import org.infodb.wax.core.WikiBridge;
import org.infodb.wax.core.WikiProcessor;
import org.infodb.wax.core.db.ColumnPair;
import org.infodb.wax.core.db.ContentResolver;
import org.infodb.wax.core.db.SQLWhere;
import org.infodb.wax.core.db.TableDef;
import org.infodb.wax.core.db.XmlQuery;
import org.infodb.wax.core.db.XmlStorage;
import org.infodb.wax.core.utils.StreamHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class Handler extends AbstractHandler {
    private static final Logger log = LoggerFactory.getLogger(Handler.class);
    
    public static class PageTableDef implements TableDef {
        @Override
        public String getTableName() {
            return "pages";
        }
        @Override
        public ColumnPair[] getIndexColumn() {
            return null;
        }
    }
    
    private File pagesFile;
    private PluginManager pluginManager;
    private XmlStorage storage;
    
    private MimeTypes mimeTypes;
    
    private List<HeaderMenu> menuList;

    public Handler() {
    }
    private String getMimeType(String filename) {
        Buffer buff = mimeTypes.getMimeByExtension(filename);
        if(buff != null) {
            return buff.toString("UTF-8");
        }
        return null;
    }
    public void init(WaxEnv env) throws IOException, ClassNotFoundException, InstantiationException, IllegalAccessException, SQLException {
        pagesFile = new File(env.getCurrentDirectory(), "/pages");
        mimeTypes = new MimeTypes();

        storage = env.newStorage("default", new PageTableDef());
        if(storage.existTable() == false) {
            storage.createTable();
        }

        pluginManager = new PluginManager();
        pluginManager.initLibrary(env);
        
        menuList = env.getMenuList();
    }
    private String getSession(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        for(Cookie c : cookies) {
            if(c.getName().equals("xsession")) {
                String id = c.getValue();
                return id;
            }
        }
        return UUID.randomUUID().toString();
    }
    @Override
    public void handle(String target, Request rqst, HttpServletRequest req, HttpServletResponse resp) throws IOException, ServletException {
        if(rqst.isHandled()) {
            return;
        }
        rqst.setHandled(true);
        try {
            String session = getSession(req);
            if(target.startsWith("/ext/")) {
                proceedExt(target.substring(5), session, req, resp);
                return;
            }
            proceedWiki(target, session, req, resp);
        } catch (SQLException | SAXException ex) {
            log.error(ex.getMessage(), ex);
        }
    }
    private void proceedExt(String target, String session, HttpServletRequest req, HttpServletResponse resp) throws IOException, SAXException, SQLException {
        int idx = target.indexOf("/");
        if(idx == -1 || target.length() <= idx) {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
            return;
        }
        String name = target.substring(0, idx);
        String arg = target.substring(idx);
        ActionParams ap = new ActionParamsImpl(req, resp, arg);
        try {
            if(pluginManager.action(name, ap) == false) {
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
            }
        } catch (PluginException ex) {
            throw new IOException(ex);
        }
    }
    private void proceedWiki(String target, String session, HttpServletRequest req, final HttpServletResponse resp) throws IOException, SAXException, SQLException {
        if(target.equals("/")) {
            target = "/FrontPage";
        }
        String mime = getMimeType(target);
        if(mime != null) {
            File file = new File(pagesFile, target);
            if(file.exists()) {
                resp.setContentType(mime);
                try (InputStream is = new FileInputStream(file)) {
                    OutputStream os = resp.getOutputStream();
                    StreamHelper.bufferdCopy(is, os);
                }
                return;
            }
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
            return;
        }
        WikiBridge bridge = new WikiBridge();
        WikiProcessor processor = new WikiProcessor(bridge, pluginManager);
        WikiResolver resolver = new WikiResolver(resp, target, target, processor, menuList);
        if(query(resolver, session, target) == false) {
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
            return;
        }
        if(bridge.isOutputMenu()) {
            query(resolver, session, "/Menu");
        }
        resolver.end();
    }
    public boolean query(ContentResolver resolver, String sessionId, String target) throws IOException {
        try {
            XmlQuery xq = new XmlQuery(resolver);
            xq.addWhere(new SQLWhere(target));
            if(storage.query(sessionId, xq) == false) {
                File file = new File(pagesFile, target + ".wiki"); // 日本語の対応
                if(file.exists() == false) {
                    return false;
                }
                xq.setContentType("text/xml");
                xq.setInputSouce(new InputSource(new FileInputStream(file)));
                xq.begin();
                xq.execute();
                xq.end();
            }
            return true;
        }
        catch(SQLException | SAXException e) {
            throw new IOException(e);
        }
    }
    @Override
    protected void doStop() throws Exception {
        pluginManager.exit();
        super.doStop();
    }
}
