/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-9 by Raw Material Software Ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the GNU General
   Public License (Version 2), as published by the Free Software Foundation.
   A copy of the license is included in the JUCE distribution, or can be found
   online at www.gnu.org/licenses.

   JUCE is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  ------------------------------------------------------------------------------

   To release a closed-source product which uses JUCE, commercial licenses are
   available: visit www.rawmaterialsoftware.com/juce for more information.

  ==============================================================================
*/

#ifndef __JUCE_FILTERGRAPH_JUCEHEADER__
#define __JUCE_FILTERGRAPH_JUCEHEADER__

namespace juce {

class FilterInGraph;
class FilterGraph;

const char* const filenameSuffix = ".filtergraph";
const char* const filenameWildcard = "*.filtergraph";

//==============================================================================
/**
    Represents a connection between two pins in a FilterGraph.
*/
class FilterConnection
{
public:
    //==============================================================================
    FilterConnection (FilterGraph& owner);
    FilterConnection (const FilterConnection& other);
    ~FilterConnection();

    //==============================================================================
	juce::uint32 sourceFilterID;
    int sourceChannel;
	juce::uint32 destFilterID;
    int destChannel;

    //==============================================================================
    juce_UseDebuggingNewOperator

private:
    FilterGraph& owner;

    FilterConnection& operator= (const FilterConnection&);
};


//==============================================================================
/**
    Represents one of the filters in a FilterGraph.
*/
/*class FilterInGraph   : public ReferenceCountedObject
{
public:
    //==============================================================================
    FilterInGraph (FilterGraph& owner, AudioPluginInstance* const plugin);
    ~FilterInGraph();

    //==============================================================================
    AudioPluginInstance* const filter;
    uint32 uid;

    //==============================================================================
    void showUI (bool useGenericUI);

    double getX() const throw()                     { return x; }
    double getY() const throw()                     { return y; }
    void setPosition (double x, double y) throw();

    XmlElement* createXml() const;

    static FilterInGraph* createForDescription (FilterGraph& owner,
                                                const PluginDescription& desc,
                                                String& errorMessage);

    static FilterInGraph* createFromXml (FilterGraph& owner, const XmlElement& xml);

    //==============================================================================
    typedef ReferenceCountedObjectPtr <FilterInGraph> Ptr;

    //==============================================================================
    juce_UseDebuggingNewOperator

private:
    friend class FilterGraphPlayer;
    FilterGraph& owner;
    double x, y;

    friend class PluginWindow;
    Component* activeUI;
    Component* activeGenericUI;
    int lastX, lastY;

    MidiBuffer outputMidi;
    AudioSampleBuffer processedAudio;
    MidiBuffer processedMidi;

    void prepareBuffers (int blockSize);
    void renderBlock (int numSamples,
                      const ReferenceCountedArray <FilterInGraph>& filters,
                      const OwnedArray <FilterConnection>& connections);

    FilterInGraph (const FilterInGraph&);
    FilterInGraph& operator= (const FilterInGraph&);
};
*/

class AddFilterCommand;
class RemoveFilterCommand;
class AddFilterConnectionCommand;
class RemoveFilterConnectionCommand;
//==============================================================================
/**
    A collection of filters and some connections between them.
*/
class FilterGraph   : public juce::FileBasedDocument//,public AudioProcessorListener
{
	friend class AddFilterCommand;
	friend class RemoveFilterCommand;
	friend class AddFilterConnectionCommand;
	friend class RemoveFilterConnectionCommand;

public:
    //==============================================================================
    FilterGraph();
    ~FilterGraph();

    //==============================================================================
    sf::AudioProcessorGraph& getGraph() throw()         { return graph; }

    int getNumFilters() const throw();
    const sf::AudioProcessorGraph::Node::Ptr getNode (const int index) const throw();
    const sf::AudioProcessorGraph::Node::Ptr getNodeForId (const uint32 uid) const throw();

	void addFilter (const PluginDescription* desc, double x, double y);

    void removeFilter (const uint32 filterUID);
    void disconnectFilter (const uint32 filterUID);

    void removeIllegalConnections();

    void setNodePosition (const int nodeId, double x, double y);
    void getNodePosition (const int nodeId, double& x, double& y) const;

    //==============================================================================
    int getNumConnections() const throw();
    const sf::AudioProcessorGraph::Connection* getConnection (const int index) const throw();

    const sf::AudioProcessorGraph::Connection* getConnectionBetween (uint32 sourceFilterUID, int sourceFilterChannel,
                                                                 uint32 destFilterUID, int destFilterChannel) const throw();

    bool canConnect (uint32 sourceFilterUID, int sourceFilterChannel,
                     uint32 destFilterUID, int destFilterChannel) const throw();

    bool addConnection (uint32 sourceFilterUID, int sourceFilterChannel,
                        uint32 destFilterUID, int destFilterChannel);

    void removeConnection (const int index);

    void removeConnection (uint32 sourceFilterUID, int sourceFilterChannel,
                           uint32 destFilterUID, int destFilterChannel);

    void clear();

 //  void audioProcessorChanged (AudioProcessor* processor) 
	//{
	//	changed();
	//	//updateComponents();
	//}

 //   virtual void audioProcessorParameterChangeGestureBegin (AudioProcessor* processor,
 //                                                           int parameterIndex)
	//{};

 //   virtual void audioProcessorParameterChangeGestureEnd (AudioProcessor* processor,
 //                                                         int parameterIndex)
	//{};
    //==============================================================================

    XmlElement* createXml() const;
    void restoreFromXml (const XmlElement& xml);

    //==============================================================================
    const String getDocumentTitle();
    const String loadDocument (const File& file);
    const String saveDocument (const File& file);
    const File getLastDocumentOpened();
    void setLastDocumentOpened (const File& file);

	//juce::UndoManager& getUndoManager(){return undoManager;};

	/** The special channel index used to refer to a filter's midi channel.
    */
    static const int midiChannelNumber;

    //==============================================================================
    juce_UseDebuggingNewOperator

private:
	//juce::UndoManager undoManager;

    sf::AudioProcessorGraph graph;
    AudioProcessorPlayer player;

	juce::uint32 lastUID;
	juce::uint32 getNextUID() throw();

    void createNodeFromXml (const XmlElement& xml);

    FilterGraph (const FilterGraph&);
    FilterGraph& operator= (const FilterGraph&);
};


class AddFilterCommand : public juce::UndoableAction
{
public:
	AddFilterCommand(FilterGraph& target,const PluginDescription* desc, double x, double y) 
		: targetFilterGraph(target),targetDescription(desc),xpos(x),ypos(y),addedFilterNode(0) {};

   //==============================================================================
    /** Overridden by a subclass to perform the action.

        This method is called by the UndoManager, and shouldn't be used directly by
        applications.

        Be careful not to make any calls in a perform() method that could call
        recursively back into the UndoManager::perform() method

        @returns    true if the action could be performed.
        @see UndoManager::perform
    */
	virtual bool perform();

    /** Overridden by a subclass to undo the action.

        This method is called by the UndoManager, and shouldn't be used directly by
        applications.

        Be careful not to make any calls in an undo() method that could call
        recursively back into the UndoManager::perform() method

        @returns    true if the action could be undone without any errors.
        @see UndoManager::perform
    */
	virtual bool undo(); 

    //==============================================================================
    /** Returns a value to indicate how much memory this object takes up.

        Because the UndoManager keeps a list of UndoableActions, this is used
        to work out how much space each one will take up, so that the UndoManager
        can work out how many to keep.

        The default value returned here is 10 - units are arbitrary and
        don't have to be accurate.

        @see UndoManager::getNumberOfUnitsTakenUpByStoredCommands,
             UndoManager::setMaxNumberOfStoredUnits
    */
    virtual int getSizeInUnits()    { return (int)(sizeof(AddFilterCommand)); }
private:
	FilterGraph& targetFilterGraph;
	const double xpos,ypos;
	const PluginDescription * targetDescription;
	sf::AudioProcessorGraph::Node* addedFilterNode;
	
};

class RemoveFilterCommand : public juce::UndoableAction
{
public:
	RemoveFilterCommand(FilterGraph& filterGraph,const uint32 id) : targetFilterGraph(filterGraph),targetId(id)
	{
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(RemoveFilterCommand)); }
private:
	FilterGraph& targetFilterGraph;
	const int targetId; 
};


class AddFilterConnectionCommand : public juce::UndoableAction
{
public:
	AddFilterConnectionCommand(FilterGraph& graph,const juce::uint32 sourceNodeId_,
                                         const int sourceChannelIndex_,
                                         const juce::uint32 destNodeId_,
                                         const int destChannelIndex_) 
										 : targetGraph(graph),
											sourceNodeId(sourceNodeId_),sourceChannelIndex(sourceChannelIndex_),
											destNodeId(destNodeId_),destChannelIndex(destChannelIndex_)
	{
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(AddFilterConnectionCommand)); }
private:
	FilterGraph& targetGraph;
	const juce::uint32 sourceNodeId;
    const int sourceChannelIndex;
    const juce::uint32 destNodeId;
    const int destChannelIndex;
};

class RemoveFilterConnectionCommand : public juce::UndoableAction
{
public:
	RemoveFilterConnectionCommand(FilterGraph& graph,const juce::uint32 sourceNodeId_,
                                         const int sourceChannelIndex_,
                                         const juce::uint32 destNodeId_,
                                         const int destChannelIndex_) 
										 : targetGraph(graph),
											sourceNodeId(sourceNodeId_),sourceChannelIndex(sourceChannelIndex_),
											destNodeId(destNodeId_),destChannelIndex(destChannelIndex_)
	{
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(RemoveFilterConnectionCommand)); }
private:
	FilterGraph& targetGraph;
	const juce::uint32 sourceNodeId;
    const int sourceChannelIndex;
    const juce::uint32 destNodeId;
    const int destChannelIndex;
};
}
#endif
