<?php

if (!defined('XOOPS_ROOT_PATH')) exit();

if(class_exists('Xoonips_InstallUtils'))
{
    return;
}

/**
 * Xoonips_InstallUtils
**/
class Xoonips_InstallUtils
{
    /**
     * installSQLAutomatically
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/

    public static function installSQLAutomatically(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $sqlFileInfo =& $module->getInfo('sqlfile');
        if(!isset($sqlFileInfo[XOOPS_DB_TYPE]))
        {
            return true;
        }
        $sqlFile = $sqlFileInfo[XOOPS_DB_TYPE];

        $dirname = $module->getVar('dirname');
        $sqlFilePath = sprintf('%s/%s/%s',XOOPS_MODULE_PATH,$dirname,$sqlFile);
        if(!file_exists($sqlFilePath))
        {
            $sqlFilePath = sprintf(
                '%s/modules/%s/%s',
                XOOPS_TRUST_PATH,
                $module->modinfo['trust_dirname'],
                $sqlFile
            );
        }

        require_once XOOPS_MODULE_PATH . '/legacy/admin/class/Legacy_SQLScanner.class.php';    // TODO will be use other class?
        $scanner = new Legacy_SQLScanner();
        $scanner->setDB_PREFIX(XOOPS_DB_PREFIX);
        $scanner->setDirname($dirname);
        if(!$scanner->loadFile($sqlFilePath))
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_SQL_FILE_NOT_FOUND,
                    $sqlFile
                )
            );
            return false;
        }

        $scanner->parse();
        $root =& XCube_Root::getSingleton();
        $db =& $root->mController->getDB();

        foreach($scanner->getSQL() as $sql)
        {
            if(!$db->query($sql))
            {
                $log->addError($db->error());
                return false;
            }
        }
        $log->addReport(_MI_XOONIPS_INSTALL_MSG_DB_SETUP_FINISHED);
        return true;
    }

    /**
     * DBquery
     *
     * @param   string  $query
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function DBquery(/*** string ***/ $query,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        require_once XOOPS_MODULE_PATH . '/legacy/admin/class/Legacy_SQLScanner.class.php';    // TODO will be use other class?
        $scanner = new Legacy_SQLScanner();
        $scanner->setDB_PREFIX(XOOPS_DB_PREFIX);
        $scanner->setDirname($module->get('dirname'));
        $scanner->setBuffer($query);
        $scanner->parse();
        $sqls = $scanner->getSQL();

        $root =& XCube_Root::getSingleton();

        $successFlag = true;
        foreach($sqls as $sql)
        {
            if($root->mController->mDB->query($sql))
            {
                $log->addReport(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_MSG_SQL_SUCCESS,
                        $sql
                    )
                );
            }
            else
            {
                $log->addReport(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_MSG_SQL_ERROR,
                        $sql
                    )
                );
                $successFlag = false;
            }
        }
        return $successFlag;
    }

    /**
     * replaceDirname
     *
     * @param   string  $from
     * @param   string  $dirname
     * @param   string  $trustDirname
     *
     * @return  {string 'public',string 'trust'}
    **/
    public static function replaceDirname(/*** string ***/ $from,/*** string ***/ $dirname,/*** string ***/ $trustDirname = null)
    {
        return array(
            'public' => str_replace('{dirname}',$dirname,$from),
            'trust' => ($trustDirname != null) ? str_replace('{dirname}',$trustDirname,$from) : null
        );
    }

    /**
     * readTemplateFile
     *
     * @param   string  $dirname
     * @param   string  $trustDirname
     * @param   string  $filename
     * @param   bool  $isBlock
     *
     * @return  string
    **/
    public static function readTemplateFile(/*** string ***/ $dirname,/*** string ***/ $trustDirname,/*** string ***/ $filename,/*** bool ***/ $isBlock = false)
    {
        $filePath = sprintf(
            '%s/%s/templates/%s%s',
            XOOPS_MODULE_PATH,
            $dirname,
            ($isBlock ? 'blocks/' : ''),
            $filename
        );

        if(!file_exists($filePath))
        {
            $filePath = sprintf(
                '%s/modules/%s/templates/%s%s',
                XOOPS_TRUST_PATH,
                $trustDirname,
                ($isBlock ? 'blocks/' : ''),
                $filename
            );
            if(!file_exists($filePath))
            {
                return false;
            }
        }

        if(!($lines = file($filePath)))
        {
            return false;
        }

        $tplData = '';
        foreach($lines as $line)
        {
            $tplData .= str_replace("\n","\r\n",str_replace("\r\n","\n",$line));
        }

        return $tplData;
    }

    /**
     * installAllOfModuleTemplates
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function installAllOfModuleTemplates(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $templates =& $module->getInfo('templates');
        if(is_array($templates) && count($templates) > 0)
        {
            foreach($templates as $template)
            {
                Xoonips_InstallUtils::installModuleTemplate($module,$template,$log);
            }
        }
    }

    /**
     * installModuleTemplate
     *
     * @param   XoopsModule  &$module
     * @param   string[]  $template
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installModuleTemplate(/*** XoopsModule ***/ &$module,/*** string[] ***/ $template,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $dirname = $module->getVar('dirname');
        $trustDirname =& $module->getInfo('trust_dirname');
        $tplHandler =&xoops_gethandler('tplfile');
        $filename   =  Xoonips_InstallUtils::replaceDirname(trim($template['file']),$dirname,$trustDirname);
        $tplData    =  Xoonips_InstallUtils::readTemplateFile($dirname,$trustDirname,$filename['trust']);

        if($tplData == false)
        {
            return false;
        }

        $tplFile =& $tplHandler->create();
        $tplFile->setVar('tpl_refid'       ,$module->getVar('mid'));
        $tplFile->setVar('tpl_lastimported',0);
        $tplFile->setVar('tpl_lastmodified',time());
        $tplFile->setVar('tpl_type'        ,(substr($filename['trust'],-4) == '.css') ? 'css' : 'module');
        $tplFile->setVar('tpl_source'      ,$tplData,true);
        $tplFile->setVar('tpl_module'      ,$module->getVar('dirname'));
        $tplFile->setVar('tpl_tplset'      ,'default');
        $tplFile->setVar('tpl_file'        ,$filename['public'],true);
        $tplFile->setVar('tpl_desc'        ,isset($template['desctiption']) ? $template['description'] : '',true);

        if($tplHandler->insert($tplFile))
        {
            $log->addReport(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_MSG_TPL_INSTALLED,
                    $filename['public']
                )
            );
        }
        else
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_TPL_INSTALLED,
                    $filename['public']
                )
            );
            return false;
        }

        return true;
    }

    /**
     * uninstallAllOfModuleTemplates
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     * @param   bool  $defaultOnly
     *
     * @return  void
    **/
    public static function uninstallAllOfModuleTemplates(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log,/*** bool ***/ $defaultOnly = true)
    {
        $tplHandler   =&xoops_gethandler('tplfile');

        $delTemplates =& $tplHandler->find($defaultOnly ? 'default' : null,'module',$module->get('mid'));

        if(is_array($delTemplates) && count($delTemplates) > 0)
        {
            $xoopsTpl = new XoopsTpl();
            $xoopsTpl->clear_cache(null,'mod_' . $module->get('dirname'));
            foreach($delTemplates as $tpl)
            {
                if(!$tplHandler->delete($tpl))
                {
                    $log->addError(
                        XCube_Utils::formatString(
                            _MI_XOONIPS_INSTALL_ERROR_TPL_UNINSTALLED,
                            $tpl->get('tpl_file')
                        )
                    );
                }
            }
        }

    	$delTemplates =& $tplHandler->find($defaultOnly ? 'default' : null,'css',$module->get('mid'));

        if(is_array($delTemplates) && count($delTemplates) > 0)
        {
            $xoopsTpl = new XoopsTpl();
            $xoopsTpl->clear_cache(null,'css_' . $module->get('dirname'));
            foreach($delTemplates as $tpl)
            {
                if(!$tplHandler->delete($tpl))
                {
                    $log->addError(
                        XCube_Utils::formatString(
                            _MI_XOONIPS_INSTALL_ERROR_TPL_UNINSTALLED,
                            $tpl->get('tpl_file')
                        )
                    );
                }
            }
        }
    }

    public static function installCommonTemplates(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log){
    	$dirname = $module->getVar('dirname');
    	$tplfile_handler =& xoops_gethandler('tplfile');
		$tpl_path = XOOPS_TRUST_PATH . '/libs/xoonips/templates';
		if( $handler = @opendir( $tpl_path . '/' ) ) {
			while( ( $file = readdir( $handler ) ) !== false ) {
				if( substr( $file , 0 , 1 ) == '.' ) continue ;
				$file_path = $tpl_path . '/' . $file ;
				if( is_file( $file_path ) ) {
					$mtime = intval( @filemtime( $file_path ) ) ;
					$tplfile =& $tplfile_handler->create() ;
					$tplfile->setVar( 'tpl_source' , file_get_contents( $file_path ) , true ) ;
					$tplfile->setVar( 'tpl_refid' , $module->getVar('mid') ) ;
					$tplfile->setVar( 'tpl_tplset' , 'default' ) ;
					$tplfile->setVar( 'tpl_file' , $dirname . '_' . $file ) ;
					$tplfile->setVar( 'tpl_desc' , '' , true ) ;
					$tplfile->setVar( 'tpl_module' , $dirname ) ;
					$tplfile->setVar( 'tpl_lastmodified' , $mtime ) ;
					$tplfile->setVar( 'tpl_lastimported' , 0 ) ;
					$tplfile->setVar( 'tpl_type' , 'module' ) ;
					if($tplfile_handler->insert($tplfile))
			        {
			            $log->addReport(
			                XCube_Utils::formatString(
			                    _MI_XOONIPS_INSTALL_MSG_TPL_INSTALLED,
			                    $dirname . '_' . $file
			                )
			            );
			        }
			        else
			        {
			            $log->addError(
			                XCube_Utils::formatString(
			                    _MI_XOONIPS_INSTALL_ERROR_TPL_INSTALLED,
			                    $dirname . '_' . $file
			                )
			            );
			            return false;
			        }
				}
			}
			closedir( $handler ) ;
		}
		return true;
    }
    /**
     * installAllOfBlocks
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installAllOfBlocks(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $blocks =& $module->getInfo('blocks');

		$bean_chk =  Xoonips_BeanFactory::chkUserBean();

        if(is_array($blocks) && count($blocks) > 0)
        {
            foreach($blocks as $idx => $block)
            {
				if (!$bean_chk) {
					if (self::omitBlocks($block)) {
						continue;
					}
				}
            	$block['func_num'] = $idx;
                $newBlock =& Xoonips_InstallUtils::createBlockByInfo($module,$block);
                Xoonips_InstallUtils::installBlock($module,$newBlock,$block,$log);
            }
        }
        return true;
    }

    /**
     * &createBlockByInfo
     *
     * @param   XoopsModule  &$module
     * @param   string[]  $block
     *
     * @return  XoopsBlock
    **/
    public static function &createBlockByInfo(/*** XoopsModule ***/ &$module,/*** string[] ***/ $block)
    {
    	$edit_func = isset($block['edit_func']) ? $block['edit_func'] : null;
        $visible = isset($block['visible']) ?
            $block['visible'] :
            (isset($block['visible_any']) ? $block['visible_any'] : 0);
        $filename = isset($block['template']) ?
            Xoonips_InstallUtils::replaceDirname($block['template'],$module->get('dirname')) :
            null;

        $blockHandler =&xoops_gethandler('block');
        $blockObj =& $blockHandler->create();

        $blockObj->set('mid',$module->getVar('mid'));
        $blockObj->set('options',isset($block['options']) ? $block['options'] : null);
        $blockObj->set('name',$block['name']);
        $blockObj->set('title',$block['name']);
        $blockObj->set('block_type','M');
        $blockObj->set('c_type','1');
        $blockObj->set('isactive',1);
        $blockObj->set('dirname',$module->getVar('dirname'));
        $blockObj->set('func_file',$block['file']);
    	//
		// IMPORTANT CONVENTION
		//
		$show_func = "";
		if (isset($block['class'])) {
			$show_func = "cl::" . $block['class'];
		}
		else {
			$show_func = $block['show_func'];
		}
		$blockObj->set('show_func', $show_func);
		$blockObj->set('edit_func', $edit_func);
        $blockObj->set('template',$filename['public']);
        $blockObj->set('last_modified',time());
        $blockObj->set('visible',$visible);
        $blockObj->set('func_num',intval($block['func_num']));
        return $blockObj;
    }

    /**
     * installBlock
     *
     * @param   XoopsModule  &$module
     * @param   XoopsBlock  &$blockObj
     * @param   string[]  &$block
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installBlock(/*** XoopsModule ***/ &$module,/*** XoopsBlock ***/ &$blockObj,/*** string[] ***/ &$block,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $isNew = $blockObj->isNew();
        $blockHandler =&xoops_gethandler('block');
        $autoLink = isset($block['show_all_module']) ? $block['show_all_module'] : false;

        if(!$blockHandler->insert($blockObj,$autoLink))
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_BLOCK_INSTALLED,
                    $blockObj->getVar('name')
                )
            );
            return false;
        }

        $log->addReport(
            XCube_Utils::formatString(
                _MI_XOONIPS_INSTALL_MSG_BLOCK_INSTALLED,
                $blockObj->getVar('name')
            )
        );

        Xoonips_InstallUtils::installBlockTemplate($blockObj,$module,$log);

        if(!$isNew)
        {
            return true;
        }

        if($autoLink)
        {
            $sql = sprintf(
                'insert into `%s` set (`block_id`,`module_id`) values (%d,0);',
                $blockHandler->db->prefix('block_module_link'),
                $blockObj->getVar('bid')
            );
            if(!$blockHandler->db->query($sql))
            {
                $log->addWarning(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_ERROR_BLOCK_COULD_NOT_LINK,
                        $blockObj->getVar('name')
                    )
                );
            }
        }

        $gpermHandler =&xoops_gethandler('groupperm');
        $perm =& $gpermHandler->create();
        $perm->setVar('gperm_itemid',$blockObj->getVar('bid'));
        $perm->setVar('gperm_name','block_read');
        $perm->setVar('gperm_modid',1);
        if(isset($block['visible_any']) && $block['visible_any'])
        {
            $memberHandler =&xoops_gethandler('member');
            $groups =& $memberHandler->getGroups();
            foreach($groups as $group)
            {
                $perm->setVar('gperm_groupid',$group->getVar('groupid'));
                $perm->setNew();
                if(!$gpermHandler->insert($perm))
                {
                    $log->addWarning(
                        XCube_Utils::formatString(
                            _MI_XOONIPS_INSTALL_ERROR_PERM_COULD_NOT_SET,
                            $blockObj->getVar('name')
                        )
                    );
                }
            }
        }
        else
        {
            $root =& XCube_Root::getSingleton();
            $groups = $root->mContext->mXoopsUser->getGroups();
            foreach($groups as $group)
            {
                $perm->setVar('gperm_groupid',$group);
                $perm->setNew();
                if(!$gpermHandler->insert($perm))
                {
                    $log->addWarning(
                        XCube_Utils::formatString(
                            _MI_XOONIPS_INSTALL_ERROR_BLOCK_PERM_SET,
                            $blockObj->getVar('name')
                        )
                    );
                }
            }
        }

        return true;
    }

    /**
     * installBlockTemplate
     *
     * @param   XoopsBlock  &$block
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installBlockTemplate(/*** XoopsBlock ***/ &$block,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        if($block->get('template') == null)
        {
            return true;
        }

        $info =& $module->getInfo('blocks');
        $filename = Xoonips_InstallUtils::replaceDirname(
            $info[$block->get('func_num')]['template'],
            $module->get('dirname'),
            $module->getInfo('trust_dirname')
        );
        $tplHandler =&xoops_gethandler('tplfile');

        $cri = new CriteriaCompo();
        $cri->add(new Criteria('tpl_type','block'));
        $cri->add(new Criteria('tpl_tplset','default'));
        $cri->add(new Criteria('tpl_module',$module->get('dirname')));
        $cri->add(new Criteria('tpl_file',$filename['public']));

        $tpls =& $tplHandler->getObjects($cri);

        if(count($tpls) > 0)
        {
            $tplFile =& $tpls[0];
        }
        else
        {
            $tplFile =& $tplHandler->create();
            $tplFile->set('tpl_refid',$block->get('bid'));
            $tplFile->set('tpl_tplset','default');
            $tplFile->set('tpl_file',$filename['public']);
            $tplFile->set('tpl_module',$module->get('dirname'));
            $tplFile->set('tpl_type','block');
            $tplFile->set('tpl_lastimported',0);
        }

        $tplSource = Xoonips_InstallUtils::readTemplateFile(
            $module->get('dirname'),
            $module->getInfo('trust_dirname'),
            $filename['trust'],
            true
        );

        $tplFile->set('tpl_source',$tplSource);
        $tplFile->set('tpl_lastmodified',time());
        if($tplHandler->insert($tplFile))
        {
            $log->addReport(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_MSG_BLOCK_TPL_INSTALLED,
                    $filename['public']
                )
            );
            return true;
        }

        $log->addError(
            XCube_Utils::formatString(
                _MI_XOONIPS_INSTALL_ERROR_BLOCK_TPL_INSTALLED,
                $filename['public']
            )
        );
        return false;
    }

    /**
     * uninstallAllOfBlocks
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function uninstallAllOfBlocks(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $successFlag = true;

        $blockHandler =&xoops_gethandler('block');
        $gpermHandler =&xoops_gethandler('groupperm');
        $cri = new Criteria('mid',$module->get('mid'));
        $blocks =& $blockHandler->getObjectsDirectly($cri);

        foreach($blocks as $block)
        {
            if($blockHandler->delete($block))
            {
                $log->addReport(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_MSG_BLOCK_UNINSTALLED,
                        $block->get('name')
                    )
                );
            }
            else
            {
                $log->addWarning(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_ERROR_BLOCK_UNINSTALLED,
                        $block->get('name')
                    )
                );
                $successFlag = false;
            }

            $cri = new CriteriaCompo();
            $cri->add(new Criteria('gperm_name','block_read'));
            $cri->add(new Criteria('gperm_itemid',$block->get('bid')));
            $cri->add(new Criteria('gperm_modid',1));
            if(!$gpermHandler->deleteAll($cri))
            {
                $log->addWarning(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_ERROR_BLOCK_PERM_DELETE,
                        $block->get('name')
                    )
                );
                $successFlag = false;
            }
        }

        return $successFlag;
    }

    /**
     * smartUpdateAllOfBlocks
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function smartUpdateAllOfBlocks(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $dirname = $module->get('dirname');

        $fileReader = new Legacy_ModinfoX2FileReader($dirname);
        $dbReader = new Legacy_ModinfoX2DBReader($dirname);

        $blocks =& $dbReader->loadBlockInformations();
        $blocks->update($fileReader->loadBlockInformations());

        foreach($blocks->mBlocks as $block)
        {
            switch($block->mStatus)
            {
                case LEGACY_INSTALLINFO_STATUS_LOADED:
                    Xoonips_InstallUtils::updateBlockTemplateByInfo($block,$module,$log);
                    break;
                case LEGACY_INSTALLINFO_STATUS_UPDATED:
                    Xoonips_InstallUtils::updateBlockByInfo($block,$module,$log);
                    break;
                case LEGACY_INSTALLINFO_STATUS_NEW:
                    Xoonips_InstallUtils::installBlockByInfo($block,$module,$log);
                    break;
                case LEGACY_INSTALLINFO_STATUS_DELETED:
                    Xoonips_InstallUtils::uninstallBlockByFuncNum($block->mFuncNum,$module,$log);
                    break;
                default:
                    break;
            }
        }
    }

    /**
     * updateBlockTemplateByInfo
     *
     * @param   Legacy_BlockInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function updateBlockTemplateByInfo(/*** Legacy_BlockInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $blockHandler =&xoops_getmodulehandler('newblocks','legacy');
        $cri = new CriteriaCompo();
        $cri->add(new Criteria('dirname',$module->get('dirname')));
        $cri->add(new Criteria('func_num',$info->mFuncNum));
        $blocks =& $blockHandler->getObjects($cri);

        foreach($blocks as $block)
        {
            Xoonips_InstallUtils::uninstallBlockTemplate($block,$module,$log,true);
            Xoonips_InstallUtils::installBlockTemplate($block,$module,$log);
        }
    }

    /**
     * updateBlockByInfo
     *
     * @param   Legacy_BlockInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function updateBlockByInfo(/*** Legacy_BlockInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $blockHandler =&xoops_getmodulehandler('newblocks','legacy');
        $cri = new CriteriaCompo();
        $cri->add(new Criteria('dirname',$module->get('dirname')));
        $cri->add(new Criteria('func_num',$info->mFuncNum));
        $blocks =& $blockHandler->getObjects($cri);

        foreach($blocks as $block)
        {
            $filename = Xoonips_InstallUtils::replaceDirname(
                $info->mTemplate,
                $module->get('dirname'),
                $module->getInfo('trust_dirname')
            );
            $block->set('options',$info->mOptions);
            $block->set('name',$info->mName);
            $block->set('func_file',$info->mFuncFile);
            $block->set('show_func',$info->mShowFunc);
            $block->set('template',$filename['public']);
            if($blockHandler->insert($block))
            {
                $log->addReport(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_MSG_BLOCK_UPDATED,
                        $block->get('name')
                    )
                );
            }
            else
            {
                $log->addError(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_ERROR_BLOCK_UPDATED,
                        $block->get('name')
                    )
                );
            }
            Xoonips_InstallUtils::uninstallBlockTemplate($block,$module,$log,true);
            Xoonips_InstallUtils::installBlockTemplate($block,$module,$log);
        }
    }

    /**
     * installBlockByInfo
     *
     * @param   Legacy_BlockInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installBlockByInfo(/*** Legacy_BlockInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $filename = Xoonips_InstallUtils::replaceDirname(
            $info->mTemplate,
            $module->get('dirname'),
            $module->getInfo('trust_dirname')
        );

        $blockHandler =&xoops_gethandler('block');

        $block =& $blockHandler->create();
        $block->set('mid',$module->get('mid'));
        $block->set('func_num',$info->mFuncNum);
        $block->set('options',$info->mOptions);
        $block->set('name',$info->mName);
        $block->set('title',$info->mName);
        $block->set('dirname',$module->get('dirname'));
        $block->set('func_file',$info->mFuncFile);
        $block->set('show_func',$info->mShowFunc);
        $block->set('template',$filename['public']);
        $block->set('block_type','M');
        $block->set('c_type',1);

        if(!$blockHandler->insert($block))
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_BLOCK_INSTALLED,
                    $block->get('name')
                )
            );
            return false;
        }

        $log->addReport(
            XCube_Utils::formatString(
                _MI_XOONIPS_INSTALL_MSG_BLOCK_INSTALLED,
                $block->get('name')
            )
        );

        Xoonips_InstallUtils::installBlockTemplate($block,$module,$log);
        return true;
    }

    /**
     * uninstallBlockByFuncNum
     *
     * @param   int  $func_num
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function uninstallBlockByFuncNum(/*** int ***/ $func_num,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $blockHandler =&xoops_getmodulehandler('newblocks','legacy');
        $cri = new CriteriaCompo();
        $cri->add(new Criteria('dirname',$module->get('dirname')));
        $cri->add(new Criteria('func_num',$func_num));
        $blocks =& $blockHandler->getObjects($cri);

        $successFlag = true;
        foreach($blocks as $block)
        {
            if($blockHandler->delete($block))
            {
                $log->addReport(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_MSG_BLOCK_UNINSTALLED,
                        $block->get('name')
                    )
                );
            }
            else
            {
                $log->addError(
                    XCube_Utils::formatString(
                        _MI_XOONIPS_INSTALL_ERROR_BLOCK_UNINSTALLED,
                        $block->get('name')
                    )
                );
                $successFlag = false;
            }
        }
        return $successFlag;
    }

    /**
     * uninstallBlockTemplate
     *
     * @param   XoopsBlock  &$block
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     * @param   bool  $defaultOnly
     *
     * @return  bool
    **/
    public static function uninstallBlockTemplate(/*** XoopsBlock ***/ &$block,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log,/*** bool ***/ $defaultOnly = false)
    {
        $tplHandler =&xoops_gethandler('tplfile');
        $delTemplates =& $tplHandler->find($defaultOnly ? 'default' : null,'block',$module->get('mid'),$module->get('dirname'),$block->get('template'));

        if(is_array($delTemplates) && count($delTemplates) > 0)
        {
            foreach($delTemplates as $tpl)
            {
                if(!$tplHandler->delete($tpl))
                {
                    $log->addError(
                        XCube_Utils::formatString(
                            _MI_XOONIPS_INSTALL_ERROR_TPL_UNINSTALLED,
                            $tpl->get('tpl_file')
                        )
                    );
                }
            }
        }

        $log->addReport(
            XCube_Utils::formatString(
                _MI_XOONIPS_INSTALL_MSG_BLOCK_TPL_UNINSTALLED,
                $block->get('template')
            )
        );
        return true;
    }

    /**
     * installAllOfConfigs
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function installAllOfConfigs(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
    	Legacy_ModuleInstallUtils::installAllOfConfigs($module,$log);
    }

    /**
     * installConfigByInfo
     *
     * @param   Legacy_PreferenceInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function installConfigByInfo(/*** Legacy_PreferenceInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $configHandler =&xoops_gethandler('config');
        $config =& $configHandler->createConfig();
        $config->set('conf_modid',$module->get('mid'));
        $config->set('conf_catid',0);
        $config->set('conf_name',$info->mName);
        $config->set('conf_title',$info->mTitle);
        $config->set('conf_desc',$info->mDescription);
        $config->set('conf_formtype',$info->mFormType);
        $config->set('conf_valuetype',$info->mValueType);
        $config->setConfValueForInput($info->mDefault);
        $config->set('conf_order',$info->mOrder);

        if(count($info->mOption->mOptions) > 0)
        {
            foreach($info->mOption->mOptions as $opt)
            {
                $option = $configHandler->createConfigOption();
                $option->set('confop_name',$opt->mName);
                $option->set('confop_value',$opt->mValue);
                $config->setConfOptions($option);
                unset($option);
            }
        }

        if($configHandler->insertConfig($config))
        {
            $log->addReport(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_MSG_CONFIG_ADDED,
                    $config->get('conf_name')
                )
            );
        }
        else
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_CONFIG_ADDED,
                    $config->get('conf_name')
                )
            );
        }

    }

    /**
     * uninstallAllOfConfigs
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function uninstallAllOfConfigs(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
    	//if ($module->get('hasconfig') == 0) {
		//	return;
		//}

		$configHandler =& xoops_gethandler('config');
		$configs =& $configHandler->getConfigs(new Criteria('conf_modid', $module->get('mid')));

		if (count($configs) == 0) {
			return;
		}

		foreach ($configs as $config) {
			$configHandler->deleteConfig($config);
		}
    }


    /**
     * smartUpdateAllOfConfigs
     *
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  void
    **/
    public static function smartUpdateAllOfConfigs(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
    	Legacy_ModuleInstallUtils::smartUpdateAllOfPreferences($module,$log);

    }

    /**
     * updateConfigByInfo
     *
     * @param   Legacy_PreferenceInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function updateConfigByInfo(/*** Legacy_PreferenceInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $configHandler =&xoops_gethandler('config');
        $cri = new CriteriaCompo();
        $cri->add(new Criteria('conf_modid',$module->get('mid')));
        $cri->add(new Criteria('conf_catid',0));
        $cri->add(new Criteria('conf_name',$info->mName));
        $configs =& $configHandler->getConfigs($cri);

        if(!(count($configs) > 0 && is_object($configs[0])))
        {
            $log->addError(_MI_XOONIPS_INSTALL_ERROR_CONFIG_NOT_FOUND);
            return false;
        }

        $config =& $configs[0];
        $config->set('conf_title',$info->mTitle);
        $config->set('conf_desc',$info->mDescription);
        if($config->get('conf_formtype') != $info->mFormType && $config->get('conf_valuetype') != $info->mValueType)
        {
            $config->set('conf_formtype',$info->mFormType);
            $config->set('conf_valuetype',$info->mValueType);
            $config->setConfValueForInput($info->mDefault);
        }
        else
        {
            $config->set('conf_formtype',$info->mFormType);
            $config->set('conf_valuetype',$info->mValueType);
        }
        $config->set('conf_order',$info->mOrder);

        $options =& $configHandler->getConfigOptions(new Criteria('conf_id',$config->get('conf_id')));
        if(is_array($options))
        {
            foreach($options as $opt)
            {
                $configHandler->_oHandler->delete($opt);  // TODO will be use other method
            }
        }

        if(count($info->mOption->mOptions) > 0)
        {
            foreach($info->mOption->mOptions as $opt)
            {
                $option =& $configHandler->createConfigOption();
                $option->set('confop_name',$opt->mName);
                $option->set('confop_value',$opt->mValue);
                $option->set('conf_id',$option->get('conf_id'));    // TODO check conf_id is right
                $config->setConfOptions($option);
                unset($option);
            }
        }

        if($configHandler->insertConfig($config))
        {
            $log->addReport(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_MSG_CONFIG_UPDATED,
                    $config->get('conf_name')
                )
            );
            return true;
        }

        $log->addError(
            XCube_Utils::formatString(
                _MI_XOONIPS_INSTALL_ERROR_CONFIG_UPDATED,
                $config->get('conf_name')
            )
        );
        return false;
    }

    /**
     * updateConfigOrderByInfo
     *
     * @param   Legacy_PreferenceInformation  &$info
     * @param   XoopsModule  &$module
     * @param   Legacy_ModuleInstallLog  &$log
     *
     * @return  bool
    **/
    public static function updateConfigOrderByInfo(/*** Legacy_PreferenceInformation ***/ &$info,/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log)
    {
        $configHandler =&xoops_gethandler('config');
        $cri = new CriteriaCompo();
        $cri->add(new Criteria('conf_modid',$module->get('mid')));
        $cri->add(new Criteria('conf_catid',0));
        $cri->add(new Criteria('conf_name',$info->mName));
        $configs =& $configHandler->getConfigs($cri);

        if(!(count($configs) > 0 && is_object($configs[0])))
        {
            $log->addError(_MI_XOONIPS_INSTALL_ERROR_CONFIG_NOT_FOUND);
            return false;
        }

        $config =& $configs[0];
        $config->set('conf_order',$info->mOrder);
        if(!$configHandler->insertConfig($config))
        {
            $log->addError(
                XCube_Utils::formatString(
                    _MI_XOONIPS_INSTALL_ERROR_CONFIG_UPDATED,
                    $config->get('conf_name')
                )
            );
            return false;
        }
        return true;
    }

	public static function deleteAllOfNotifications(&$module, &$log)
	{
		Legacy_ModuleInstallUtils::deleteAllOfNotifications($module,$log);
	}

	public static function deleteAllOfComments(&$module, &$log)
	{
		Legacy_ModuleInstallUtils::deleteAllOfComments($module,$log);
	}
 	/**
   	* fix invalid xoops group permissions
   	*  - refer: http://www.xugj.org/modules/d3forum/index.php?topic_id=791
   	*
   	* @access public
   	* @return bool false if failure
   	*/
   	public static function fixGroupPermissions() {
	    global $xoopsDB;
	    // get invalid group ids
	    $table = $xoopsDB->prefix( 'group_permission' );
	    $table2 = $xoopsDB->prefix( 'groups' );
	    $sql = sprintf( 'SELECT DISTINCT `gperm_groupid` FROM `%s` LEFT JOIN `%s` ON `%s`.`gperm_groupid`=`%s`.`groupid` WHERE `gperm_modid`=1 AND `groupid` IS NULL', $table, $table2, $table, $table2 );
	    $result = $xoopsDB->query( $sql );
	    if ( ! $result ) {
	      return false;
	    }
	    $gids = array();
	    while( $myrow = $xoopsDB->fetchArray( $result ) ) {
	      $gids[] = $myrow['gperm_groupid'];
	    }
	    $xoopsDB->freeRecordSet( $result );
	    // remove all invalid group id entries
	    if ( count( $gids ) != 0 ) {
	      $sql = sprintf( 'DELETE FROM `%s` WHERE `gperm_groupid` IN (%s) AND `gperm_modid`=1', $table, implode( ',', $gids ) );
	      $result = $xoopsDB->query( $sql );
	      if ( ! $result ) {
	        return false;
	      }
	    }
	    return true;
  	}

	/**
	* create xoops group
	*
	* @access public
	* @param string $name group name
	* @param string $description group description
	* @return int created group id
	*/
	public static function createGroup($name, $description) {
		$member_handler =& xoops_gethandler('member');
		$group =& $member_handler->createGroup();
		$group->setVar('name', $name, true); // not gpc
		$group->setVar('description', $description, true); // not gpc
		$ret = $member_handler->insertGroup($group);
		if ($ret == false) {
			return false;
		}
		$gid = $group->getVar('groupid', 'n');
		return $gid;
	}

	/**
	* delete xoops group
	*
	* @access public
	* @param string $name group name
	* @param string $description group description
	* @return int created group id
	*/
	public static function deleteGroup($name, $description) {
		$cri = new CriteriaCompo();
		$cri->add(new Criteria('name', $name));
		$cri->add(new Criteria('description', $description));

		$grouphandler =& xoops_gethandler('group');
		$memberhandler =& xoops_gethandler('member');
		$grouppermhandler =& xoops_gethandler('groupperm');
		foreach($grouphandler->getObjects($cri) as $group) {

			if (!$memberhandler->delete($group)) {
				return false;
			}


			if (!$grouppermhandler->deleteByGroup($group->getVar('groupid'))) {
				return false;
			}
		}
		return true;
	}

	/**
   	* add user to xoops group
   	*
   	* @access public
   	* @param int $gid group id
   	* @param int $uid user id
   	* @return bool false if failure
   	*/
  	public static function addUserToXoopsGroup( $gid, $uid ) {
    	$member_handler =& xoops_gethandler('member');
    	if ( ! $member_handler->addUserToGroup( $gid, $uid ) ) {
      		return false;
    	}
    	$myuid = $GLOBALS['xoopsUser']->getVar( 'uid', 'n' );
    	if ( $myuid == $uid ) {
      		// update group cache and session
      		$mygroups = $member_handler->getGroupsByUser( $uid );
      		$GLOBALS['xoopsUser']->setGroups( $mygroups );
      		if ( isset( $_SESSION['xoopsUserGroups'] ) ) {
        		$_SESSION['xoopsUserGroups'] = $mygroups;
      		}
    	}
    	return true;
  	}

	/**
   	* set module read right
   	*
   	* @access public
   	* @param int $mid module id
   	* @param int $gid group id
   	* @param int $right has read right ?
   	* @return bool false if failure
   	*/
  	public static function setModuleReadRight( $mid, $gid, $right ) {
    	return self::setReadRight( true, $mid, $gid, $right );
  	}

	/**
   	* set block read right
   	*
   	* @access public
   	* @param int $bid block id
   	* @param int $gid group id
   	* @param int $right has read right ?
   	* @return bool false if failure
   	*/
  	public static function setBlockReadRight( $bid, $gid, $right ) {
    	return self::setReadRight( false, $bid, $gid, $right );
  	}

	/**
	* set xoops module/block read right
	*
	* @access private
	* @param bool $is_module true is module, false is block
	* @param int $iid module id or block id
	* @param int $gid group id
	* @param bool $right has read right?
	* @return bool false if failure
	*/
	private static function setReadRight( $is_module, $iid, $gid, $right ) {
    	$name = $is_module ? 'module_read' : 'block_read';
    	$criteria = new CriteriaCompo( new Criteria( 'gperm_name', $name ) );
    	$criteria->add( new Criteria( 'gperm_groupid', $gid ) );
    	$criteria->add( new Criteria( 'gperm_itemid', $iid ) );
    	$criteria->add( new Criteria( 'gperm_modid', 1 ) );
    	$gperm_handler =& xoops_gethandler( 'groupperm' );
    	$gperm_objs =& $gperm_handler->getObjects( $criteria );
    	if ( count( $gperm_objs ) > 0 ) {
      		// already exists
      		$gperm_obj = $gperm_objs[0];
      		if ( ! $right ) {
        		$gperm_handler->delete( $gperm_obj );
      		}
    	} else {
      		// not found
      		if ( $right ) {
        		$gperm_handler->addRight( $name, $iid, $gid );
      		}
    	}
		return true;
	}

	/**
	* set xoops module admin right
	*
	* @access public
	* @param int $mid module id
	* @param int $gid group id
	* @return bool false if failure
	*/
	public static function setAdminRight($mid, $gid) {
    	$criteria = new CriteriaCompo( new Criteria( 'gperm_name', 'module_admin' ) );
    	$criteria->add( new Criteria( 'gperm_groupid', $gid ) );
    	$criteria->add( new Criteria( 'gperm_itemid', $mid ) );
    	$criteria->add( new Criteria( 'gperm_modid', 1 ) );
    	$gperm_handler =& xoops_gethandler( 'groupperm' );
    	$gperm_objs =& $gperm_handler->getObjects( $criteria );
    	if ( count( $gperm_objs ) == 0 ) {
    		$gperm_handler->addRight( 'module_admin', $mid, $gid );
		}
    	return true;
	}

	/**
   	* get module block ids
   	*
   	* @access public
   	* @param int $mid block id
   	* @param int $show_func show function name
   	* @return array block ids
   	*/
  	public static function getBlockIds( $mid, $show_func ) {
    	global $xoopsDB;
    	$table = $xoopsDB->prefix( 'newblocks' );
    	$sql = sprintf( 'SELECT bid FROM `%s` WHERE `mid`=%u AND `show_func`=\'%s\'', $table, $mid, addslashes( $show_func ) );
    	$result = $xoopsDB->query( $sql );
    	if ( ! $result ) {
      		return false;
    	}
    	$ret = array();
    	while( $myrow = $xoopsDB->fetchArray( $result ) ) {
      		$ret[] = $myrow['bid'];
    	}
    	$xoopsDB->freeRecordSet( $result );
    	return $ret;
  	}

	/**
   	* set block position
   	*
   	* @access public
   	* @param int $bid block id
   	* @param bool $visible visible flag
   	* @param int $side
   	*        0: sideblock - left
   	*        1: sideblock - right
   	*        2: sideblock - left and right
   	*        3: centerblock - left
   	*        4: centerblock - right
   	*        5: centerblock - center
   	*        6: centerblock - left, right, center
   	* @param int $weight weight
   	* @return bool false if failure
   	*/
  	public static function setBlockPosition( $bid, $visible, $side, $weight ) {
    	$block = new XoopsBlock();
    	$block->load( $bid );
    	if ( ! is_null( $visible ) ) {
      		$block->setVar( 'visible', $visible ? 1 : 0, true ); // not gpc
    	}
    	if ( ! is_null( $side ) ) {
      		$block->setVar( 'side', $side, true ); // not gpc
    	}
    	if ( ! is_null( $weight ) ) {
      		$block->setVar( 'weight', $weight, true ); // not gpc
    	}
    	return $block->store();
  	}
	/**
   	* set block show page
   	*
   	* @access public
   	* @param int $bid block id
   	* @param int $mid
   	*     -1 : top page
   	*      0 : all pages
  	*    >=1 : module id
   	* @param bool $is_show
   	* @return bool false if failure
   	*/
  	public static function setBlockShowPage( $bid, $mid, $is_show ) {
    	global $xoopsDB;
    	$table = $xoopsDB->prefix( 'block_module_link' );
    	// check current status
    	$sql = sprintf( 'SELECT `block_id`,`module_id` FROM `%s` WHERE `block_id`=%u AND `module_id`=%d', $table, $bid, $mid );
    	if ( ! $result = $xoopsDB->query( $sql ) ) {
      		return false;
    	}
    	$count = $xoopsDB->getRowsNum( $result );
    	$xoopsDB->freeRecordSet( $result );
    	if ( $count == 0 ) {
      		// not exists
      		if ( $is_show ) {
        		$sql = sprintf( 'INSERT INTO `%s` (`block_id`,`module_id`) VALUES ( %u, %d )', $table, $bid, $mid );
        		if ( ! $result = $xoopsDB->query( $sql ) ) {
          			return false;
        		}
      		}
    	} else {
      		// already exists
      		if ( ! $is_show ) {
        		$sql = sprintf( 'DELETE FROM `%s` WHERE `block_id`=%u AND `module_id`=%d', $table, $bid, $mid );
        		if ( ! $result = $xoopsDB->query( $sql ) ) {
          			return false;
        		}
      		}
    	}
    	return true;
  	}
	/**
   	* get module id
   	*
   	* @access public
   	* @param string $dirname module directory name
   	* @return int module id
   	*/
  	public static function getModuleId( $dirname ) {
    	$module_handler =& xoops_gethandler( 'module' );
    	$module =& $module_handler->getByDirname( $dirname );
    	if ( ! is_object( $module ) ) {
      		return false;
    	}
    	$mid = $module->getVar( 'mid', 'n' );
    	return $mid;
  	}
	/**
   	* set start module
   	*
   	* @access public
   	* @param string $dirname module directory name,  '--' means no module.
   	* @return bool false if failure
   	*/
  	function setStartupPageModule( $dirname ) {
    	$config_handler =& xoops_gethandler( 'config' );
    	$criteria = new CriteriaCompo( new Criteria( 'conf_modid', 0 ) );
    	$criteria->add( new Criteria('conf_catid', XOOPS_CONF ) );
    	$criteria->add( new Criteria('conf_name', 'startpage') );
    	$configs =& $config_handler->getConfigs( $criteria );
    	if ( count( $configs ) != 1 ){
      		return false;
    	}
    	list( $config ) = $configs;
    	$config->setConfValueForInput( $dirname );
    	return $config_handler->insertConfig( $config );
  	}

  	/**
   	* enable xoops notificaiton
   	*
  	* @access public
   	* @param string $mid module id
   	* @param string $category
   	* @param string $event
   	* @return bool false if failure
   	*/
  	function enableNotification( $mid, $category, $event ) {
    	global $xoopsDB;
    	$config_handler =& xoops_gethandler( 'config' );
    	$criteria = new CriteriaCompo();
    	$criteria->add( new Criteria( 'conf_name', 'notification_events' ) );
    	$criteria->add( new Criteria( 'conf_modid', $mid ) );
    	$criteria->add( new Criteria( 'conf_catid', 0 ) );
    	$config_items = $config_handler->getConfigs( $criteria );
    	if ( count( $config_items ) != 1 ) {
      		return false;
    	} else {
      		$config_item = $config_items[0];
      		$option_value = $category.'-'.$event;
      		$option_values = $config_item->getConfValueForOutput();
      		if ( ! in_array( $option_value, $option_values ) ){
        		$option_values[] = $option_value;
        		$config_item->setConfValueForInput( $option_values );
        		$config_item_handler = new XoopsConfigItemHandler( $xoopsDB );
        		$config_item_handler->insert( $config_item );
      		}
    	}
    	return true;
  	}

  	/**
   	* subscribe user to xoops notificaiton
   	*
   	* @access public
   	* @param string $mid module id
   	* @param string $uid user id
   	* @param string $category
   	* @param string $event
   	* @return bool false if failure
   	*/
  	function subscribeNotification( $mid, $uid, $category, $event ) {
    	$notification_handler =& xoops_gethandler( 'notification' );
    	$notification_handler->subscribe( $category, 0, $event, null, $mid, $uid );
    	return true;
  	}

    /**
     * XOOPS user pickup
     *
     * @access public
     * @param int uid user id
     * @param bool is_certified initial certification state
     * @param string dirname
     * @return bool false if failure
     */
    public static function pickupXoopsUser($uid, $activate, $dirname, $isModerator) {
        // create user root index
        $indexBean = Xoonips_BeanFactory::getBean('IndexBean', $dirname);
        $index_id = $indexBean->insertPrivateIndex($uid);
        if ($index_id === false) {
            return false;
        }

        // set xoonips user information
        $usersBean = Xoonips_BeanFactory::getBean('UsersBean', XCUBE_CORE_USER_MODULE_NAME);

		$userInfo = $usersBean->getUserBasicInfo($uid);
		if ($userInfo == false) {
			return false;
		}

        $userInfo['activate'] = $activate;
        $userInfo['index_id'] = $index_id;
        if ($isModerator) {
            $configBean = Xoonips_BeanFactory::getBean('ConfigBean', $dirname);
            $userInfo['index_number_limit'] = $configBean->getConfig('private_index_number_limit');
            $userInfo['item_number_limit'] = $configBean->getConfig('private_item_number_limit');
            $userInfo['item_storage_limit'] = $configBean->getConfig('private_item_storage_limit');
        }

		if (!$usersBean->pickupUser($userInfo)) {
			return false;
		}

        // record event logs
        $log = Xoonips_BeanFactory::getBean('EventLogBean', $dirname);
        $log->recordRequestInsertAccountEvent($uid);
        if ($activate == Xoonips_Enum::USER_CERTIFIED) {
            $log->recordCertifyAccountEvent($uid);
        }
        return true;
    }

    public static function installDataAutomatically(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log) {
		$dirname = $module->getVar('dirname');
		$trustDirname = $module->modinfo['trust_dirname'];
		if (self::installConfig($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_config'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_config'));
			return false;
		}
		if (self::installDataAndViewType($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_view_type,' . $dirname . '_data_type'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_view_type,' . $dirname . '_data_type'));
			return false;
		}
		if (self::installComplement($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_complement'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_complement'));
			return false;
		}
		if (self::installItemFieldValue($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_item_field_value_set'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_item_field_value_set'));
			return false;
		}
		if (self::installOaipmh($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_oaipmh_schema'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_oaipmh_schema'));
			return false;
		}

		if (self::installItemtypeUtility($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_item_type_sort,' . $dirname . '_item_type_search_condition'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_item_type_sort,' . $dirname . '_item_type_search_condition'));
			return false;
		}
		if (self::installDefaultItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_DATA_INSTALLED, $dirname . '_item_field_group,' . $dirname . '_item_field_detail'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_DATAL_INSTALLED, $dirname . '_item_field_group,' . $dirname . '_item_field_detail'));
			return false;
		}

		if (self::installArticleItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Article'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Article'));
			return false;
		}
		if (self::installBinderItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Binder'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Binder'));
			return false;
		}
		if (self::installBookItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Book'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Book'));
			return false;
		}
		if (self::installConferenceItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Conference'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Conference'));
			return false;
		}
		if (self::installDataItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Data'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Data'));
			return false;
		}
		if (self::installFileItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'File'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'File'));
			return false;
		}
		if (self::installMemoItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Memo'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Memo'));
			return false;
		}
		if (self::installModelItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Model'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Model'));
			return false;
		}
		if (self::installPaperItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Paper'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Paper'));
			return false;
		}
		if (self::installPresentationItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Presentation'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Presentation'));
			return false;
		}
		if (self::installSimulatorItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Simulator'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Simulator'));
			return false;
		}
		if (self::installStimulusItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Stimulus'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Stimulus'));
			return false;
		}
		if (self::installToolItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Tool'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Tool'));
			return false;
		}
		if (self::installUrlItemtype($dirname, $trustDirname)) {
			$log->addReport(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_MSG_ITEMTYPE_INSTALLED, 'Url'));
		} else {
			$log->addError(XCube_Utils::formatString(_MI_XOONIPS_INSTALL_ERROR_ITEMTYPEL_INSTALLED, 'Url'));
			return false;
		}
		if (!self::installIndex($dirname, $trustDirname)) {
			return false;
		}
		if (self::chkXleprogress()) {
			if (!self::installXleprogress($dirname, $trustDirname)) {
				return false;
			}
		}
	}

    public static function installXleprogressAutomatically(/*** XoopsModule ***/ &$module,/*** Legacy_ModuleInstallLog ***/ &$log) {
		$dirname = $module->getVar('dirname');
		$trustDirname = $module->modinfo['trust_dirname'];
		return self::installXleprogress($dirname, $trustDirname);
	}

    private static function installConfig($dirname, $trustDirname) {
    	$configlist = array(
			array('name' => 'moderator_gid', 'value' => '1'),
			array('name' => 'upload_dir', 'value' => '/var/tmp'),
			array('name' => 'magic_file_path', 'value' => '/usr/share/misc/magic'),
			array('name' => 'repository_name', 'value' => ''),
			array('name' => 'repository_nijc_code', 'value' => ''),
			array('name' => 'repository_deletion_track', 'value' => '30'),
			array('name' => 'proxy_host', 'value' => ''),
			array('name' => 'proxy_port', 'value' => '80'),
			array('name' => 'proxy_user', 'value' => ''),
			array('name' => 'proxy_pass', 'value' => ''),
			array('name' => 'private_item_number_limit', 'value' => '0'),
			array('name' => 'private_index_number_limit', 'value' => '0'),
			array('name' => 'private_item_storage_limit', 'value' => '0'),
			array('name' => 'group_item_number_limit', 'value' => '0'),
			array('name' => 'group_index_number_limit', 'value' => '0'),
			array('name' => 'group_item_storage_limit', 'value' => '0'),
			array('name' => 'group_making', 'value' => 'off'),
			array('name' => 'group_making_certify', 'value' => 'on'),
			array('name' => 'group_publish_certify', 'value' => 'on'),
			array('name' => 'certify_item', 'value' => 'on'),
			array('name' => 'download_file_compression', 'value' => 'on'),
			array('name' => 'export_enabled', 'value' => 'off'),
			array('name' => 'private_import_enabled', 'value' => 'off'),
			array('name' => 'message_sign', 'value' =>
'{X_SITENAME}({X_SITEURL})
' . _MI_XOONIPS_ISTALL_MESSAGE_SIGN . ':{X_ADMINMAIL}'),
			array('name' => 'access_key', 'value' => 'AKIAJAOLV5RWPY4ECXMA'),
			array('name' => 'secret_access_key', 'value' => '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'),
		);
    	$configBean = Xoonips_BeanFactory::getBean('ConfigBean', $dirname, $trustDirname);
		foreach ($configlist as $config) {
    		if (!$configBean->insertConfig($config['name'], $config['value'])) {
    			return false;
    		}
    	}
    	return true;
	}

    private static function installDataAndViewType($dirname, $trustDirname) {
    	$datatypelist = array(
    		array('name' => 'int', 'module' => 'DataTypeInt'),
			array('name' => 'float', 'module' => 'DataTypeFloat'),
			array('name' => 'double', 'module' => 'DataTypeDouble'),
			array('name' => 'char', 'module' => 'DataTypeChar'),
			array('name' => 'varchar', 'module' => 'DataTypeVarchar'),
			array('name' => 'text', 'module' => 'DataTypeText'),
			array('name' => 'date', 'module' => 'DataTypeDate'),
			array('name' => 'datetime', 'module' => 'DataTypeDatetime'),
			array('name' => 'blob', 'module' => 'DataTypeBlob')
		);
		$viewtypelist = array(
			array('preselect' => 0, 'multi' => 1, 'name' => 'hidden', 'module' => 'ViewTypeHidden'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'text', 'module' => 'ViewTypeText'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'textarea', 'module' => 'ViewTypeTextArea'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'radio', 'module' => 'ViewTypeRadioBox'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'checkbox', 'module' => 'ViewTypeCheckBox'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'select', 'module' => 'ViewTypeComboBox'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'id', 'module' => 'ViewTypeId'),
			array('preselect' => 1, 'multi' => 1, 'name' => 'title', 'module' => 'ViewTypeTitle'),
			array('preselect' => 1, 'multi' => 1, 'name' => 'keyword', 'module' => 'ViewTypeKeyword'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'last update', 'module' => 'ViewTypeLastUpdate'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'create date', 'module' => 'ViewTypeCreateDate'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'create user', 'module' => 'ViewTypeCreateUser'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'change log', 'module' => 'ViewTypeChangeLog'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'index', 'module' => 'ViewTypeIndex'),
			array('preselect' => 1, 'multi' => 0, 'name' => 'relation item', 'module' => 'ViewTypeRelatedTo'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'date(yyyy mm dd)', 'module' => 'ViewTypeDate'),
			array('preselect' => 0, 'multi' => 0, 'name' => 'preview', 'module' => 'ViewTypePreview'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'file upload', 'module' => 'ViewTypeFileUpload'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'file type', 'module' => 'ViewTypeFileType'),
			array('preselect' => 0, 'multi' => 0, 'name' => 'download limit', 'module' => 'ViewTypeDownloadLimit'),
			array('preselect' => 0, 'multi' => 0, 'name' => 'download notify', 'module' => 'ViewTypeDownloadNotify'),
			array('preselect' => 0, 'multi' => 0, 'name' => 'readme', 'module' => 'ViewTypeReadme'),
			array('preselect' => 0, 'multi' => 0, 'name' => 'rights', 'module' => 'ViewTypeRights'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'url', 'module' => 'ViewTypeUrl'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'pubmed id', 'module' => 'ViewTypePubmedId'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'isbn', 'module' => 'ViewTypeIsbn'),
			array('preselect' => 0, 'multi' => 1, 'name' => 'kana', 'module' => 'ViewTypeKana')
    	);
		$relationlist = array(
			array('view_type_id' => 'hidden', 'data_type_id' => 'int', 'data_length' => 11, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'float', 'data_length' => 24, 'data_decimal_places' => 1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'double', 'data_length' => 53, 'data_decimal_places' => 1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'char', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'date', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'datetime', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'hidden', 'data_type_id' => 'blob', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'int', 'data_length' => 11, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'float', 'data_length' => 24, 'data_decimal_places' => 1),
			array('view_type_id' => 'text', 'data_type_id' => 'double', 'data_length' => 53, 'data_decimal_places' => 1),
			array('view_type_id' => 'text', 'data_type_id' => 'char', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'date', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'datetime', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'text', 'data_type_id' => 'blob', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'textarea', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'radio', 'data_type_id' => 'int', 'data_length' => 11, 'data_decimal_places' => -1),
			array('view_type_id' => 'radio', 'data_type_id' => 'float', 'data_length' => 24, 'data_decimal_places' => 1),
			array('view_type_id' => 'radio', 'data_type_id' => 'double', 'data_length' => 53, 'data_decimal_places' => 1),
			array('view_type_id' => 'radio', 'data_type_id' => 'char', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'radio', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'checkbox', 'data_type_id' => 'int', 'data_length' => 11, 'data_decimal_places' => -1),
			array('view_type_id' => 'checkbox', 'data_type_id' => 'float', 'data_length' => 24, 'data_decimal_places' => 1),
			array('view_type_id' => 'checkbox', 'data_type_id' => 'double', 'data_length' => 53, 'data_decimal_places' => 1),
			array('view_type_id' => 'checkbox', 'data_type_id' => 'char', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'checkbox', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'select', 'data_type_id' => 'int', 'data_length' => 11, 'data_decimal_places' => -1),
			array('view_type_id' => 'select', 'data_type_id' => 'float', 'data_length' => 24, 'data_decimal_places' => 1),
			array('view_type_id' => 'select', 'data_type_id' => 'double', 'data_length' => 53, 'data_decimal_places' => 1),
			array('view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'select', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'id', 'data_type_id' => 'blob', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1),
			array('view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'date', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1),
			array('view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1),
			array('view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1),
			array('view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1),
			array('view_type_id' => 'url', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'pubmed id', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1),
			array('view_type_id' => 'isbn', 'data_type_id' => 'char', 'data_length' => 13, 'data_decimal_places' => -1),
			array('view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1),
			array('view_type_id' => 'kana', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1)
		);
    	$dataTypeBean = Xoonips_BeanFactory::getBean('DataTypeBean', $dirname, $trustDirname);
    	$viewTypeBean = Xoonips_BeanFactory::getBean('ViewTypeBean', $dirname, $trustDirname);
		$datatypeid = array();
    	foreach ($datatypelist as $datatype) {
    		$id = '';
    		if (!$dataTypeBean->insert($datatype, $id)) {
    			return false;
    		}
    		$datatypeid[$datatype['name']] = $id;
    	}
		$viewtypeid = array();
    	foreach ($viewtypelist as $viewtype) {
    		$id = '';
    		if (!$viewTypeBean->insert($viewtype, $id)) {
    			return false;
    		}
    		$viewtypeid[$viewtype['name']] = $id;
    	}
		foreach ($relationlist as $relation) {
			$relation['view_type_id'] = $viewtypeid[$relation['view_type_id']];
			$relation['data_type_id'] = $datatypeid[$relation['data_type_id']];
    		if (!$dataTypeBean->insertRelation($relation)) {
    			return false;
    		}
    	}
    	return true;
    }

    private static function installComplement($dirname, $trustDirname) {
    	$viewTypeBean = Xoonips_BeanFactory::getBean('ViewTypeBean', $dirname, $trustDirname);
    	$viewtypelist = $viewTypeBean->getViewtypeList();
		$viewtypeid = array();
    	foreach ($viewtypelist as $viewtype) {
    		$viewtypeid[$viewtype['name']] = $viewtype['view_type_id'];
    	}

    	$complementlist = array(
    		array('view_type_id' => $viewtypeid['preview'], 'title' => 'Preview', 'module' => NULL),
    		array('view_type_id' => $viewtypeid['url'], 'title' => 'URL', 'module' => NULL),
    		array('view_type_id' => $viewtypeid['pubmed id'], 'title' => 'Pubmed ID', 'module' => 'ComplementPubmedId'),
    		array('view_type_id' => $viewtypeid['isbn'], 'title' => 'ISBN', 'module' => 'ComplementIsbn'),
    		array('view_type_id' => $viewtypeid['kana'], 'title' => 'KANA', 'module' => 'ComplementKana'),
    	);

    	$detaillist = array(
    		array('complement_id' => 'Preview', 'code' => 'caption', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_CAPTION),
    		array('complement_id' => 'URL', 'code' => 'hits', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_HITS),
    		array('complement_id' => 'Pubmed ID', 'code' => 'title', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_TITLE),
    		array('complement_id' => 'Pubmed ID', 'code' => 'keyword', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_KEYWORD),
    		array('complement_id' => 'Pubmed ID', 'code' => 'author', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_AUTHOR),
    		array('complement_id' => 'Pubmed ID', 'code' => 'journal', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_JOURNAL),
    		array('complement_id' => 'Pubmed ID', 'code' => 'publicationyear', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_PUBLICATION_YEAR),
    		array('complement_id' => 'Pubmed ID', 'code' => 'volume', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_VOLUME),
   			array('complement_id' => 'Pubmed ID', 'code' => 'number', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_NUMBER),
    		array('complement_id' => 'Pubmed ID', 'code' => 'page', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_PAGE),
    		array('complement_id' => 'Pubmed ID', 'code' => 'abstract', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_ABSTRACT),
    		array('complement_id' => 'ISBN', 'code' => 'title', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_TITLE),
    		array('complement_id' => 'ISBN', 'code' => 'author', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_AUTHOR),
    		array('complement_id' => 'ISBN', 'code' => 'publisher', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_PUBLISHER),
    		array('complement_id' => 'ISBN', 'code' => 'publicationyear', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_PUBLICATION_YEAR),
    		array('complement_id' => 'ISBN', 'code' => 'url', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_URL),
    		array('complement_id' => 'KANA', 'code' => 'romaji', 'title' => _MI_XOONIPS_INSTALL_COMPLEMENT_ROMAJI),
    	);

    	$complementBean = Xoonips_BeanFactory::getBean('ComplementBean', $dirname, $trustDirname);
    	$complementid = array();
    	foreach ($complementlist as $complement) {
    		$id = '';
    		if (!$complementBean->insert($complement, $id)) {
    			return false;
    		}
    		$complementid[$complement['title']] = $id;
    	}
	   	foreach ($detaillist as $detail) {
    		$id = '';
    		$detail['complement_id'] = $complementid[$detail['complement_id']];
    		if (!$complementBean->insertDetail($detail, $id)) {
    			return false;
    		}
    	}
    	return true;
    }

    private function installItemFieldValue($dirname, $trustDirname) {
        $valueList = array(
        	array('select_name' => 'Language', 'title_id' => 'eng', 'title' => _MI_XOONIPS_INSTALL_LANG_ENGLISH),
        	array('select_name' => 'Language', 'title_id' => 'jpn', 'title' => _MI_XOONIPS_INSTALL_LANG_JAPANES),
        	array('select_name' => 'Language', 'title_id' => 'fra', 'title' => _MI_XOONIPS_INSTALL_LANG_FRENCH),
        	array('select_name' => 'Language', 'title_id' => 'deu', 'title' => _MI_XOONIPS_INSTALL_LANG_GERMAN),
        	array('select_name' => 'Language', 'title_id' => 'esl', 'title' => _MI_XOONIPS_INSTALL_LANG_SPANISH),
        	array('select_name' => 'Language', 'title_id' => 'ita', 'title' => _MI_XOONIPS_INSTALL_LANG_ITALIAN),
        	array('select_name' => 'Language', 'title_id' => 'dut', 'title' => _MI_XOONIPS_INSTALL_LANG_DUTCH),
        	array('select_name' => 'Language', 'title_id' => 'sve', 'title' => _MI_XOONIPS_INSTALL_LANG_SWEDISH),
        	array('select_name' => 'Language', 'title_id' => 'nor', 'title' => _MI_XOONIPS_INSTALL_LANG_NORWEGIAN),
        	array('select_name' => 'Language', 'title_id' => 'dan', 'title' => _MI_XOONIPS_INSTALL_LANG_DANISH),
        	array('select_name' => 'Language', 'title_id' => 'fin', 'title' => _MI_XOONIPS_INSTALL_LANG_FINNISH),
        	array('select_name' => 'Language', 'title_id' => 'por', 'title' => _MI_XOONIPS_INSTALL_LANG_PORTUGUESE),
        	array('select_name' => 'Language', 'title_id' => 'chi', 'title' => _MI_XOONIPS_INSTALL_LANG_CHINESE),
        	array('select_name' => 'Language', 'title_id' => 'kor', 'title' => _MI_XOONIPS_INSTALL_LANG_KOREAN),
    		//array('select_name' => 'Download limit', 'title_id' => '1', 'title' => _MI_XOONIPS_INSTALL_DOWNLOAD_LIMIT_LOGIN_USER),
    		//array('select_name' => 'Download limit', 'title_id' => '0', 'title' => _MI_XOONIPS_INSTALL_DOWNLOAD_LIMIT_EVERYONE),
    		//array('select_name' => 'Download notify', 'title_id' => '1', 'title' => _MI_XOONIPS_INSTALL_DOWNLOAD_NOTIFY_YES),
    		//array('select_name' => 'Download notify', 'title_id' => '0', 'title' => _MI_XOONIPS_INSTALL_DOWNLOAD_NOTIFY_NO),
    		array('select_name' => 'Conference file type', 'title_id' => 'powerpoint', 'title' => _MI_XOONIPS_INSTALL_CONFERENCE_FILE_TYPE_POWERPOINT),
    		array('select_name' => 'Conference file type', 'title_id' => 'pdf', 'title' => _MI_XOONIPS_INSTALL_CONFERENCE_FILE_TYPE_PDF),
    		array('select_name' => 'Conference file type', 'title_id' => 'illustrator', 'title' => _MI_XOONIPS_INSTALL_CONFERENCE_FILE_TYPE_ILLUSTRATOR),
    		array('select_name' => 'Conference file type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_CONFERENCE_FILE_TYPE_OTHER),
    		array('select_name' => 'Data type', 'title_id' => 'excel', 'title' => _MI_XOONIPS_INSTALL_DATA_TYPE_EXCEL),
    		array('select_name' => 'Data type', 'title_id' => 'movie', 'title' => _MI_XOONIPS_INSTALL_DATA_TYPE_MOVIE),
    		array('select_name' => 'Data type', 'title_id' => 'text', 'title' => _MI_XOONIPS_INSTALL_DATA_TYPE_TEXT),
    		array('select_name' => 'Data type', 'title_id' => 'picture', 'title' => _MI_XOONIPS_INSTALL_DATA_TYPE_PICTURE),
    		array('select_name' => 'Data type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_DATA_TYPE_OTHER),
    		array('select_name' => 'Model type', 'title_id' => 'matlab', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_MATLAB),
    		array('select_name' => 'Model type', 'title_id' => 'neuron', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_NEURON),
    		array('select_name' => 'Model type', 'title_id' => 'originalprogram', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_ORIGINALPROGRAM),
    		array('select_name' => 'Model type', 'title_id' => 'satellite', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_SATELLITE),
    		array('select_name' => 'Model type', 'title_id' => 'genesis', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_Genesis),
    		array('select_name' => 'Model type', 'title_id' => 'a-cell', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_ACELL),
    		array('select_name' => 'Model type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_MODAL_TYPE_OTHER),
    		array('select_name' => 'Presentation file type', 'title_id' => 'powerpoint', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_POWERPOINT),
    		array('select_name' => 'Presentation file type', 'title_id' => 'lotus', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_LOTUS),
    		array('select_name' => 'Presentation file type', 'title_id' => 'justsystem', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_JUSTSYSTEM),
    		array('select_name' => 'Presentation file type', 'title_id' => 'html', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_HTML),
    		array('select_name' => 'Presentation file type', 'title_id' => 'pdf', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_PDF),
    		array('select_name' => 'Presentation file type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_PRESENTATION_FILE_TYPE_OTHER),
    		array('select_name' => 'Simulator file type', 'title_id' => 'matlab', 'title' => _MI_XOONIPS_INSTALL_SIMULATOR_FILE_TYPE_MATLAB),
    		array('select_name' => 'Simulator file type', 'title_id' => 'mathematica', 'title' => _MI_XOONIPS_INSTALL_SIMULATOR_FILE_TYPE_MATHEMATICA),
    		array('select_name' => 'Simulator file type', 'title_id' => 'program', 'title' => _MI_XOONIPS_INSTALL_SIMULATOR_FILE_TYPE_PROGRAM),
    		array('select_name' => 'Simulator file type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_SIMULATOR_FILE_TYPE_OTHER),
    		array('select_name' => 'Stimulus type', 'title_id' => 'picture', 'title' => _MI_XOONIPS_INSTALL_STIMULUS_TYPE_PICTURE),
    		array('select_name' => 'Stimulus type', 'title_id' => 'movie', 'title' => _MI_XOONIPS_INSTALL_STIMULUS_TYPE_MOVIE),
    		array('select_name' => 'Stimulus type', 'title_id' => 'program', 'title' => _MI_XOONIPS_INSTALL_STIMULUS_TYPE_PROGRAM),
    		array('select_name' => 'Stimulus type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_STIMULUS_TYPE_OTHER),
    		array('select_name' => 'Tool file type', 'title_id' => 'matlab', 'title' => _MI_XOONIPS_INSTALL_TOOL_FILE_TYPE_MATLAB),
    		array('select_name' => 'Tool file type', 'title_id' => 'mathematica', 'title' => _MI_XOONIPS_INSTALL_TOOL_FILE_TYPE_MATHEMATICA),
    		array('select_name' => 'Tool file type', 'title_id' => 'program', 'title' => _MI_XOONIPS_INSTALL_TOOL_FILE_TYPE_PROGRAM),
    		array('select_name' => 'Tool file type', 'title_id' => 'other', 'title' => _MI_XOONIPS_INSTALL_TOOL_FILE_TYPE_OTHER),
    		/*array('select_name' => 'Rights', 'title_id' => 'Unported', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_UNPORTED),
    		array('select_name' => 'Rights', 'title_id' => 'Argentina', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_ARGENTINA),
    		array('select_name' => 'Rights', 'title_id' => 'Australia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_AUSTRALIA),
    		array('select_name' => 'Rights', 'title_id' => 'Austria', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_AUSTRIA),
    		array('select_name' => 'Rights', 'title_id' => 'Belgium', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_BELGIUM),
    		array('select_name' => 'Rights', 'title_id' => 'Brazil', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_BRAZIL),
    		array('select_name' => 'Rights', 'title_id' => 'Bulgaria', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_BULGARIA),
    		array('select_name' => 'Rights', 'title_id' => 'Canada', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_CANADA),
    		array('select_name' => 'Rights', 'title_id' => 'Chile', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_CHILE),
    		array('select_name' => 'Rights', 'title_id' => 'China_Mainland', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_CHINA_MAINLAND),
    		array('select_name' => 'Rights', 'title_id' => 'Colombia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_COLOMBIA),
    		array('select_name' => 'Rights', 'title_id' => 'Croatia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_CROATIA),
    		array('select_name' => 'Rights', 'title_id' => 'Denmark', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_DENMARK),
    		array('select_name' => 'Rights', 'title_id' => 'Ecuador', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_ECUADOR),
    		array('select_name' => 'Rights', 'title_id' => 'Finland', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_FINLAND),
    		array('select_name' => 'Rights', 'title_id' => 'France', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_FRANCE),
    		array('select_name' => 'Rights', 'title_id' => 'Germany', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_GERMANY),
    		array('select_name' => 'Rights', 'title_id' => 'Greece', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_GREECE),
    		array('select_name' => 'Rights', 'title_id' => 'Guatemala', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_GUATEMALA),
    		array('select_name' => 'Rights', 'title_id' => 'Hong_Kong', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_HONG_KONG),
    		array('select_name' => 'Rights', 'title_id' => 'Hungary', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_HUNGARY),
    		array('select_name' => 'Rights', 'title_id' => 'India', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_INDIA),
    		array('select_name' => 'Rights', 'title_id' => 'Israel', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_ISRAEL),
    		array('select_name' => 'Rights', 'title_id' => 'Italy', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_ITALY),
    		array('select_name' => 'Rights', 'title_id' => 'Japan', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_JAPAN),
    		array('select_name' => 'Rights', 'title_id' => 'Luxembourg', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_LUXEMBOURG),
    		array('select_name' => 'Rights', 'title_id' => 'Macedonia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_MACEDONIA),
    		array('select_name' => 'Rights', 'title_id' => 'Malaysia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_MALAYSIA),
    		array('select_name' => 'Rights', 'title_id' => 'Malta', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_MALTA),
    		array('select_name' => 'Rights', 'title_id' => 'Mexico', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_MEXICO),
    		array('select_name' => 'Rights', 'title_id' => 'Netherlands', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_NETHERLANDS),
    		array('select_name' => 'Rights', 'title_id' => 'New_Zealand', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_NEW_ZEALAND),
    		array('select_name' => 'Rights', 'title_id' => 'Norway', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_NORWAY),
    		array('select_name' => 'Rights', 'title_id' => 'Peru', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_PERU),
    		array('select_name' => 'Rights', 'title_id' => 'Philippines', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_PHILIPPINES),
    		array('select_name' => 'Rights', 'title_id' => 'Poland', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_POLAND),
    		array('select_name' => 'Rights', 'title_id' => 'Portugal', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_PORTUGAL),
    		array('select_name' => 'Rights', 'title_id' => 'Puerto_Rico', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_PUERTO_RICO),
    		array('select_name' => 'Rights', 'title_id' => 'Romania', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_ROMANIA),
    		array('select_name' => 'Rights', 'title_id' => 'Serbia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SERBIA),
    		array('select_name' => 'Rights', 'title_id' => 'Singapore', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SINGAPORE),
    		array('select_name' => 'Rights', 'title_id' => 'Slovenia', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SLOVENIA),
    		array('select_name' => 'Rights', 'title_id' => 'South_Africa', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SOUTH_AFRICA),
    		array('select_name' => 'Rights', 'title_id' => 'South_Korea', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SOUTH_KOREA),
    		array('select_name' => 'Rights', 'title_id' => 'Spain', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SPAIN),
    		array('select_name' => 'Rights', 'title_id' => 'Sweden', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SWEDEN),
    		array('select_name' => 'Rights', 'title_id' => 'Switzerland', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_SWITZERLAND),
    		array('select_name' => 'Rights', 'title_id' => 'Taiwan', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_TAIWAN),
    		array('select_name' => 'Rights', 'title_id' => 'UK_England_and_Wales', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_UK_ENGLAND_AND_WALES),
    		array('select_name' => 'Rights', 'title_id' => 'UK_Scotland', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_UK_SCOTLAND),
    		array('select_name' => 'Rights', 'title_id' => 'United_States', 'title' => _MI_XOONIPS_INSTALL_RIGHTS_UNITED_STATES),*/
		);
		$valueSetBean = Xoonips_BeanFactory::getBean('ItemFieldValueSetBean', $dirname, $trustDirname);
		$select_name = '';
    	foreach ($valueList as $value) {
    		if ($select_name != $value['select_name']) {
    			$weight = 1;
    			$select_name = $value['select_name'];
    		} else {
    			$weight = $weight + 1;
    		}
    		$value['weight'] = $weight;
    		if (!$valueSetBean->insertValue($value)) {
    			return false;
    		}
    	}
    	return true;
    }

    private function installOaipmh($dirname, $trustDirname) {
		$oaipmhlist = array(
			array('metadata_prefix' => 'junii2', 'name' => 'title', 'min_occurences' => 1, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'alternative', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'creator', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'subject', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NDC', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NDLC', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NDLSH', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'BSH', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'UDC', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'MeSH', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'DDC', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'LCC', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'LCSH', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NIIsubject', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'description', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'publisher', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'contributor', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'date', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'type', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NIItype', 'min_occurences' => 1, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'format', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'identifier', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'URI', 'min_occurences' => 1, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'fullTextURL', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'issn', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NCID', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'jtitle', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'volume', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'issue', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'spage', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'epage', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'dateofissued', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'source', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'language', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'relation', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'pmid', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'doi', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'junii2', 'name' => 'isVersionOf', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'hasVersion', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'isReplacedBy', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'replaces', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'isRequiredBy', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'requires', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'isPartOf', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'hasPart', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'isReferencedBy', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'references', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'isFormatOf', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'hasFormat', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'coverage', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'spatial', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NIIspatial', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'temporal', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'NIItemporal', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'rights', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'junii2', 'name' => 'textversion', 'min_occurences' => 0, 'max_occurences' => 1),
			array('metadata_prefix' => 'oai_dc', 'name' => 'title', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'creator', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'subject', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'description', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'publisher', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'contributor', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'date', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'type', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'type:NIItype', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'format', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'identifier', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'source', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'language', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'relation', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'coverage', 'min_occurences' => 0, 'max_occurences' => 0),
			array('metadata_prefix' => 'oai_dc', 'name' => 'rights', 'min_occurences' => 0, 'max_occurences' => 0),
		);
		$oaipmhvaluelist = array(
			array('schema_id' => 'junii2:NIItype', 'value' => 'Journal Article'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Thesis or Dissertation'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Departmental Bulletin Paper'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Conference Paper'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Presentation'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Book'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Technical Report'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Research Paper'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Article'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Preprint'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Learning Material'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Data or Dataset'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Software'),
			array('schema_id' => 'junii2:NIItype', 'value' => 'Others'),
			array('schema_id' => 'junii2:textversion', 'value' => 'author'),
			array('schema_id' => 'junii2:textversion', 'value' => 'publisher'),
			array('schema_id' => 'junii2:textversion', 'value' => 'none'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Journal Article'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Thesis or Dissertation'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Departmental Bulletin Paper'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Conference Paper'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Presentation'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Book'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Technical Report'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Research Paper'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Article'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Preprint'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Learning Material'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Data or Dataset'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Software'),
			array('schema_id' => 'oai_dc:type:NIItype', 'value' => 'Others'),
		);
		$oaipmhlinklist = array(
			array('schema_id1' => 'junii2:title', 'schema_id2' => 'oai_dc:title', 'number' => 1),
			array('schema_id1' => 'junii2:alternative', 'schema_id2' => 'oai_dc:title', 'number' => 2),
			array('schema_id1' => 'junii2:creator', 'schema_id2' => 'oai_dc:creator', 'number' => 1),
			array('schema_id1' => 'junii2:subject', 'schema_id2' => 'oai_dc:subject', 'number' => 1),
			array('schema_id1' => 'junii2:NDC', 'schema_id2' => 'oai_dc:subject', 'number' => 2),
			array('schema_id1' => 'junii2:NDLC', 'schema_id2' => 'oai_dc:subject', 'number' => 3),
			array('schema_id1' => 'junii2:NDLSH', 'schema_id2' => 'oai_dc:subject', 'number' => 4),
			array('schema_id1' => 'junii2:BSH', 'schema_id2' => 'oai_dc:subject', 'number' => 5),
			array('schema_id1' => 'junii2:UDC', 'schema_id2' => 'oai_dc:subject', 'number' => 6),
			array('schema_id1' => 'junii2:MeSH', 'schema_id2' => 'oai_dc:subject', 'number' => 7),
			array('schema_id1' => 'junii2:DDC', 'schema_id2' => 'oai_dc:subject', 'number' => 8),
			array('schema_id1' => 'junii2:LCC', 'schema_id2' => 'oai_dc:subject', 'number' => 9),
			array('schema_id1' => 'junii2:LCSH', 'schema_id2' => 'oai_dc:subject', 'number' => 10),
			array('schema_id1' => 'junii2:NIIsubject', 'schema_id2' => 'oai_dc:subject', 'number' => 11),
			array('schema_id1' => 'junii2:description', 'schema_id2' => 'oai_dc:description', 'number' => 1),
			array('schema_id1' => 'junii2:publisher', 'schema_id2' => 'oai_dc:publisher', 'number' => 1),
			array('schema_id1' => 'junii2:contributor', 'schema_id2' => 'oai_dc:contributor', 'number' => 1),
			array('schema_id1' => 'junii2:date', 'schema_id2' => 'oai_dc:date', 'number' => 1),
			array('schema_id1' => 'junii2:type', 'schema_id2' => 'oai_dc:type', 'number' => 1),
			array('schema_id1' => 'junii2:NIItype', 'schema_id2' => 'oai_dc:type:NIItype', 'number' => 1),
			array('schema_id1' => 'junii2:format', 'schema_id2' => 'oai_dc:format', 'number' => 1),
			array('schema_id1' => 'junii2:identifier', 'schema_id2' => 'oai_dc:identifier', 'number' => 1),
			array('schema_id1' => 'junii2:URI', 'schema_id2' => 'oai_dc:identifier', 'number' => 2),
			array('schema_id1' => 'junii2:fullTextURL', 'schema_id2' => 'oai_dc:identifier', 'number' => 3),
			array('schema_id1' => 'junii2:issn', 'schema_id2' => 'oai_dc:identifier', 'number' => 4),
			array('schema_id1' => 'junii2:NCID', 'schema_id2' => 'oai_dc:identifier', 'number' => 5),
			array('schema_id1' => 'junii2:jtitle', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:volume', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:issue', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:spage', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:epage', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:dateofissued', 'schema_id2' => 'oai_dc:identifier', 'number' => 6),
			array('schema_id1' => 'junii2:source', 'schema_id2' => 'oai_dc:source', 'number' => 1),
			array('schema_id1' => 'junii2:language', 'schema_id2' => 'oai_dc:language', 'number' => 1),
			array('schema_id1' => 'junii2:relation', 'schema_id2' => 'oai_dc:relation', 'number' => 1),
			array('schema_id1' => 'junii2:pmid', 'schema_id2' => 'oai_dc:relation', 'number' => 2),
			array('schema_id1' => 'junii2:doi', 'schema_id2' => 'oai_dc:relation', 'number' => 3),
			array('schema_id1' => 'junii2:isVersionOf', 'schema_id2' => 'oai_dc:relation', 'number' => 4),
			array('schema_id1' => 'junii2:hasVersion', 'schema_id2' => 'oai_dc:relation', 'number' => 5),
			array('schema_id1' => 'junii2:isReplacedBy', 'schema_id2' => 'oai_dc:relation', 'number' => 6),
			array('schema_id1' => 'junii2:replaces', 'schema_id2' => 'oai_dc:relation', 'number' => 7),
			array('schema_id1' => 'junii2:isRequiredBy', 'schema_id2' => 'oai_dc:relation', 'number' => 8),
			array('schema_id1' => 'junii2:requires', 'schema_id2' => 'oai_dc:relation', 'number' => 9),
			array('schema_id1' => 'junii2:isPartOf', 'schema_id2' => 'oai_dc:relation', 'number' => 10),
			array('schema_id1' => 'junii2:hasPart', 'schema_id2' => 'oai_dc:relation', 'number' => 11),
			array('schema_id1' => 'junii2:isReferencedBy', 'schema_id2' => 'oai_dc:relation', 'number' => 12),
			array('schema_id1' => 'junii2:references', 'schema_id2' => 'oai_dc:relation', 'number' => 13),
			array('schema_id1' => 'junii2:isFormatOf', 'schema_id2' => 'oai_dc:relation', 'number' => 14),
			array('schema_id1' => 'junii2:hasFormat', 'schema_id2' => 'oai_dc:relation', 'number' => 15),
			array('schema_id1' => 'junii2:coverage', 'schema_id2' => 'oai_dc:coverage', 'number' => 1),
			array('schema_id1' => 'junii2:spatial', 'schema_id2' => 'oai_dc:coverage', 'number' => 2),
			array('schema_id1' => 'junii2:NIIspatial', 'schema_id2' => 'oai_dc:coverage', 'number' => 3),
			array('schema_id1' => 'junii2:temporal', 'schema_id2' => 'oai_dc:coverage', 'number' => 4),
			array('schema_id1' => 'junii2:NIItemporal', 'schema_id2' => 'oai_dc:coverage', 'number' => 5),
			array('schema_id1' => 'junii2:rights', 'schema_id2' => 'oai_dc:rights', 'number' => 1),
		);

 		$oaipmhBean = Xoonips_BeanFactory::getBean('OaipmhSchemaBean', $dirname, $trustDirname);
		$schemaid = array();
        $metadata_prefix = '';
    	foreach ($oaipmhlist as &$oaipmh) {
    		if ($metadata_prefix != $oaipmh['metadata_prefix']) {
    			$weight = 1;
    			$metadata_prefix = $oaipmh['metadata_prefix'];
    		} else {
    			$weight = $weight + 1;
    		}
    		$oaipmh['weight'] = $weight;
			$id = '';
			if (!$oaipmhBean->insert($oaipmh, $id)) {
				return false;
			}
			$schemaid[($oaipmh['metadata_prefix'] . ':' . $oaipmh['name'])] = $id;
    	}
    	foreach ($oaipmhvaluelist as &$oaipmhvalue) {
    		$oaipmhvalue['schema_id'] = $schemaid[$oaipmhvalue['schema_id']];
			$id = '';
			if (!$oaipmhBean->insertValue($oaipmhvalue, $id)) {
				return false;
			}
    	}
    	foreach ($oaipmhlinklist as &$oaipmhlink) {
    		$oaipmhlink['schema_id1'] = $schemaid[$oaipmhlink['schema_id1']];
    		$oaipmhlink['schema_id2'] = $schemaid[$oaipmhlink['schema_id2']];
			if (!$oaipmhBean->insertLink($oaipmhlink)) {
				return false;
			}
    	}
		return true;
	}

    private function installDefaultItemtype($dirname, $trustDirname) {
    	$grouplist = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0)
    	);

    	$detaillist = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text',
    			'data_length' => -1, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
        	array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text',
    			'data_length' => -1, 'data_decimal_places' => -1, 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
        	array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar',
    			'data_length' => 255, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
        	array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int',
    			'data_length' => 10, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        	array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int',
    			'data_length' => 10, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        	array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int',
    			'data_length' => 10, 'data_decimal_places' => -1, 'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
        	array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text',
    			'data_length' => -1, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        	array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int',
    			'data_length' => 10, 'data_decimal_places' => -1, 'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
        	array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links',
    			'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int',
    			'data_length' => 10, 'data_decimal_places' => -1, 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);

		$groupBean = Xoonips_BeanFactory::getBean('ItemFieldGroupBean', $dirname, $trustDirname);
		$groupid = array();
		$weight = 1;
    	foreach ($grouplist as &$group) {
    		$group['item_type_id'] = 0;
    		$group['weight'] = $weight;
    		$group['preselect'] = 1;
    		$group['released'] = 1;
    		$group['update_id'] = NULL;
    		if (!$groupBean->insert($group, $id)) {
    			return false;
    		}
    		$groupid[$group['xml']] = $id;
    		$weight++;
    	}

       $viewTypeBean = Xoonips_BeanFactory::getBean('ViewTypeBean', $dirname, $trustDirname);
    	$viewtypelist = $viewTypeBean->getViewtypeList();
		$viewtypeid = array();
    	foreach ($viewtypelist as $viewtype) {
    		$viewtypeid[$viewtype['name']] = $viewtype['view_type_id'];
    	}

    	$dataTypeBean = Xoonips_BeanFactory::getBean('DataTypeBean', $dirname, $trustDirname);
    	$datatypelist = $dataTypeBean->getDatatypeList();
		$datatypeid = array();
    	foreach ($datatypelist as $datatype) {
    		$datatypeid[$datatype['name']] = $datatype['data_type_id'];
    	}

		$detailBean = Xoonips_BeanFactory::getBean('ItemFieldDetailBean', $dirname, $trustDirname);
		$group_id = '';
		$searchObj = array();
		foreach ($detaillist as &$detail) {
    		$id = '';
    		$l_group_id = $detail['group_id'];
    		$detail['table_name'] = $dirname . '_' . $detail['table_name'];
    		if ($group_id != $detail['group_id']) {
    			$weight = 1;
    			$group_id = $detail['group_id'];
    		} else {
    			$weight = $weight + 1;
    		}
    		$detail['view_type_id'] = $viewtypeid[$detail['view_type_id']];
			$detail['data_type_id'] = $datatypeid[$detail['data_type_id']];
    		$detail['weight'] = 1;
    		$detail['default_value'] = NULL;
    		$detail['list'] = NULL;
    		$detail['detail_display'] = 1;
    		$detail['nondisplay'] = 0;
    		$detail['item_type_id'] = 0;
    		$detail['group_id'] = 0;
    		$detail['preselect'] = 1;
    		$detail['released'] = 1;
    		$detail['update_id'] = NULL;
    		if (!$detailBean->insert($detail, $id)) {
    			return false;
    		}

    		$link = array('group_id' => $groupid[$l_group_id]
    		,'item_field_detail_id' => $id
    		,'weight' => $weight
    		,'edit' => 1
    		,'edit_weight' => $weight
    		,'released' => 1);
    		if (!$groupBean->insertLink($link, $lid)) {
    			return false;
    		}

    		// Quick Search Setting
    		$searchObj[] = array('condition_id' => _MI_XOONIPS_INSTALL_SEARCH_ALL, 'item_field_detail_id' => $id);
    		if ($detail['name'] == _MI_XOONIPS_INSTALL_COMMON_TITLE
    		|| $detail['name'] == _MI_XOONIPS_INSTALL_COMMON_KEYWORD) {
    			$searchObj[] = array('condition_id' => _MI_XOONIPS_INSTALL_SEARCH_TITLE_KEYWORD, 'item_field_detail_id' => $id);
    		}
        }

		if (!self::installItemSearch($dirname, $trustDirname, $searchObj)) {
			return false;
       	}

		return true;
    }

    private function installItemtypeUtility($dirname, $trustDirname) {
		$sortlist = array(
			array('title' => _MI_XOONIPS_INSTALL_COMMON_TITLE),
			array('title' => _MI_XOONIPS_INSTALL_COMMON_ID),
			array('title' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE),
			array('title' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE),
		);
		$sortBean = Xoonips_BeanFactory::getBean('ItemTypeSortBean', $dirname, $trustDirname);
        foreach ($sortlist as $sort) {
			$id = '';
    		if (!$sortBean->insert($sort['title'], $id)) {
    			return false;
    		}
    	}

		$searchlist = array(
			array('title' => _MI_XOONIPS_INSTALL_SEARCH_ALL),
			array('title' => _MI_XOONIPS_INSTALL_SEARCH_TITLE_KEYWORD),
		);
		$searchBean = Xoonips_BeanFactory::getBean('ItemTypeSearchConditionBean', $dirname, $trustDirname);
        foreach ($searchlist as $search) {
			$id = '';
    		if (!$searchBean->insert($search, $id)) {
    			return false;
    		}
    	}
		return true;
	}

    private function installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj,
		$complementObj, $sortObj, $searchObj, $oaipmhObj) {
    	$weight = 1;
    	foreach ($groupObj as &$obj) {
    		$obj['preselect'] = 0;

    		$obj['released'] = 1;
    		$obj['group_id'] = $obj['xml'];
    		$obj['weight'] = $weight;
    		$weight = $weight + 1;
    	}

    	$viewTypeBean = Xoonips_BeanFactory::getBean('ViewTypeBean', $dirname, $trustDirname);
    	$viewtypelist = $viewTypeBean->getViewtypeList();
		$viewtypeid = array();
    	foreach ($viewtypelist as $viewtype) {
    		$viewtypeid[$viewtype['name']] = $viewtype['view_type_id'];
    	}
    	$dataTypeBean = Xoonips_BeanFactory::getBean('DataTypeBean', $dirname, $trustDirname);
    	$datatypelist = $dataTypeBean->getDatatypeList();
		$datatypeid = array();
    	foreach ($datatypelist as $datatype) {
    		$datatypeid[$datatype['name']] = $datatype['data_type_id'];
    	}
       $group_id = '';
    	foreach ($detailObj as &$obj) {
    		$obj['preselect'] = 0;

    		$obj['released'] = 1;
    		$obj['table_name'] = $dirname . '_' . $obj['table_name'];
			if (!isset($obj['default_value'])) {
				$obj['default_value'] = NULL;
			}
			if (!isset($obj['list'])) {
				$obj['list'] = NULL;
			}
    		$obj['detail_display'] = 1;
    		$obj['nondisplay'] = 0;
			$obj['item_field_detail_id'] = $obj['group_id'] . ':' . $obj['xml'];
			$obj['view_type_id'] = $viewtypeid[$obj['view_type_id']];
			$obj['data_type_id'] = $datatypeid[$obj['data_type_id']];
    		if ($group_id != $obj['group_id']) {
    			$weight = 1;
    			$group_id = $obj['group_id'];
    		} else {
    			$weight = $weight + 1;
    		}
    		$obj['weight'] = $weight;
    	}

		if (count($complementObj) > 0) {
    		$complementBean = Xoonips_BeanFactory::getBean('ComplementBean', $dirname, $trustDirname);
    		$complementlist = $complementBean->getComplementInfo();
			$complementid = array();
			$complementtitle = array();
    		foreach ($complementlist as $complement) {
    			$complementid[$complement['title']] = $complement['complement_id'];
    			$complementtitle[$complement['complement_id']] = $complement['title'];
    		}
    		$complementdetaillist = $complementBean->getComplementDetailInfo();
			$complementdetailid = array();
    		foreach ($complementdetaillist as $complementdetail) {
    			$complementdetailid[($complementtitle[$complementdetail['complement_id']] . ':' . $complementdetail['code'])] = $complementdetail['complement_detail_id'];
    		}
    		foreach ($complementObj as &$obj) {
    			$obj['released'] = 1;
				$obj['complement_detail_id'] = $complementdetailid[$obj['complement_id']. ':' . $obj['complement_detail_id']];
    			$obj['complement_id'] = $complementid[$obj['complement_id']];
			}
		}

    	$map = array();
		$itemtypeBean = Xoonips_BeanFactory::getBean('ItemTypeBean', $dirname, $trustDirname);
		if (!$itemtypeBean->copyByObj($itemtypeObj, $map, false, true, false)) {
			return false;
		}

		$groupBean = Xoonips_BeanFactory::getBean('ItemFieldGroupBean', $dirname, $trustDirname);
		if (!$groupBean->copyByObjNG($groupObj, $map, false, true)) {
			return false;
		}

		foreach ($groupObj as $group) {
			$link = array('item_type_id' => $map['itemtype'][$itemtypeObj['item_type_id']]
    		,'group_id' => $map['group'][$group['group_id']]
    		,'weight' => $group['weight']
    		,'edit' => 1
    		,'edit_weight' => $group['weight']
    		,'released' => 1);
    		if (!$itemtypeBean->insertLink($link, $lid)) {
    			return false;
    		}
    	}

		$detailBean = Xoonips_BeanFactory::getBean('ItemFieldDetailBean', $dirname, $trustDirname);
		if (!$detailBean->copyByObjNG($detailObj, $map, false, true)) {
			return false;
		}

		foreach ($detailObj as $detail) {
			$link = array('group_id' => $map['group'][$detail['group_id']]
    		,'item_field_detail_id' => $map['detail'][$detail['item_field_detail_id']]
    		,'weight' => $detail['weight']
    		,'edit' => 1
    		,'edit_weight' => $detail['weight']
    		,'released' => 1);

    		$link_info = $groupBean->getGroupDetailById($link['group_id'], $link['item_field_detail_id']);
			if (count($link_info) == 0) {
	    		if (!$groupBean->insertLink($link, $lid)) {
    				return false;
    			}
			}
		}

		$linkBean = Xoonips_BeanFactory::getBean('ItemFieldDetailComplementLinkBean', $dirname, $trustDirname);
		if (!$linkBean->copyByObj($complementObj, $map, false, true)) {
			return false;
		}

		// create extend table
		$detailObj = $detailBean->getReleasedDetail($map['itemtype'][$itemtypeObj['item_type_id']]);
		if (!$detailBean->createExtendTable($detailObj)) {
			return false;
		}

		$sortBean = Xoonips_BeanFactory::getBean('ItemTypeSortBean', $dirname, $trustDirname);
		$sortlist = $sortBean->select();
		$sortid = array();
    	foreach ($sortlist as $sort) {
    		$sortid[$sort['title']] = $sort['sort_id'];
    	}
		$sortDetailBean = Xoonips_BeanFactory::getBean('ItemTypeSortDetailBean', $dirname, $trustDirname);
    	foreach ($sortObj as &$obj) {
    		$obj['sort_id'] = $sortid[$obj['sort_id']];
    		$obj['item_type_id'] = $map['itemtype'][$obj['item_type_id']];
			$obj['item_field_detail_id'] = $map['detail'][$obj['item_field_detail_id']];
    		if (!$sortDetailBean->insert($obj['sort_id'], $obj['item_type_id'], $obj['item_field_detail_id'])) {
				return false;
			}
		}

 		$oaipmhBean = Xoonips_BeanFactory::getBean('OaipmhSchemaBean', $dirname, $trustDirname);
		$schemalist = $oaipmhBean->getSchemaList();
		$schemaid = array();
		$schemaname = array();
    	foreach ($schemalist as $schema) {
			$schemaid[($schema['metadata_prefix'] . ':' . $schema['name'])] = $schema['schema_id'];
			$schemaname[$schema['schema_id']] = ($schema['metadata_prefix'] . ':' . $schema['name']);
    	}
		$valuesetid = array();
		$prefixlist = $oaipmhBean->getPrefixList();
    	foreach ($prefixlist as $prefix) {
			$valuesetlist = $oaipmhBean->getSchemaValueSetList($prefix);
			foreach ($valuesetlist as $valueset) {
				$valuesetid[$schemaname[$valueset['schema_id']] . ':' . $valueset['value']] = $valueset['seq_id'];
			}
		}
		$linkBean = Xoonips_BeanFactory::getBean('OaipmhSchemaItemtypeLinkBean', $dirname, $trustDirname);
    	foreach ($oaipmhObj as &$obj) {
			$schema = $obj['schema_id'];
    		$obj['schema_id'] = $schemaid[$obj['schema_id']];
    		$obj['item_type_id'] = $map['itemtype'][$obj['item_type_id']];
			if (strpos($schema, 'NIItype') != false || strpos($schema, 'textversion') != false) {
				$obj['item_field_detail_id'] = $valuesetid[$schema . ':' . $obj['item_field_detail_id']];
			} elseif ($obj['item_field_detail_id'] != 'http://' && $obj['item_field_detail_id'] != 'ID'
					&& $obj['item_field_detail_id'] != 'itemtype' && $obj['item_field_detail_id'] != 'meta_author') {
				$list = array();
				$idlist = explode(',', $obj['item_field_detail_id']);
				foreach ($idlist as $id) {
					$list[] = $map['detail'][$id];
				}
				$obj['item_field_detail_id'] = implode(',', $list);
			}
    		if (!$linkBean->insert($obj)) {
				return false;
			}
		}
		return true;
    }

    private function installArticleItemtype($dirname, $trustDirname) {
		$itemtypeObj = array('item_type_id' => 'Article', 'preselect' => 1, 'released' => 1, 'weight' => 1, 'name' => 'Article', 'description' => _MI_XOONIPS_INSTALL_ARTICEL_TITLE,
    			'icon' => 'icon_article.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_article" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br>
        <{if (count($item->get(authors,name)) != 0)}><{$item->get(authors,name)|xoops_escape}><br><{/if}>
        <{if ($item->get(jtitle,name)|count_characters != 0)}><{$item->get(jtitle,name)|xoops_escape}>.<{/if}>
        <{if ($item->get(jtitle,jtitle_volume)|count_characters != 0)}><{$item->get(jtitle,jtitle_volume)|xoops_escape}>,<{/if}>
        <{if (count($item->get(jtitle,jtitle_issue)) != 0)}><{$item->get(jtitle,jtitle_issue)|xoops_escape}><{/if}>
        <{if ($item->get(jtitle,jtitle_year)|count_characters != 0)}>(<{$item->get(jtitle,jtitle_year)|xoops_escape}><{/if}>
        <{if ($item->get(jtitle,jtitle_month)|count_characters != 0)}>.<{$item->get(jtitle,jtitle_month)|xoops_escape}><{/if}>
        <{if ($item->get(jtitle,jtitle_year)|count_characters != 0 && $item->get(jtitle,jtitle_month)|count_characters != 0)}>)<{/if}>
        <{if ($item->get(jtitle,jtitle_spage)|count_characters != 0 || $item->get(jtitle,jtitle_epage)|count_characters != 0) }>,p.<{/if}>
        <{if ($item->get(jtitle,jtitle_spage)|count_characters != 0)}><{$item->get(jtitle,jtitle_spage)|xoops_escape}>-<{/if}>
        <{if ($item->get(jtitle,jtitle_epage)|count_characters != 0)}><{$item->get(jtitle,jtitle_epage)|xoops_escape}><{/if}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>'
);

        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_SUBTITLE, 'xml' => 'sub_title', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR, 'xml' => 'authors', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_EDITION, 'xml' => 'edition', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_PUBLISH_PLACE, 'xml' => 'publish_place', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_PUBLISHER, 'xml' => 'publisher', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_PHYSICAL_DESCRIPTIONS, 'xml' => 'physical_descriptions', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE, 'xml' => 'jtitle', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_OTHER, 'xml' => 'id_locals', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_ABSTRACT, 'xml' => 'abstract', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_CONTENTS, 'xml' => 'table_of_contents', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_NDC, 'xml' => 'ndc_classifications', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_NOTES, 'xml' => 'notes', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_RESOURCE, 'xml' => 'type_of_resource', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_GENRE, 'xml' => 'genre', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_URI, 'xml' => 'uris', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_ARTICEL_ACCESS_CONDITION, 'xml' => 'access_condition', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);

    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_TITLE_NAME,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'sub_title', 'name' => _MI_XOONIPS_INSTALL_COMMON_NAME,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'sub_title', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'sub_title', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR_ID,
    			'xml' => 'author_id', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_NAME,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR_AFFILIATION,
    			'xml' => 'author_affiliation', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR_AFFILIATION_TRANSLATION,
    			'xml' => 'author_affiliation_translation', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR_ROLE,
    			'xml' => 'author_role', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_ARTICLE_AUTHOR_EXTERNAL_LINK,
    			'xml' => 'author_link', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'edition', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_EDITION,
    			'xml' => 'edition', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publish_place', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_PUBLISH_PLACE,
    			'xml' => 'publish_place', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_COMMON_NAME,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE_FROM,
    			'xml' => 'year_f', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE_TO,
    			'xml' => 'year_t', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE_CREATE,
    			'xml' => 'date_create', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE_UPDATE,
    			'xml' => 'date_update', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_DATE_RECORD,
    			'xml' => 'date_record', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'physical_descriptions', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_NAME,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_TRANSLATION,
    			'xml' => 'jtitle_translation', 'view_type_id' => 'text', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_VOLUME,
    			'xml' => 'jtitle_volume', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_ISSUE,
    			'xml' => 'jtitle_issue', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_YEAR,
    			'xml' => 'jtitle_year', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_MONTH,
    			'xml' => 'jtitle_month', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_FROM_PAGE,
    			'xml' => 'jtitle_spage', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_TO_PAGE,
    			'xml' => 'jtitle_epage', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'id_locals', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_OTHER_ISSN,
    			'xml' => 'issn', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'id_locals', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_OTHER_ISBN,
    			'xml' => 'isbn', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 50, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'id_locals', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_OTHER_URI,
    			'xml' => 'uri', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 1000, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'id_locals', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_OTHER_OTHER,
    			'xml' => 'etc', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 1000, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'abstract', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_ABSTRACT,
    			'xml' => 'abstract', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'table_of_contents', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_CONTENTS,
    			'xml' => 'table_of_contents', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'ndc_classifications', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_NDC,
    			'xml' => 'ndc_classification', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'notes', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'type_of_resource', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_RESOURCE,
    			'xml' => 'type_of_resource', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'genre', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_GENRE,
    			'xml' => 'genre', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_IMAGES,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_COMMON_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'uris', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_URI,
    			'xml' => 'uri', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 1000, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'access_condition', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_ACCESS_CONDITION,
    			'xml' => 'access_condition', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);

		foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

    	$complementObj = array(
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'sub_title:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'sub_title:romaji'),
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'authors:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'authors:romaji'),
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'publisher:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'publisher:romaji'),
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
    	);

    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

    	$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
    	);

    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

    	$searchObj = array();

    	foreach ($searchObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:alternative', 'item_field_detail_id' => 'sub_title:name', 'value' => NULL),
			array('schema_id' => 'junii2:alternative', 'item_field_detail_id' => 'sub_title:kana', 'value' => NULL),
			array('schema_id' => 'junii2:alternative', 'item_field_detail_id' => 'sub_title:romaji', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:NDC', 'item_field_detail_id' => 'ndc_classifications:ndc_classification', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'abstract:abstract', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'table_of_contents:table_of_contents', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'notes:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'publisher:name', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:year_f,date:year_t', 'value' => 'sprintf("%s-%s",1,2)'),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date_create', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date_update', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date_record', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'type_of_resource:type_of_resource', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'id_locals:isbn', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'id_locals:uri', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'id_locals:etc', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:issn', 'item_field_detail_id' => 'id_locals:issn', 'value' => NULL),
			array('schema_id' => 'junii2:jtitle', 'item_field_detail_id' => 'jtitle:name', 'value' => NULL),
			array('schema_id' => 'junii2:volume', 'item_field_detail_id' => 'jtitle:jtitle_volume', 'value' => NULL),
			array('schema_id' => 'junii2:issue', 'item_field_detail_id' => 'jtitle:jtitle_issue', 'value' => NULL),
			array('schema_id' => 'junii2:spage', 'item_field_detail_id' => 'jtitle:jtitle_spage', 'value' => NULL),
			array('schema_id' => 'junii2:epage', 'item_field_detail_id' => 'jtitle:jtitle_epage', 'value' => NULL),
			array('schema_id' => 'junii2:dateofissued', 'item_field_detail_id' => 'jtitle:jtitle_year,jtitle:jtitle_month', 'value' => 'sprintf("%s.%s",1,2)'),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'junii2:relation', 'item_field_detail_id' => 'authors:author_link', 'value' => NULL),
			array('schema_id' => 'junii2:textversion', 'item_field_detail_id' => 'none', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'sub_title:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'sub_title:kana', 'value' => NULL),
			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'sub_title:romaji', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'ndc_classifications:ndc_classification', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'abstract:abstract', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'table_of_contents:table_of_contents', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'notes:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'publisher:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:year_f,date:year_t', 'value' => 'sprintf("%s-%s",1,2)'),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date_create', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date_update', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date_record', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'type_of_resource:type_of_resource', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'id_locals:isbn', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'id_locals:uri', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'id_locals:etc', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'id_locals:issn', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' =>
				'jtitle:name,jtitle:jtitle_volume,jtitle:jtitle_issue,jtitle:jtitle_year,jtitle:jtitle_month,jtitle:jtitle_spage,jtitle:jtitle_epage',
				'value' => 'sprintf("%s Vol.%s, No.%s (%s.%s) p.%s-%s",1,2,3,4,5,6,7)'),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'oai_dc:relation', 'item_field_detail_id' => 'authors:author_link', 'value' => NULL),
		);

    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Article';
    	}

		return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installBinderItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Binder', 'preselect' => 1, 'released' => 1, 'weight' => 2, 'name' => 'Binder', 'description' => _MI_XOONIPS_INSTALL_BINDER_TITLE,
    			'icon' => 'icon_binder.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_binder" />
    </td>
    <td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
    	$groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Binder';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Binder';
    	}
    	$complementObj = array();
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Binder';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Data or Dataset', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'junii2:hasPart', 'item_field_detail_id' => 'links:link', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'oai_dc:relation', 'item_field_detail_id' => 'links:link', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Binder';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
	}

	private function installBookItemtype($dirname, $trustDirname) {
     	$itemtypeObj = array('item_type_id' => 'Book', 'preselect' => 1, 'released' => 1, 'weight' => 3, 'name' => 'Book', 'description' => _MI_XOONIPS_INSTALL_BOOK_TITLE,
    			'icon' => 'icon_article.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_book" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(authors,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
    	$groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_ISBN, 'xml' => 'isbn', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_AUTHOR, 'xml' => 'authors', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_EDITOR, 'xml' => 'editor', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_PUBLISHER, 'xml' => 'publisher', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_PUBLICATION_YEAR, 'xml' => 'publication_year', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_URL, 'xml' => 'url', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_PDF_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
     		$obj['item_type_id'] = 'Book';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'isbn', 'name' => _MI_XOONIPS_INSTALL_BOOK_ISBN,
    			'xml' => 'isbn', 'view_type_id' => 'isbn', 'data_type_id' => 'char', 'data_length' => 13, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_BOOK_AUTHOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'editor', 'name' => _MI_XOONIPS_INSTALL_BOOK_EDITOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_BOOK_PUBLISHER,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publisher', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publication_year', 'name' => _MI_XOONIPS_INSTALL_BOOK_PUBLICATION_YEAR,
    			'xml' => 'publication_year', 'view_type_id' => 'text', 'data_type_id' => 'int', 'data_length' => 4, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'url', 'name' => _MI_XOONIPS_INSTALL_BOOK_URL,
    			'xml' => 'url', 'view_type_id' => 'url', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_BOOK_PDF_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Book';
    	}
    	$complementObj = array(
			array('complement_id' => 'ISBN', 'base_item_field_detail_id' => 'isbn:isbn', 'complement_detail_id' => 'title', 'item_field_detail_id' => 'title:title'),
			array('complement_id' => 'ISBN', 'base_item_field_detail_id' => 'isbn:isbn', 'complement_detail_id' => 'author', 'item_field_detail_id' => 'authors:name'),
			array('complement_id' => 'ISBN', 'base_item_field_detail_id' => 'isbn:isbn', 'complement_detail_id' => 'publisher', 'item_field_detail_id' => 'publisher:name'),
			array('complement_id' => 'ISBN', 'base_item_field_detail_id' => 'isbn:isbn', 'complement_detail_id' => 'publicationyear', 'item_field_detail_id' => 'publication_year:publication_year'),
			array('complement_id' => 'ISBN', 'base_item_field_detail_id' => 'isbn:isbn', 'complement_detail_id' => 'url', 'item_field_detail_id' => 'url:url'),
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'authors:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'authors:romaji'),
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'publisher:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'publisher:romaji'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Book';
    	}
    	$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Book';
    	}

    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			#array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'editor:name', 'value' => 'sprintf("editor:%s",1)'),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'publisher:name', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'publication_year:publication_year', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Book', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			#array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'editor:name', 'value' => 'sprintf("editor:%s",1)'),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'publisher:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'publication_year:publication_year', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Book', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Book';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
	}

    private function installConferenceItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Conference', 'preselect' => 1, 'released' => 1, 'weight' => 4, 'name' => 'Conference', 'description' => _MI_XOONIPS_INSTALL_CONFERENCE_TITLE,
    			'icon' => 'icon_conference.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_conference" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(ctitle,ctitle)|xoops_escape}> (<{$item->get(presentation_type,presentation_type)}>)<br />
        <{$item->get(authors,name)|xoops_escape}>
    </td>
    <td width="65">
    <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
    	$groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_CONFERENCE_TITLE, 'xml' => 'ctitle', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_PLACE, 'xml' => 'place', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_PRESENTATION_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_AUTHOR, 'xml' => 'authors', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_ABSTRACT, 'xml' => 'abstract', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_PRESENTATION_FILE, 'xml' => 'presentation_file', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_PRESENTATION_TYPE, 'xml' => 'presentation_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_CONFERENCE_CONFERENCE_PAPER, 'xml' => 'conference', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Conference';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'ctitle', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_CONFERENCE_TITLE,
    			'xml' => 'ctitle', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'place', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_PLACE,
    			'xml' => 'place', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_PRESENTATION_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_AUTHOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'abstract', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_ABSTRACT,
    			'xml' => 'abstract', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'presentation_file', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_PRESENTATION_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'presentation_type', 'name' => _MI_XOONIPS_INSTALL_COMMON_FILE_TYPE,
    			'xml' => 'file_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Conference file type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'conference', 'name' => _MI_XOONIPS_INSTALL_CONFERENCE_CONFERENCE_PAPER,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Conference';
    	}
    	$complementObj = array(
    		array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'authors:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'authors:romaji'),
    	);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Conference';
    	}
    	$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Conference';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'place:place', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'abstract:abstract', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'presentation_type:file_type', 'value' => 'sprintf("presentationType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Presentation', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'place:place', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'abstract:abstract', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'presentation_type:file_type', 'value' => 'sprintf("presentationType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Presentation', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Conference';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installDataItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Data', 'preselect' => 1, 'released' => 1, 'weight' => 5, 'name' => 'Data', 'description' => _MI_XOONIPS_INSTALL_DATA_TITLE,
    			'icon' => 'icon_data.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_data" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(data_type,data_type)}><br />
        <{$item->get(experimenters,experimenter)|xoops_escape}>
    </td>
    <td width="65">
    <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
    	$groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_DATA_DATA_TYPE, 'xml' => 'data_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_DATA_EXPERIMENTER, 'xml' => 'experimenters', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_DATA_DATA_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Data';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'data_type', 'name' => _MI_XOONIPS_INSTALL_DATA_DATA_TYPE,
    			'xml' => 'data_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Data type', 'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'experimenters', 'name' => _MI_XOONIPS_INSTALL_DATA_EXPERIMENTER,
    			'xml' => 'experimenter', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_DATA_DATA_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Data';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Data';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Data';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'experimenters:experimenter', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'data_type:data_type', 'value' => 'sprintf("dataType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Data or Dataset', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'experimenters:experimenter', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'data_type:data_type', 'value' => 'sprintf("dataType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Data or Dataset', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Data';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installFileItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'File', 'preselect' => 1, 'released' => 1, 'weight' => 6, 'name' => 'File', 'description' => _MI_XOONIPS_INSTALL_FILE_TITLE,
    			'icon' => 'icon_files.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_file" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        (<{foreach item=contributor from=$item->get(contributors,contributor) name=contributor}><{if !$smarty.foreach.contributor.first}> . <{/if}><{$contributor.name|xoops_escape}><{if !empty($contributor.name)}>(<{/if}><{$contributor.uname|xoops_escape}><{if !empty($contributor.name)}>)<{/if}><{/foreach}>)<br />
        (<{foreach item=file from=$item->get(files,file)}><{if $smarty.foreach.file.iteration > 1 }> . <{/if}><{$file.mime_type}><{/foreach}>)
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        	$groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_FILE_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'File';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_FILE_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
		);
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'File';
    	}
    	$complementObj = array();
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'File';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'junii2:format', 'item_field_detail_id' => 'files:file', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'oai_dc:format', 'item_field_detail_id' => 'files:file', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'File';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installMemoItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Memo', 'preselect' => 1, 'released' => 1, 'weight' => 7, 'name' => 'Memo', 'description' => _MI_XOONIPS_INSTALL_MEMO_TITLE,
    			'icon' => 'icon_memo.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_memo" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{if ($item->get(link,item_link)|count_characters != 0)}>Link to <a href="<{$item->get(link,item_link)}>"><{$item->get(link,item_link)}></a><{/if}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_MEMO_LINK, 'xml' => 'link', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_MEMO_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Memo';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'link', 'name' => _MI_XOONIPS_INSTALL_MEMO_FILE,
    			'xml' => 'item_link', 'view_type_id' => 'url', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_FILE_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Memo';
    	}
    	$complementObj = array();
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Memo';
    	}
		$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Memo';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installModelItemtype($dirname, $trustDirname) {
     	$itemtypeObj = array('item_type_id' => 'Model', 'preselect' => 1, 'released' => 1, 'weight' => 8, 'name' => 'Model', 'description' => _MI_XOONIPS_INSTALL_MODEL_TITLE,
    			'icon' => 'icon_model.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_model" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(creators,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_MODEL_MODEL_TYPE, 'xml' => 'model_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_MODEL_CREATOR, 'xml' => 'creators', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_MODEL_MODEL_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Model';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'model_type', 'name' => _MI_XOONIPS_INSTALL_MODEL_MODEL_TYPE,
    			'xml' => 'model_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Model type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'creators', 'name' => _MI_XOONIPS_INSTALL_MODEL_CREATOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_MODEL_MODEL_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Model';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Model';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Model';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'creators:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'creation_date:creation_date', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'model_type:model_type', 'value' => 'sprintf("modelType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'creators:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'creation_date:creation_date', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'model_type:model_type', 'value' => 'sprintf("modelType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Model';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installPaperItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Paper', 'preselect' => 1, 'released' => 1, 'weight' => 9, 'name' => 'Paper', 'description' => _MI_XOONIPS_INSTALL_PAPER_TITLE,
    			'icon' => 'icon_paper.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_paper" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(authors,name)|xoops_escape}><br />
        <{$item->get(jtitle,name)|xoops_escape}><{$item->get(publication_year,publication_year)|xoops_escape}><{if ($item->get(volume,volume)|count_characters != 0)}>;<{$item->get(volume,volume)|xoops_escape}> <{/if}><{if ($item->get(number,number)|count_characters != 0)}>(<{$item->get(number,number)|xoops_escape}>) <{/if}><{if (count($item->get(page,page)) != 0)}>:<{$item->get(page,page)|xoops_escape}> <{/if}><{if ($item->get(pubmedid,pubmedid)|count_characters != 0)}> [PMID: <{$item->get(pubmedid,pubmedid)|xoops_escape}>]<{/if}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PAPER_PUBMEDID, 'xml' => 'pubmedid', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PAPER_AUTHOR, 'xml' => 'authors', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_PAPER_JOURNAL, 'xml' => 'jtitle', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_BOOK_PUBLICATION_YEAR, 'xml' => 'publication_year', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PAPER_ABSTRACT, 'xml' => 'abstract', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PAPER_PDF_REPRINT, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Paper';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'pubmedid', 'name' => _MI_XOONIPS_INSTALL_PAPER_PUBMEDID,
    			'xml' => 'pubmedid', 'view_type_id' => 'pubmed id', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_PAPER_AUTHOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_KANA,
    			'xml' => 'kana', 'view_type_id' => 'kana', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'authors', 'name' => _MI_XOONIPS_INSTALL_COMMON_ROMAJI,
    			'xml' => 'romaji', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_NAME,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_TRANSLATION,
    			'xml' => 'jtitle_translation', 'view_type_id' => 'text', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_VOLUME,
    			'xml' => 'jtitle_volume', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_ISSUE,
    			'xml' => 'jtitle_issue', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_YEAR,
    			'xml' => 'jtitle_year', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_MONTH,
    			'xml' => 'jtitle_month', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_FROM_PAGE,
    			'xml' => 'jtitle_spage', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'jtitle', 'name' => _MI_XOONIPS_INSTALL_ARTICEL_JTITLE_TO_PAGE,
    			'xml' => 'jtitle_epage', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'publication_year', 'name' => _MI_XOONIPS_INSTALL_BOOK_PUBLICATION_YEAR,
    			'xml' => 'publication_year', 'view_type_id' => 'text', 'data_type_id' => 'int', 'data_length' => 4, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'abstract', 'name' => _MI_XOONIPS_INSTALL_PAPER_ABSTRACT,
    			'xml' => 'abstract', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_PAPER_PDF_REPRINT,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Paper';
    	}
    	$complementObj = array(
			array('complement_id' => 'Pubmed ID', 'base_item_field_detail_id' => 'pubmedid:pubmedid', 'complement_detail_id' => 'title', 'item_field_detail_id' => 'title:title'),
			array('complement_id' => 'Pubmed ID', 'base_item_field_detail_id' => 'pubmedid:pubmedid', 'complement_detail_id' => 'keyword', 'item_field_detail_id' => 'keywords:keyword'),
			array('complement_id' => 'Pubmed ID', 'base_item_field_detail_id' => 'pubmedid:pubmedid', 'complement_detail_id' => 'author', 'item_field_detail_id' => 'authors:name'),
			array('complement_id' => 'Pubmed ID', 'base_item_field_detail_id' => 'pubmedid:pubmedid', 'complement_detail_id' => 'abstract', 'item_field_detail_id' => 'abstract:abstract'),
			array('complement_id' => 'KANA', 'base_item_field_detail_id' => 'authors:kana', 'complement_detail_id' => 'romaji', 'item_field_detail_id' => 'authors:romaji'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Paper';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Paper';
    	}
		$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:jtitle', 'item_field_detail_id' => 'jtitle:name', 'value' => NULL),
			array('schema_id' => 'junii2:volume', 'item_field_detail_id' => 'jtitle:jtitle_volume', 'value' => NULL),
			array('schema_id' => 'junii2:issue', 'item_field_detail_id' => 'jtitle:jtitle_issue', 'value' => NULL),
			array('schema_id' => 'junii2:spage', 'item_field_detail_id' => 'jtitle:jtitle_spage', 'value' => NULL),
			array('schema_id' => 'junii2:epage', 'item_field_detail_id' => 'jtitle:jtitle_epage', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'publication_year:publication_year', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Journal Article', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'junii2:pmid', 'item_field_detail_id' => 'pubmedid:pubmedid', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'authors:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' =>
				'jtitle:name,jtitle:jtitle_volume,jtitle:jtitle_issue,jtitle:jtitle_year,jtitle:jtitle_month,jtitle:jtitle_spage,jtitle:jtitle_epage',
				'value' => 'sprintf("%s Vol.%s, No.%s (%s.%s) p.%s-%s",1,2,3,4,5,6,7)'),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'publication_year:publication_year', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Journal Article', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
			array('schema_id' => 'oai_dc:relation', 'item_field_detail_id' => 'pubmedid:pubmedid', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Paper';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installPresentationItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Presentation', 'preselect' => 1, 'released' => 1, 'weight' => 10, 'name' => 'Presentation', 'description' => _MI_XOONIPS_INSTALL_PRESENTATION_TITLE,
    			'icon' => 'icon_presentation.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_presentation" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(presentation_type,presentation_type)}><br />
        <{$item->get(creators,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PRESENTATION_PRESENTATION_TYPE, 'xml' => 'presentation_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PRESENTATION_CREATOR, 'xml' => 'creators', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_PRESENTATION_PRESENTATION_FILE, 'xml' => 'presentation_file', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Presentation';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'presentation_type', 'name' => _MI_XOONIPS_INSTALL_PRESENTATION_PRESENTATION_TYPE,
    			'xml' => 'presentation_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Presentation file type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'creators', 'name' => _MI_XOONIPS_INSTALL_PRESENTATION_CREATOR,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'presentation_file', 'name' => _MI_XOONIPS_INSTALL_PRESENTATION_PRESENTATION_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Presentation';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Presentation';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Presentation';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'creators:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'presentation_type:presentation_type', 'value' => 'sprintf("presentationType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Presentation', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'creators:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'presentation_type:presentation_type', 'value' => 'sprintf("presentationType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Presentation', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Presentation';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
     }

    private function installSimulatorItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Simulator', 'preselect' => 1, 'released' => 1, 'weight' => 11, 'name' => 'Simulator', 'description' => _MI_XOONIPS_INSTALL_SIMULATOR_TITLE,
    			'icon' => 'icon_simulator.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_simulator" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(simulator_type,simulator_type)}><br />
        <{$item->get(developers,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_FILE_TYPE, 'xml' => 'simulator_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_SIMULATOR_DEVELOPER, 'xml' => 'developers', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_SIMULATOR_SIMULATOR_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Simulator';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'simulator_type', 'name' => _MI_XOONIPS_INSTALL_COMMON_FILE_TYPE,
    			'xml' => 'simulator_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Simulator file type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'developers', 'name' => _MI_XOONIPS_INSTALL_SIMULATOR_DEVELOPER,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_SIMULATOR_SIMULATOR_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Simulator';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Simulator';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Simulator';
    	}
		$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'simulator_type:simulator_type', 'value' => 'sprintf("simulatorType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'simulator_type:simulator_type', 'value' => 'sprintf("simulatorType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Simulator';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installStimulusItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Stimulus', 'preselect' => 1, 'released' => 1, 'weight' => 12, 'name' => 'Stimulus', 'description' => _MI_XOONIPS_INSTALL_STIMULUS_TITLE,
    			'icon' => 'icon_stimulus.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_stimulus" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(stimulus_type,stimulus_type)}><br />
        <{$item->get(developers,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_STIMULUS_STIMULUS_TYPE, 'xml' => 'stimulus_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_STIMULUS_DEVELOPER, 'xml' => 'developers', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_STIMULUS_STIMULUS_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Stimulus';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'stimulus_type', 'name' => _MI_XOONIPS_INSTALL_STIMULUS_STIMULUS_TYPE,
    			'xml' => 'stimulus_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Stimulus type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'developers', 'name' => _MI_XOONIPS_INSTALL_STIMULUS_DEVELOPER,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_STIMULUS_STIMULUS_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Stimulus';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Stimulus';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Stimulus';
    	}
		$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'stimulus_type:stimulus_type', 'value' => 'sprintf("stimulusType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'stimulus_type:stimulus_type', 'value' => 'sprintf("stimulusType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Stimulus';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installToolItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Tool', 'preselect' => 1, 'released' => 1, 'weight' => 13, 'name' => 'Tool', 'description' => _MI_XOONIPS_INSTALL_TOOL_TITLE,
    			'icon' => 'icon_tool.gif', 'mime_type' => 'image/gif', 'template' =>
'<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_tool" />
    </td>
    <td>
        <a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{$item->get(tool_type,tool_type)}><br />
        <{$item->get(developers,name)|xoops_escape}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DATE, 'xml' => 'date', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_TOOL_TOOL_TYPE, 'xml' => 'tool_type', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_TOOL_DEVELOPER, 'xml' => 'developers', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW, 'xml' => 'preview', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_TOOL_TOOL_FILE, 'xml' => 'files', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION, 'xml' => 'download_limitation', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION, 'xml' => 'download_notification', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_README, 'xml' => 'readme', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS, 'xml' => 'rights', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
     		$obj['item_type_id'] = 'Tool';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'date', 'name' => _MI_XOONIPS_INSTALL_COMMON_DATE,
    			'xml' => 'date', 'view_type_id' => 'date(yyyy mm dd)', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 1),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'tool_type', 'name' => _MI_XOONIPS_INSTALL_TOOL_TOOL_TYPE,
    			'xml' => 'tool_type', 'view_type_id' => 'file type', 'data_type_id' => 'varchar', 'data_length' => 30, 'data_decimal_places' => -1,
    			'default_value' => 'Other', 'list' => 'Tool file type', 'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'developers', 'name' => _MI_XOONIPS_INSTALL_TOOL_DEVELOPER,
    			'xml' => 'name', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_PREVIEW,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'caption', 'group_id' => 'preview', 'name' => _MI_XOONIPS_INSTALL_COMMON_CAPTION,
    			'xml' => 'caption', 'view_type_id' => 'text', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'files', 'name' => _MI_XOONIPS_INSTALL_TOOL_TOOL_FILE,
    			'xml' => 'file', 'view_type_id' => 'file upload', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_limitation', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_LIMITATION,
    			'xml' => 'attachment_dl_limit', 'view_type_id' => 'download limit', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'download_notification', 'name' => _MI_XOONIPS_INSTALL_COMMON_DOWNLOAD_NOTIFICATION,
    			'xml' => 'attachment_dl_notify', 'view_type_id' => 'download notify', 'data_type_id' => 'int', 'data_length' => 1, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'readme', 'name' => _MI_XOONIPS_INSTALL_COMMON_README,
    			'xml' => 'readme', 'view_type_id' => 'readme', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'rights', 'name' => _MI_XOONIPS_INSTALL_COMMON_RIGHTS,
    			'xml' => 'rights', 'view_type_id' => 'rights', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Tool';
    	}
    	$complementObj = array(
			array('complement_id' => 'Preview', 'base_item_field_detail_id' => 'preview:image', 'complement_detail_id' => 'caption', 'item_field_detail_id' => 'preview:caption'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Tool';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Tool';
    	}
    	$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'tool_type:tool_type', 'value' => 'sprintf("toolType:%s",1)'),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:creator', 'item_field_detail_id' => 'developers:name', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'readme:readme', 'value' => 'sprintf("readme:%s",1)'),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'rights:rights', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:date', 'item_field_detail_id' => 'date:date', 'value' => 'date("Y/m/d",sprintf("%s",1))'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'tool_type:tool_type', 'value' => 'sprintf("toolType:%s",1)'),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Software', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Tool';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

    private function installUrlItemtype($dirname, $trustDirname) {
    	$itemtypeObj = array('item_type_id' => 'Url', 'preselect' => 1, 'released' => 1, 'weight' => 14, 'name' => 'Url', 'description' => _MI_XOONIPS_INSTALL_URL_TITLE,
    			'icon' => 'icon_url.gif', 'mime_type' => 'image/gif', 'template' =>
'<div class="content"><script type="text/javascript">
function countup_link_<{$item->getItemId()}>(itemtype_id, detail_id) {
    var url = "<{$smarty.const.XOONIPS_URL}>/linkcount.php?itemtype_id="+itemtype_id+"&base_detail_id="+detail_id+"&item_id=<{$item->getItemId()}>";
    $.get(url);
}
</script>
<table><tr>
    <td width="65" style="vertical-align:middle; text-align:center;">
        <img src="<{$item->getIconUrl()}>" alt="icon_url" />
    </td>
    <td><a href="detail.php?item_id=<{$item->getItemId()}>"><{$item->get(title,title)|xoops_escape}></a><br />
        <{if (count($item->get(url,url)) != 0)}>Link to <a href="<{$item->get(url,url)}>" target="_blank" onclick="return countup_link_<{$item->getItemId()}>(<{$item->getItemTypeId()}>,284);"><{$item->get(url,url)}></a><{/if}>
    </td>
    <td width="65">
        <{if $item->isPending()}>(pending)<{/if}>
    </td>
</tr></table></div>');
        $groupObj = array(
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_ID, 'xml' => 'doi', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LANGS, 'xml' => 'langs', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'xml' => 'title', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD, 'xml' => 'keywords', 'occurrence' => 1),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION, 'xml' => 'description', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_URL_URL, 'xml' => 'url', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_URL_BANNER_FILE, 'xml' => 'banner_file', 'occurrence' => 0),
			array('name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'xml' => 'last_update_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'xml' => 'creation_date', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR, 'xml' => 'contributors', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG, 'xml' => 'changelogs', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_INDEX, 'xml' => 'indexes', 'occurrence' => 0),
    		array('name' => _MI_XOONIPS_INSTALL_COMMON_RELATION, 'xml' => 'links', 'occurrence' => 0),
    	);
    	foreach ($groupObj as &$obj) {
    		$obj['item_type_id'] = 'Url';
    	}
    	$detailObj = array(
    		array('table_name' => 'item', 'column_name' => 'doi', 'group_id' => 'doi', 'name' => _MI_XOONIPS_INSTALL_COMMON_ID,
    			'xml' => 'doi', 'view_type_id' => 'id', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'langs', 'name' => _MI_XOONIPS_INSTALL_COMMON_LANGS,
    			'xml' => 'lang', 'view_type_id' => 'select', 'data_type_id' => 'char', 'data_length' => 3, 'data_decimal_places' => -1,
    			'default_value' => 'eng', 'list' => 'Language', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_title', 'column_name' => 'title', 'group_id' => 'title', 'name' => _MI_XOONIPS_INSTALL_COMMON_TITLE,
    			'xml' => 'title', 'view_type_id' => 'title', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_keyword', 'column_name' => 'keyword', 'group_id' => 'keywords', 'name' => _MI_XOONIPS_INSTALL_COMMON_KEYWORD,
    			'xml' => 'keyword', 'view_type_id' => 'keyword', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'description', 'name' => _MI_XOONIPS_INSTALL_COMMON_DESCRIPTION,
    			'xml' => 'description', 'view_type_id' => 'textarea', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'url', 'name' => _MI_XOONIPS_INSTALL_URL_URL,
    			'xml' => 'url', 'view_type_id' => 'url', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item_extend', 'column_name' => 'value', 'group_id' => 'url', 'name' => _MI_XOONIPS_INSTALL_URL_HITS,
    			'xml' => 'hits', 'view_type_id' => 'hidden', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
			array('table_name' => 'item_file', 'column_name' => 'file_id', 'group_id' => 'banner_file', 'name' => _MI_XOONIPS_INSTALL_URL_BANNER_FILE,
    			'xml' => 'image', 'view_type_id' => 'preview', 'data_type_id' => 'varchar', 'data_length' => 255, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 1, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'last_update_date', 'group_id' => 'last_update_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE,
    			'xml' => 'last_update_date', 'view_type_id' => 'last update', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item', 'column_name' => 'creation_date', 'group_id' => 'creation_date', 'name' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE,
    			'xml' => 'creation_date', 'view_type_id' => 'create date', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'default_value' => '0', 'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_users_link', 'column_name' => 'uid', 'group_id' => 'contributors', 'name' => _MI_XOONIPS_INSTALL_COMMON_CONTRIBUTOR,
    			'xml' => 'contributor', 'view_type_id' => 'create user', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_changelog', 'column_name' => 'log', 'group_id' => 'changelogs', 'name' => _MI_XOONIPS_INSTALL_COMMON_CHANGE_LOG,
    			'xml' => 'changelog', 'view_type_id' => 'change log', 'data_type_id' => 'text', 'data_length' => -1, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'index_item_link', 'column_name' => 'index_id', 'group_id' => 'indexes', 'name' => _MI_XOONIPS_INSTALL_COMMON_INDEX,
    			'xml' => 'index', 'view_type_id' => 'index', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 1, 'detail_target' => 0, 'scope_search' => 0),
    		array('table_name' => 'item_related_to', 'column_name' => 'child_item_id', 'group_id' => 'links', 'name' => _MI_XOONIPS_INSTALL_COMMON_RELATION,
    			'xml' => 'link', 'view_type_id' => 'relation item', 'data_type_id' => 'int', 'data_length' => 10, 'data_decimal_places' => -1,
    			'essential' => 0, 'detail_target' => 0, 'scope_search' => 0),
        );
    	foreach ($detailObj as &$obj) {
    		$obj['item_type_id'] = 'Url';
    	}
    	$complementObj = array(
			array('complement_id' => 'URL', 'base_item_field_detail_id' => 'url:url', 'complement_detail_id' => 'hits', 'item_field_detail_id' => 'url:hits'),
		);
    	foreach ($complementObj as &$obj) {
    		$obj['item_type_id'] = 'Url';
    	}
		$sortObj = array(
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_TITLE, 'item_field_detail_id' => 'title:title'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_ID, 'item_field_detail_id' => 'doi:doi'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_LAST_UPDATE, 'item_field_detail_id' => 'last_update_date:last_update_date'),
			array('sort_id' => _MI_XOONIPS_INSTALL_COMMON_CREATION_DATE, 'item_field_detail_id' => 'creation_date:creation_date'),
		);
    	foreach ($sortObj as &$obj) {
    		$obj['item_type_id'] = 'Url';
    	}
		$searchObj = array();

		$oaipmhObj = array(
			array('schema_id' => 'junii2:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'junii2:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'junii2:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'junii2:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'junii2:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'junii2:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'junii2:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'junii2:URI', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'junii2:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),

			array('schema_id' => 'oai_dc:title', 'item_field_detail_id' => 'title:title', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'keywords:keyword', 'value' => NULL),
			array('schema_id' => 'oai_dc:subject', 'item_field_detail_id' => 'indexes:index', 'value' => NULL),
			array('schema_id' => 'oai_dc:description', 'item_field_detail_id' => 'description:description', 'value' => NULL),
			array('schema_id' => 'oai_dc:publisher', 'item_field_detail_id' => 'meta_author', 'value' => NULL),
			array('schema_id' => 'oai_dc:type', 'item_field_detail_id' => 'itemtype', 'value' => 'sprintf("itemType:%s",1)'),
			array('schema_id' => 'oai_dc:type:NIItype', 'item_field_detail_id' => 'Others', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'ID', 'value' => NULL),
			array('schema_id' => 'oai_dc:identifier', 'item_field_detail_id' => 'http://', 'value' => NULL),
			array('schema_id' => 'oai_dc:language', 'item_field_detail_id' => 'langs:lang', 'value' => NULL),
		);
    	foreach ($oaipmhObj as &$obj) {
    		$obj['item_type_id'] = 'Url';
    	}
    	return self::installItemtype($dirname, $trustDirname, $itemtypeObj, $groupObj, $detailObj, $complementObj, $sortObj, $searchObj, $oaipmhObj);
    }

	private function installIndex($dirname, $trustDirname) {
		$indexlist = array(
			array('parent_index_id' => 0, 'open_level' => XOONIPS_OL_PUBLIC, 'weight' => 1, 'title' => 'Root'),
			array('parent_index_id' => 1, 'open_level' => XOONIPS_OL_PUBLIC, 'weight' => 1, 'title' => 'Public'),
		);
		$indexBean = Xoonips_BeanFactory::getBean('IndexBean', $dirname, $trustDirname);
		foreach ($indexlist as $index) {
			$index['uid'] = NULL;
			$index['groupid'] = NULL;
			$index['description'] = NULL;
			if (!$indexBean->insertIndex($index)) {
				return false;
			}
		}
		return true;
	}

	private function installXleprogress($dirname, $trustDirname) {
		$typelist = array(
			array('dirname' => $dirname, 'dataname' => _MI_XOONIPS_INSTALL_XLEPROGRESS_CERTIFY_PUBLIC_ITEMS, 'module' => $dirname . '.WorkflowPublicItems'),
			array('dirname' => $dirname, 'dataname' => _MI_XOONIPS_INSTALL_XLEPROGRESS_WITHDRAW_PUBLIC_ITEMS, 'module' => $dirname . '.WorkflowPublicItemsWithdrawal'),
			array('dirname' => $dirname, 'dataname' => _MI_XOONIPS_INSTALL_XLEPROGRESS_CERTIFY_GROUP_ITEMS, 'module' => $dirname . '.WorkflowGroupItems'),
			array('dirname' => $dirname, 'dataname' => _MI_XOONIPS_INSTALL_XLEPROGRESS_WITHDRAW_GROUP_ITEMS, 'module' => $dirname . '.WorkflowGroupItemsWithdrawal')
		);
		$typeHandler = Legacy_Utils::getModuleHandler('type', 'xleprogress');
		if (is_object($typeHandler)) {
			$objs = $typeHandler->getObjects(new Criteria('dirname', $dirname));
			if (count($objs) == 0) {
				foreach ($typelist as $type) {
					$obj = $typeHandler->create();
					foreach ($type as $k => $v) {
			        	$obj->set($k, $v);
					}
					if (!$typeHandler->insert($obj)) {
						return false;
					}
				}
			}
		}
		return true;
	}

    /**
     * omitBlocks
     *
     * @param   array $block
     *
     * @return  bool
    **/
	private function omitBlocks($block) {
		$omit = array('b_xoonips_login_show','b_xoonips_user_show');
		foreach ($omit as $val) {
			if ($block['show_func'] == $val) {
				return true;
			}
		}
		return false;
	}

    /**
     * chkXleprogress
     *
     * @return  bool
    **/
	public function chkXleprogress() {
		$xle = 'xleprogress';
		$chk_dir = XOOPS_TRUST_PATH.'/modules/' . $xle;
		if (is_dir($chk_dir)) {
			$module_handler =& xoops_gethandler('module');
			$module =& $module_handler->getByDirname($xle);
			if (strtolower($module->getVar('dirname') == $xle)) {
				return true;
			}
		}
		return false;
	}

    /**
     * installItemSearch
     *
     * @param   $dirname, $trustDirname, $searchObj
     * @return  bool
    **/
	public function installItemSearch($dirname, $trustDirname, $searchObj) {
		$searchBean = Xoonips_BeanFactory::getBean('ItemTypeSearchConditionBean', $dirname, $trustDirname);
		$searchlist = $searchBean->select();
		$conditionid = array();
    	foreach ($searchlist as $search) {
    		$conditionid[$search['title']] = $search['condition_id'];
    	}
		$searchDetailBean = Xoonips_BeanFactory::getBean('ItemTypeSearchConditionDetailBean', $dirname, $trustDirname);
    	foreach ($searchObj as &$obj) {
			if (isset($conditionid[$obj['condition_id']])) {
	    		$obj['condition_id'] = $conditionid[$obj['condition_id']];
			} else {
				$id = '';
	    		if (!$searchBean->insert(array('title' => $obj['condition_id']), $id)) {
    				return false;
    			}
				$conditionid[$obj['condition_id']] = $id;
				$obj['condition_id'] = $id;
    		}
    		$obj['item_type_id'] = 0;
    		if (!$searchDetailBean->insert($obj)) {
				return false;
			}
		}
		return true;
	}

}
?>
