﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi
{
    public class EnuCoordinate
    {
        #region fields
        
        #endregion

        #region constructors

        public EnuCoordinate(double e, double n, double u)
            : this(e, n, u, null)
        {
        }

        public EnuCoordinate(double e, double n, double u, 
                                                        EcefCoordinate origin)
        {
            E = e;
            N = n;
            U = u;
            Origin = origin;
        }

        public EnuCoordinate(EcefCoordinate target, EcefCoordinate origin)
            : this(target, origin, origin.ToGeodeticCoordinate())
        {
        }

        public EnuCoordinate(GeodeticCoordinate target,
                                                    GeodeticCoordinate origin)
            : this(target.ToEcefCoordinate(), 
                                            origin.ToEcefCoordinate(), origin)
        {
        }

        private EnuCoordinate(EcefCoordinate target,
                        EcefCoordinate origin, GeodeticCoordinate llhOrigin)
        {
            Origin = origin;
            double dx = target.X - origin.X;
            double dy = target.Y - origin.Y;
            double dz = target.Z - origin.Z;
            double longitude = llhOrigin.Longitude.Radian;
            double latitude = llhOrigin.Latitude.Radian;
            double sinLongitude = Math.Sin(longitude);
            double cosLongitude = Math.Cos(longitude);
            double sinLatitude = Math.Sin(latitude);
            double cosLatitude = Math.Cos(latitude);
            
            E = -sinLongitude * dx + 
                    cosLongitude * dy;
            N = -sinLatitude * cosLongitude * dx -
                    sinLatitude * sinLongitude * dy +
                        cosLatitude * dz;
            U = cosLatitude * cosLongitude * dx +
                    cosLatitude * sinLongitude * dy +
                        sinLatitude * dz;
        }

        #endregion

        #region properties

        public double E
        {
            get;
            set;
        }

        public double N
        {
            get;
            set;
        }

        public double U
        {
            get;
            set;
        }

        public EcefCoordinate Origin
        {
            get;
            private set;
        }

        #endregion

        #region public methods
        public EcefCoordinate ToEcefCoordinate()
        {
            GeodeticCoordinate llhOrigin = Origin.ToGeodeticCoordinate();
            
            double longitude = llhOrigin.Longitude.Radian;
            double latitude = llhOrigin.Latitude.Radian;
            double sinLongitude = Math.Sin(longitude);
            double cosLongitude = Math.Cos(longitude);
            double sinLatitude = Math.Sin(latitude);
            double cosLatitude = Math.Cos(latitude);

            double dx = -sinLongitude * E -
                            sinLatitude*cosLongitude * N + 
                                cosLatitude*cosLongitude * U;
            double dy = cosLongitude * E -
                            sinLatitude*sinLongitude * N + 
                                cosLatitude*sinLongitude * U;
            double dz = cosLatitude * N + sinLatitude * U;
            
            return new EcefCoordinate(
                                dx + Origin.X, dy + Origin.Y, + dz + Origin.Z);
        }

        #endregion
    }
}
