﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public abstract class Packet : UnknownPacket
    {
        #region type definitions

        protected delegate Packet PacketBuilder(byte[] sentence, int length);

        public enum MessageClass : byte
        { 
            /// <summary>
            ///     Navigation
            /// </summary>
            Nav = 0x01,
            /// <summary>
            ///     Reciever Manager
            /// </summary>
            Rxm = 0x02,
            Inf = 0x04,
            Ack = 0x05,
            Cfg = 0x06,
            Mon = 0x0A,
            Aid = 0x0B,
            Tim = 0x0D,
        }

        public enum MessageID : ushort
        {
            Unknown = 0x0000,
            NavPosEcef = 0x0101,
            NavPosLlh = 0x0102,
            NavStatus = 0x0103,
            NavDop = 0x0104,
            NavSol = 0x0106,
            NavVelEcef = 0x0111,
            NavVelNed = 0x0112,
            NavTimeGps = 0x0120,
            NavTimeUtc = 0x0121,
            NavClock = 0x0122,
            NavSvInfo = 0x0130,
            NavSbas = 0x0132,
            RxmRaw = 0x0210,
            RxmSfrB = 0x0211,
            RxmSvSI = 0x0220,
            RxmAlm = 0x0230,
            RxmEph = 0x0231,
            RxmPosReq = 0x0240,
            RxmPmReq = 0x0241,
            InfError = 0x0400,
            InfWarning = 0x0401,
            InfNotice = 0x0402,
            InfTest = 0x0403,
            InfDebug = 0x0404,
            CfgPrt = 0x0600,
            CfgMsg = 0x0601,
            CfgInf = 0x0602,
            CfgRst = 0x0604,
            CfgDat = 0x0606,
            CfgTp = 0x0607,
            CfgRate = 0x0608,
            CfgCfg = 0x0609,
            CfgFxn = 0x060E,
            CfgRxm = 0x0611,
            CfgAnt = 0x0613,
            CfgSbas = 0x0616,
            CfgNmea = 0x0617,
            CfgUsb = 0x061B,
            CfgTMode = 0x061D,
            CfgNavX5 = 0x0623,
            CfgNav = 0x0624,
            CfgPm = 0x0632,
            CfgRInv = 0x0634,
            MonIO = 0x0A02,
            MonVer = 0x0A04,
            MonMsgPP = 0x0A06,
            MonRxBuf = 0x0A07,
            MonTxBuf = 0x0A08,
            MonHw = 0x0A09,
            MonHw2 = 0x0A0B,
            MonRxR = 0x0A21,
            AidReq = 0x0B00,
            AidIni = 0x0B01,
            AidHui = 0x0B02,
            AidData = 0x0B10,
            AidAlm = 0x0B30,
            AidEph = 0x0B31,
            AidAlpSrv = 0x0B32,
            AidAlp = 0x0B50,
        }

        public enum GpsFixType : byte
        { 
            NoFix = 0,
            DeadReckoningOnly = 1,
            TwoDimensional = 2,
            ThreeDimensional = 3,
            GpsAndDeadReckoning = 4,
            TimeOnly = 5,
        }

        public enum FixStatusFlag : byte
        { 
        
        }
        #endregion

        #region fields
        public static readonly byte[] SyncCharacters = 
                                                    new byte[] { 0xB5, 0x62 };
        public const byte SyncCharacter1 = 0xB5;
        public const byte SyncCharacter2 = 0x62;
        #endregion

        #region constructors

        public Packet(byte[] sentence)
            : this(sentence, sentence.Length)
        { 
        }

        public Packet(byte[] sentence, int length)
            : base(sentence, length)
        {
        }

        protected Packet(MessageID id)
            : this(
                new byte[]
                {
                    Packet.SyncCharacter1, Packet.SyncCharacter2,
                    (byte)((int)id >> 8), (byte)id, // ID
                    0x00, 0x00, // length
                    0x00, 0x00  // checksum
                })
        {
            CheckSum = GenerateCheckSum();
        }
        #endregion

        #region properties

        public MessageClass Class
        {
            get
            {
                return (MessageClass)(Raw[2]);
            }
        }

        public MessageID ID
        {
            get
            { 
                return (MessageID)(Raw[2] << 8 | Raw[3]);
            }
        }

        public int CheckSum
        {
            get
            { 
                if (Raw == null)
                {
                    return -1;
                }
                return BitConverter.ToUInt16(Raw, Raw.Length - 2);
            }
            protected set
            {
                if (Raw != null)
                {
                    Raw[Raw.Length - 2] = (byte)(value);
                    Raw[Raw.Length - 1] = (byte)(value >> 8);
                }
            }
        }
        
        #endregion

        #region public methods

        public int GenerateCheckSum()
        {
            byte a = 0;
            byte b = 0;
            for (int i = 2; i < Raw.Length - 2; ++i)
            {
                a += Raw[i];
                b += a;
            }
            return b << 8 | a;
        }
        
        #endregion

        #region protected methods

        protected static bool TryParse(
                        byte[] sentence, out UnknownPacket packet,
                        MessageID id, PacketBuilder builder)
        {
            return TryParse(sentence, out packet, id, -1, builder);
        }

        protected static bool TryParse(
                        byte[] sentence, out UnknownPacket packet,
                        MessageID id, int fixedLength, PacketBuilder builder)
        {
            packet = null;
            if (sentence.Length < 8)
            {
                return false;
            }
            if (sentence[0] != SyncCharacter1 || sentence[1] != SyncCharacter2)
            {
                return false;
            }
            MessageID mid = (MessageID)(sentence[2] << 8 | sentence[3]);
            if (id != mid && id != MessageID.Unknown)
            {
                return false;
            }
            int length = sentence[5] << 8 | sentence[4];
            if (fixedLength >= 0 && fixedLength != length)
            {
                return false;
            }
            if (sentence.Length < length + 8)
            {
                return false;
            }
            try
            {
                packet = builder(sentence, length + 8);
            }
            catch
            {
                packet = null;
                return false;
            }
            return true;
        }
        
        #endregion


    }
}
