﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using Yubeshi;
using C = YubeshiTest.SampleCoordinates;

namespace YubeshiTest
{
    class EnuCoordinateTest
    {
        private static double metreError = 1e-9; // 1nm
        [Test]
        public void ZeroDistance()
        {
            EnuCoordinate enu = new EnuCoordinate(C.SkyTreeTop, C.SkyTreeTop);
            Assert.AreEqual(0.0, enu.E);
            Assert.AreEqual(0.0, enu.N);
            Assert.AreEqual(0.0, enu.U);
        }

        [Test]
        public void UpOnly()
        {
            EnuCoordinate enu = new EnuCoordinate(
                                                C.SkyTreeTop, C.SkyTreeBottom);

            Assert.AreEqual(0.0, enu.E, metreError);
            Assert.AreEqual(0.0, enu.N, metreError);
            Assert.AreEqual(634.0, enu.U, metreError);
        }

        [Test]
        public void TwoPoints()
        {
            GeodeticCoordinate c1 = C.SkyTreeTop;
            GeodeticCoordinate c2 = C.TokyoTowerTop;
            GeodeticCoordinate c0 = new GeodeticCoordinate(
                                        0.5 * (c1.Latitude + c2.Latitude),
                                        0.5 * (c1.Longitude + c2.Longitude),
                                        0.0);
            EnuCoordinate enu1 = new EnuCoordinate(c1, c0);
            EnuCoordinate enu2 = new EnuCoordinate(c2, c0);
            double de = enu1.E - enu2.E;
            double dn = enu1.N - enu2.N;
            double du = enu1.U - enu2.U;

            // approximation
            double distance = Math.Sqrt(de * de + dn * dn);
            double direction = Math.Atan2(de, dn) * 180.0 / Math.PI;
            
            Assert.AreEqual(8222.097, distance, 15.0);

            // c1->c2 45.926 deg
            // c2->c1 45.965 + 180 deg
            // 45.9455 = 0.5 * (45.926 + 45.965)
            Assert.AreEqual(45.946, direction, 1.0 / 60.0);
            Assert.AreEqual(283.0, du, 1e-3);
        }

        [Test]
        public void ToEcefCoordinate()
        {
            EnuCoordinate enu = new EnuCoordinate(
                                            C.SkyTreeTop, C.TokyoTowerTop);
            EcefCoordinate ecef1 = enu.ToEcefCoordinate();
            EcefCoordinate ecef2 = C.SkyTreeTop.ToEcefCoordinate();

            Assert.AreEqual(ecef2.X, ecef1.X, metreError);
            Assert.AreEqual(ecef2.Y, ecef1.Y, metreError);
            Assert.AreEqual(ecef2.Z, ecef1.Z, metreError);
        }
    }
}
