/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Nmea
{
    /// <summary>
    ///     Global Positioning System Fix Data
    /// </summary>
	public class GpGga : Packet
    {
        #region type definitions
        public enum FixQualityClass
        {
            Invalid = 0,
            Gps = 1,
            Dgps = 2,
            Pps = 3,
            Rtk = 4,
            FloatRtk = 5,
            Estimated = 6,
        }
        #endregion
        #region fields

        private static readonly byte[] header;
        private const int elementNum = 14;

        #endregion

        #region constructors

        static GpGga()
        {
            header = Encoding.ASCII.GetBytes("$GPGGA,");
        }

		public GpGga()
		{
		}

		public GpGga(byte[] sentence)
            : this(sentence, GetElements(sentence, elementNum))
		{
        }

        private GpGga(byte[] sentence, Elements elements)
            : base(sentence, elements.PacketLength)
        {
            string[] v = elements.Values;
            TimeOfFix = ParseTime(v[0]);
            Position = new GeodeticCoordinate(
                                    Degree.FromNmeaFormat(v[1], v[2]),
                                    Degree.FromNmeaFormat(v[3], v[4]));
            FixQuality = (FixQualityClass)Int32.Parse(v[5]);
            TrackedSatellites = Int32.Parse(v[6]);
            HorizontalDop = ParseDouble(v[7]);
            MslAltitude = new Height(GetLength(v[8], v[9]), 
                                                    Height.Base.MeanSeaLevel);
            GeoidSeparation = new Height(GetLength(v[10], v[11]),
                                                Height.Base.Wgs84Ellipsoid);
            double age = ParseDouble(v[12]);
            
            AgeOfDgpsCorrections = Double.IsNaN(age)?
                                TimeSpan.MinValue :TimeSpan.FromSeconds(age);
            DgpsStation = ParseInt(v[13], -1);
            CheckSum = elements.CheckSum;
        }

        #endregion

        #region properties

        public TimeSpan TimeOfFix
        {
            get;
            private set;
        }
        
        public GeodeticCoordinate Position
        {
            get;
            private set;
        }

        public FixQualityClass FixQuality
        {
            get;
            private set;
        }

        public int TrackedSatellites
        {
            get;
            private set;
        }

        public double HorizontalDop
        {
            get;
            private set;
        }

        public Height MslAltitude
        {
            get;
            private set;
        }

        public Height GeoidSeparation
        {
            get;
            private set;
        }

        public TimeSpan AgeOfDgpsCorrections
        {
            get;
            private set;
        }

        public int DgpsStation
        {
            get;
            private set;
        }
        #endregion


        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, header, elementNum, Build);
        }

        private static Packet Build(byte[] sentence, Elements elements)
        {
            return new GpGga(sentence, elements);
        }

	}
}