﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using Const = Yubeshi.Constants.Wgs84;

namespace Yubeshi
{
    public class EcefCoordinate
    {
        #region fields
        private const double a = Const.SemiMajorAxisA;
        private const double b = Const.SemiMajorAxisB;
        private const double e1sq = Const.FirstEccentricitySquared;
        private const double e2sq = Const.SecondEccentricitySquared;
        private const double pi = Const.Pi;
        #endregion

        #region constructors

        public EcefCoordinate(double x, double y, double z)
            : this(x, y, z, Double.NaN)
        {
        }

        public EcefCoordinate(double x, double y, double z, double accuracy)
        {
            X = x;
            Y = y;
            Z = z;
            Accuracy = accuracy;
        }
        #endregion

        #region properties
        public double X
        {
            get;
            set;
        }

        public double Y
        {
            get;
            set;
        }

        public double Z
        {
            get;
            set;
        }

        public double Accuracy
        {
            get;
            set;
        }
        #endregion

        #region public methods
        public GeodeticCoordinate ToGeodeticCoordinate()
        {
            // approximation
            double lambda = Math.Atan2(Y, X);
            double p = Math.Sqrt(X * X + Y * Y);
            double theta = Math.Atan2(Z * a, p * b);
            double sin = Math.Sin(theta);
            double cos = Math.Cos(theta);

            double phi = Math.Atan2(Z + e2sq * b * sin * sin * sin,
                                            p - e1sq * a * cos * cos * cos);
            double sinPhi = Math.Sin(phi);
            double cosPhi = Math.Cos(phi);
            double khi = Math.Sqrt(1.0 - e1sq * sinPhi * sinPhi);
            double n = a / khi;
            double ht;
            if (cosPhi < 0.5)
            {
                ht = Z / sinPhi - n * (1.0 - e1sq);
            }
            else
            { 
                ht = p / Math.Cos(phi) - n;
            }
            Height h = new Height(ht, Height.Base.Wgs84Ellipsoid);
            return new GeodeticCoordinate(
                                        phi * 180 / pi, lambda * 180 / pi, h);
        }
        #endregion
    }
}
