﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using Yubeshi;
using Yubeshi.Nmea;
using P = YubeshiTest.NmeaTest.SamplePackets;

namespace YubeshiTest.NmeaTest
{

    class PacketTest
    {

        [Test]
        public void GpDtmTest()
        {
            UnknownPacket packet;
            Assert.IsTrue(GpDtm.TryParse(P.GpDtm, out packet));
            GpDtm p = packet as GpDtm;
            Assert.AreEqual(GpDtm.Code.Wgs84, p.LocalDatumCode);
            Assert.AreEqual("", p.LocalDatumSubCode);
            Assert.AreEqual(0.0, (double)p.Offset.Latitude);
            Assert.AreEqual(0.0, (double)p.Offset.Longitude);
            Assert.AreEqual(0.0, (double)p.Offset.Altitude);
            Assert.AreEqual(GpDtm.Code.Wgs84, p.ReferenceDatumCode);
            Assert.AreEqual(0x6F, p.CheckSum);
        }

        [Test]
        public void GpGbsTest()
        {
            UnknownPacket packet;
            Assert.IsTrue(GpGbs.TryParse(P.GpGbs, out packet));
            GpGbs p = packet as GpGbs;
            Assert.AreEqual(new TimeSpan(23, 55, 03), p.Utc);
            Assert.AreEqual(1.6, (double)p.LatitudeError);
            Assert.AreEqual(1.4, (double)p.LongitudeError);
            Assert.AreEqual(3.2, (double)p.AltitudeError);
            Assert.AreEqual(-1, p.FailedSatellite);
            Assert.True(Double.IsNaN(p.ProbabilityOfMissedDetection));
            Assert.True(Double.IsNaN(p.Bias));
            Assert.True(Double.IsNaN(p.StandardDeviationOfBias));
            Assert.AreEqual(0x40, p.CheckSum);
        }

        [Test]
        public void GpGgaTest()
        {
            UnknownPacket packet;
            Assert.IsTrue(GpGga.TryParse(P.GpGga, out packet));
            GpGga p = packet as GpGga;
            Assert.AreEqual(new TimeSpan(0, 9, 27, 25, 0), p.TimeOfFix);
            Assert.AreEqual(2837.11399, p.Position.Latitude * 60.0);
            Assert.AreEqual(513.9159, p.Position.Longitude * 60.0);
            Assert.AreEqual(GpGga.FixQualityClass.Gps, p.FixQuality);
            Assert.AreEqual(8, p.TrackedSatellites);
            Assert.AreEqual(1.01, p.HorizontalDop);
            Assert.AreEqual(499.6, (double)p.MslAltitude);
            Assert.True(p.MslAltitude.IsAltitude);
            Assert.AreEqual(48.0, (double)p.GeoidSeparation);
            Assert.AreEqual(Height.Base.Wgs84Ellipsoid,
                                                p.GeoidSeparation.BaseLevel);
            Assert.Less(p.AgeOfDgpsCorrections.TotalSeconds, 0.0);
            Assert.AreEqual(0, p.DgpsStation);
            Assert.AreEqual(0x5B, p.CheckSum);
        }

        [Test]
        public void GpGllTest()
        {
            UnknownPacket packet;
            Assert.IsTrue(GpGll.TryParse(P.GpGll, out packet));
            GpGll p = packet as GpGll;
            Assert.AreEqual(2837.11364, p.Position.Latitude * 60.0);
            Assert.AreEqual(513.91565, p.Position.Longitude * 60.0);
            Assert.AreEqual(new TimeSpan(0, 9, 23, 21, 0), p.CurrentTime);
            Assert.AreEqual(GpGll.Status.Valid, p.DataStatus);
            Assert.AreEqual(GpGll.Mode.Autonomous, p.PositioningMode);
            Assert.AreEqual(0x60, p.CheckSum);
        }
    }
}
