/*
** c_console.cpp
** Implements the console itself
**
**---------------------------------------------------------------------------
** Copyright 1998-2006 Randy Heit
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
**
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote products
**    derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
** INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
** NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
** THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**---------------------------------------------------------------------------
**
*/

#include "templates.h"
#include "p_setup.h"
#include <stdarg.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
// [BC] New #includes.
#include <time.h>

#include "version.h"
#include "g_game.h"
#include "c_bind.h"
#include "c_console.h"
#include "c_cvars.h"
#include "c_dispatch.h"
#include "hu_stuff.h"
#include "i_system.h"
#include "i_video.h"
#include "i_input.h"
#include "m_swap.h"
#include "v_palette.h"
#include "v_video.h"
#include "v_text.h"
#include "w_wad.h"
#include "sbar.h"
#include "s_sound.h"
#include "s_sndseq.h"
#include "doomstat.h"
#include "d_gui.h"
#include "v_video.h"
#include "cmdlib.h"
#include "d_net.h"
#include "g_level.h"
#include "d_event.h"
#include "d_player.h"
#include "gstrings.h"
#include "c_consolebuffer.h"
#include "g_levellocals.h"

#include "gi.h"
// [BC] New #includes.
#include "chat.h"
#include "cl_demo.h"
#include "cl_main.h"
#include "cl_commands.h"
#include "deathmatch.h"
#include "network.h"
#include "win32/g15/g15.h"
#include "sv_rcon.h"

#define LEFTMARGIN 8
#define RIGHTMARGIN 8
#define BOTTOMARGIN 12


CUSTOM_CVAR(Int, con_buffersize, -1, CVAR_ARCHIVE | CVAR_GLOBALCONFIG)
{
	// ensure a minimum size
	if (self >= 0 && self < 128) self = 128;
}

FConsoleBuffer *conbuffer;

static void C_TabComplete (bool goForward);
static bool C_TabCompleteList ();
static bool TabbedLast;		// True if last key pressed was tab
static bool TabbedList;		// True if tab list was shown
CVAR(Bool, con_notablist, false, CVAR_ARCHIVE)


static FTextureID conback;
static DWORD conshade;
static bool conline;

extern int		gametic;
extern bool		automapactive;	// in AM_map.c
extern bool		advancedemo;

extern FBaseCVar *CVars;
extern FConsoleCommand *Commands[FConsoleCommand::HASH_SIZE];

unsigned	ConCols;
int			ConWidth;
bool		vidactive = false;
bool		cursoron = false;
int			ConBottom, ConScroll, RowAdjust;
int			CursorTicker;
constate_e	ConsoleState = c_up;

// [TP] Some functions print result directly to console.. when we need it to a string
// instead. To keep as much ZDoom code unchanged, here's a hack to capture the result
// into a string instead.
static bool g_IsCapturing = false;
static FString g_CaptureBuffer;


static int TopLine, InsertLine;

static void ClearConsole ();

struct GameAtExit
{
	GameAtExit(FString str) : Command(str) {}

	GameAtExit *Next;
	FString Command;
};

static GameAtExit *ExitCmdList;

#define SCROLLUP 1
#define SCROLLDN 2
#define SCROLLNO 0

EXTERN_CVAR (Bool, show_messages)

static unsigned int TickerAt, TickerMax;
static bool TickerPercent;
static const char *TickerLabel;

static bool TickerVisible;
static bool ConsoleDrawing;

// Buffer for AddToConsole()
static char *work = NULL;
static int worklen = 0;

CVAR(Float, con_notifytime, 3.f, CVAR_ARCHIVE)
CVAR(Bool, con_centernotify, false, CVAR_ARCHIVE)
// [BB] Zandronum handles con_scaletext differently
CVAR (Bool, con_scaletext, 0, CVAR_ARCHIVE)		// Scale text at high resolutions?
/*
CUSTOM_CVAR(Int, con_scaletext, 1, CVAR_ARCHIVE)		// Scale notify text at high resolutions?
{
	if (self < 0) self = 0;
	if (self > 3) self = 3;
}
*/

CUSTOM_CVAR(Int, con_scale, 0, CVAR_ARCHIVE)
{
	if (self < 0) self = 0;
}

int active_con_scale()
{
	int scale = con_scale;
	if (scale <= 0)
	{
		scale = CleanXfac - 1;
		if (scale <= 0)
		{
			scale = 1;
		}
	}
	return scale;
}

int active_con_scaletext()
{
	switch (con_scaletext)
	{
	default:
	case 0: return 1;
	case 1: return uiscale;
	case 2: return 2;
	case 3: return 4;
	}
}

CUSTOM_CVAR(Float, con_alpha, 0.75f, CVAR_ARCHIVE)
{
	if (self < 0.f) self = 0.f;
	if (self > 1.f) self = 1.f;
}

// Command to run when Ctrl-D is pressed at start of line
CVAR(String, con_ctrl_d, "", CVAR_ARCHIVE | CVAR_GLOBALCONFIG)

// [BC] Allow users to specify a virtual width and height when text scaling is enabled.
CUSTOM_CVAR( Int, con_virtualwidth, 640, CVAR_ARCHIVE )
{
	// [RC] Less than 4 crashes in the menu, less than 8 in game. Set to 32 to be safe.
	if ( self < 32 )
		self = 32;
}

CUSTOM_CVAR( Int, con_virtualheight, 480, CVAR_ARCHIVE )
{
	// [RC] Less than 4 crashes in the menu, less than 8 in game. Set to 32 to be safe.
	if ( self < 32 )
		self = 32;
}

// [BC] Allow text colors?
// [RC] Now a three-level setting. No/Yes/Not in chat.
CVAR( Int, con_colorinmessages, 1, CVAR_ARCHIVE )


struct History
{
	struct History *Older;
	struct History *Newer;
	FString String;
};

struct FCommandBuffer
{
	FString Text;	// The actual command line text
	unsigned CursorPos;
	unsigned StartPos;	// First character to display

	FCommandBuffer()
	{
		CursorPos = StartPos = 0;
	}

	FCommandBuffer(const FCommandBuffer &o)
	{
		Text = o.Text;
		CursorPos = o.CursorPos;
		StartPos = o.StartPos;
	}

	void Draw(int x, int y, int scale, bool cursor)
	{
		if (scale == 1)
		{
			screen->DrawChar(ConFont, CR_ORANGE, x, y, '\x1c', TAG_DONE);
			screen->DrawText(ConFont, CR_ORANGE, x + ConFont->GetCharWidth(0x1c), y,
				&Text[StartPos], TAG_DONE);

			if (cursor)
			{
				screen->DrawChar(ConFont, CR_YELLOW,
					x + ConFont->GetCharWidth(0x1c) + (CursorPos - StartPos) * ConFont->GetCharWidth(0xb),
					y, '\xb', TAG_DONE);
			}
		}
		else
		{
			screen->DrawChar(ConFont, CR_ORANGE, x, y, '\x1c',
				DTA_VirtualWidth, screen->GetWidth() / scale,
				DTA_VirtualHeight, screen->GetHeight() / scale,
				DTA_KeepRatio, true, TAG_DONE);

			screen->DrawText(ConFont, CR_ORANGE, x + ConFont->GetCharWidth(0x1c), y,
				&Text[StartPos],
				DTA_VirtualWidth, screen->GetWidth() / scale,
				DTA_VirtualHeight, screen->GetHeight() / scale,
				DTA_KeepRatio, true, TAG_DONE);

			if (cursor)
			{
				screen->DrawChar(ConFont, CR_YELLOW,
					x + ConFont->GetCharWidth(0x1c) + (CursorPos - StartPos) * ConFont->GetCharWidth(0xb),
					y, '\xb',
					DTA_VirtualWidth, screen->GetWidth() / scale,
					DTA_VirtualHeight, screen->GetHeight() / scale,
					DTA_KeepRatio, true, TAG_DONE);
			}
		}
	}

	void MakeStartPosGood()
	{
		int n = StartPos;
		unsigned cols = ConCols / active_con_scale();

		if (StartPos >= Text.Len())
		{ // Start of visible line is beyond end of line
			n = CursorPos - cols + 2;
		}
		if ((CursorPos - StartPos) >= cols - 2)
		{ // The cursor is beyond the visible part of the line
			n = CursorPos - cols + 2;
		}
		if (StartPos > CursorPos)
		{ // The cursor is in front of the visible part of the line
			n = CursorPos;
		}
		StartPos = MAX(0, n);
	}

	void CursorStart()
	{
		CursorPos = 0;
		StartPos = 0;
	}

	void CursorEnd()
	{
		CursorPos = (unsigned)Text.Len();
		StartPos = 0;
		MakeStartPosGood();
	}

	void CursorLeft()
	{
		if (CursorPos > 0)
		{
			CursorPos--;
			MakeStartPosGood();
		}
	}

	void CursorRight()
	{
		if (CursorPos < Text.Len())
		{
			CursorPos++;
			MakeStartPosGood();
		}
	}

	void DeleteLeft()
	{
		if (CursorPos > 0)
		{
			Text.Remove(CursorPos - 1, 1);
			CursorPos--;
			MakeStartPosGood();
		}
	}

	void DeleteRight()
	{
		if (CursorPos < Text.Len())
		{
			Text.Remove(CursorPos, 1);
			MakeStartPosGood();
		}
	}

	void AddChar(int character)
	{
		///FIXME: Not Unicode-aware
		if (CursorPos == Text.Len())
		{
			Text += char(character);
		}
		else
		{
			char foo = char(character);
			Text.Insert(CursorPos, &foo, 1);
		}
		CursorPos++;
		MakeStartPosGood();
	}

	void AddString(FString clip)
	{
		if (clip.IsNotEmpty())
		{
			// Only paste the first line.
			long brk = clip.IndexOfAny("\r\n\b");
			if (brk >= 0)
			{
				clip.Truncate(brk);
			}
			if (Text.IsEmpty())
			{
				Text = clip;
			}
			else
			{
				Text.Insert(CursorPos, clip);
			}
			CursorPos += (unsigned)clip.Len();
			MakeStartPosGood();
		}
	}

	void SetString(FString str)
	{
		Text = str;
		CursorPos = (unsigned)Text.Len();
		MakeStartPosGood();
	}
};
static FCommandBuffer CmdLine;

#define MAXHISTSIZE 50
static struct History *HistHead = NULL, *HistTail = NULL, *HistPos = NULL;
static int HistSize;

#define NUMNOTIFIES 4
#define NOTIFYFADETIME 6

struct FNotifyText
{
	int TimeOut;
	int PrintLevel;
	FString Text;
};

struct FNotifyBuffer
{
public:
	FNotifyBuffer();
	void AddString(int printlevel, FString source);
	void Shift(int maxlines);
	void Clear() { Text.Clear(); }
	void Tick();
	void Draw();

private:
	TArray<FNotifyText> Text;
	int Top;
	int TopGoal;
	enum { NEWLINE, APPENDLINE, REPLACELINE } AddType;
};
static FNotifyBuffer NotifyStrings;

CUSTOM_CVAR(Int, con_notifylines, NUMNOTIFIES, CVAR_GLOBALCONFIG | CVAR_ARCHIVE)
{
	NotifyStrings.Shift(self);
}


// [BC] Is there a player executing a remote control command? If so, display messages that
// are printed in the console as a result of his actions to him as well.
static	ULONG	g_ulRCONPlayer = MAXPLAYERS;

int PrintColors[PRINTLEVELS+2] = { CR_RED, CR_GOLD, CR_GRAY, CR_GREEN, CR_BRICK, CR_GOLD, CR_ORANGE };

static void setmsgcolor (int index, int color);

FILE *Logfile = NULL;

// [BC] The user's desired name of the logfile.
char g_szDesiredLogFilename[256];

// [RC] The actual name of the logfile (most likely g_szLogFilename with a timestamp).
char g_szActualLogFilename[256];


FIntCVar msglevel ("msg", 0, CVAR_ARCHIVE);

CUSTOM_CVAR (Int, msg0color, 6, CVAR_ARCHIVE)
{
	setmsgcolor (0, self);
}

CUSTOM_CVAR (Int, msg1color, 5, CVAR_ARCHIVE)
{
	setmsgcolor (1, self);
}

CUSTOM_CVAR (Int, msg2color, 2, CVAR_ARCHIVE)
{
	setmsgcolor (2, self);
}

CUSTOM_CVAR (Int, msg3color, 3, CVAR_ARCHIVE)
{
	setmsgcolor (3, self);
}

CUSTOM_CVAR (Int, msg4color, 0, CVAR_ARCHIVE)
{
	setmsgcolor (4, self);
}

CUSTOM_CVAR (Int, msgmidcolor, 5, CVAR_ARCHIVE)
{
	setmsgcolor (PRINTLEVELS, self);
}

CUSTOM_CVAR (Int, msgmidcolor2, 4, CVAR_ARCHIVE)
{
	setmsgcolor (PRINTLEVELS+1, self);
}

static void maybedrawnow (bool tick, bool force)
{
	// FIXME: Does not work right with hw2d
	if (ConsoleDrawing || screen == NULL || screen->IsLocked () || screen->Accel2D || ConFont == NULL)
	{
		return;
	}

	if (vidactive &&
		(((tick || gameaction != ga_nothing) && ConsoleState == c_down)
		|| gamestate == GS_STARTUP))
	{
		static size_t lastprinttime = 0;
		size_t nowtime = I_GetTime(false);

		if (nowtime - lastprinttime > 1 || force)
		{
			screen->Lock (false);
			C_DrawConsole (false);
			screen->Update ();
			lastprinttime = nowtime;
		}
	}
}

struct TextQueue
{
	TextQueue (bool notify, int printlevel, const char *text)
		: Next(NULL), bNotify(notify), PrintLevel(printlevel), Text(text)
	{
	}
	TextQueue *Next;
	bool bNotify;
	int PrintLevel;
	FString Text;
};

TextQueue *EnqueuedText, **EnqueuedTextTail = &EnqueuedText;

void EnqueueConsoleText (bool notify, int printlevel, const char *text)
{
	TextQueue *queued = new TextQueue (notify, printlevel, text);
	*EnqueuedTextTail = queued;
	EnqueuedTextTail = &queued->Next;
}

void DequeueConsoleText ()
{
	TextQueue *queued = EnqueuedText;

	while (queued != NULL)
	{
		TextQueue *next = queued->Next;
		if (queued->bNotify)
		{
			NotifyStrings.AddString(queued->PrintLevel, queued->Text);
		}
		else
		{
			AddToConsole (queued->PrintLevel, queued->Text);
		}
		delete queued;
		queued = next;
	}
	EnqueuedText = NULL;
	EnqueuedTextTail = &EnqueuedText;
}

void C_InitConback()
{
	// [BC] Initialize the name of the logfile.
	g_szDesiredLogFilename[0] = 0;
	g_szActualLogFilename[0] = 0;

	// [BC] The server has no use for a console.
	if ( Args->CheckParm( "-host" ))
		return;

	conback = TexMan.CheckForTexture ("CONBACK", FTexture::TEX_MiscPatch);

	if (!conback.isValid())
	{
		conback = TexMan.GetTexture (gameinfo.TitlePage, FTexture::TEX_MiscPatch);
		conshade = MAKEARGB(175,0,0,0);
		conline = true;
	}
	else
	{
		conshade = 0;
		conline = false;
	}
}

void C_InitConsole (int width, int height, bool ingame)
{
	int cwidth, cheight;

	vidactive = ingame;
	if (ConFont != NULL)
	{
		cwidth = ConFont->GetCharWidth ('M');
		cheight = ConFont->GetHeight();
	}
	else
	{
		cwidth = cheight = 8;
	}
	ConWidth = (width - LEFTMARGIN - RIGHTMARGIN);
	ConCols = ConWidth / cwidth;

	if (conbuffer == NULL) conbuffer = new FConsoleBuffer;
}

//==========================================================================
//
// CCMD atexit
//
//==========================================================================

CCMD (atexit)
{
	if (argv.argc() == 1)
	{
		Printf ("Registered atexit commands:\n");
		GameAtExit *record = ExitCmdList;
		while (record != NULL)
		{
			Printf ("%s\n", record->Command.GetChars());
			record = record->Next;
		}
		return;
	}
	for (int i = 1; i < argv.argc(); ++i)
	{
		GameAtExit *record = new GameAtExit(argv[i]);
		record->Next = ExitCmdList;
		ExitCmdList = record;
	}
}

//==========================================================================
//
// C_DeinitConsole
//
// Executes the contents of the atexit cvar, if any, at quit time.
// Then releases all of the console's memory.
//
//==========================================================================

void C_DeinitConsole ()
{
	GameAtExit *cmd = ExitCmdList;

	while (cmd != NULL)
	{
		GameAtExit *next = cmd->Next;
		AddCommandString (cmd->Command.LockBuffer());
		delete cmd;
		cmd = next;
	}

	// Free command history
	History *hist = HistTail;

	while (hist != NULL)
	{
		History *next = hist->Newer;
		delete hist;
		hist = next;
	}
	HistTail = HistHead = HistPos = NULL;

	// Free cvars allocated at runtime
	FBaseCVar *var, *next, **nextp;
	for (var = CVars, nextp = &CVars; var != NULL; var = next)
	{
		next = var->m_Next;
		if (var->GetFlags() & CVAR_UNSETTABLE)
		{
			delete var;
			*nextp = next;
		}
		else
		{
			nextp = &var->m_Next;
		}
	}

	// Free alias commands. (i.e. The "commands" that can be allocated
	// at runtime.)
	for (size_t i = 0; i < countof(Commands); ++i)
	{
		FConsoleCommand *cmd = Commands[i];

		while (cmd != NULL)
		{
			FConsoleCommand *next = cmd->m_Next;
			if (cmd->IsAlias())
			{
				delete cmd;
			}
			cmd = next;
		}
	}

	// Make sure all tab commands are cleared before the memory for
	// their names is deallocated.
	C_ClearTabCommands ();

	// Free AddToConsole()'s work buffer
	if (work != NULL)
	{
		free (work);
		work = NULL;
		worklen = 0;
	}

	if (conbuffer != NULL)
	{
		delete conbuffer;
		conbuffer = NULL;
	}
}

static void ClearConsole ()
{
	// [BC] The server has no need for this.
	if ( Args->CheckParm( "-host" ))
		return;

	if (conbuffer != NULL)
	{
		conbuffer->Clear();
	}
	TopLine = InsertLine = 0;
}

static void setmsgcolor (int index, int color)
{
	if ((unsigned)color >= (unsigned)NUM_TEXT_COLORS)
		color = 0;
	PrintColors[index] = color;
}

extern int DisplayWidth;

FNotifyBuffer::FNotifyBuffer()
{
	Top = TopGoal = 0;
	AddType = NEWLINE;
}

void FNotifyBuffer::Shift(int maxlines)
{
	if (maxlines >= 0 && Text.Size() > (unsigned)maxlines)
	{
		Text.Delete(0, Text.Size() - maxlines);
	}
}

void FNotifyBuffer::AddString(int printlevel, FString source)
{
	// [BC] The server has no need for this.
	if ( Args->CheckParm( "-host" ))
		return;

	FBrokenLines *lines;
	int i, width;

	if ((printlevel != 128 && !show_messages) ||
		source.IsEmpty() ||
		gamestate == GS_FULLCONSOLE ||
		gamestate == GS_DEMOSCREEN ||
		con_notifylines == 0)
		return;

	if (ConsoleDrawing)
	{
		EnqueueConsoleText (true, printlevel, source);
		return;
	}

	// [BC] If text scaling is enabled, allow users to specify a virtual screen width/height.
	//if (active_con_scaletext() == 0)
	if ( con_scaletext )
		width = con_virtualwidth;
	else
		width = DisplayWidth;
	/* [BB] Zandronum handles con_scaletext differently
	if ( con_scaletext )
	{
		width = DisplayWidth / CleanXfac;
	}
	else
	{
		width = DisplayWidth / active_con_scaletext();
	}
	*/

	if (AddType == APPENDLINE && Text.Size() > 0 && Text[Text.Size() - 1].PrintLevel == printlevel)
	{
		FString str = Text[Text.Size() - 1].Text + source;
		lines = V_BreakLines (SmallFont, width, str);
	}
	else
	{
		lines = V_BreakLines (SmallFont, width, source);
		if (AddType == APPENDLINE)
		{
			AddType = NEWLINE;
		}
	}

	if (lines == NULL)
		return;

	for (i = 0; lines[i].Width >= 0; i++)
	{
		FNotifyText newline;

		newline.Text = lines[i].Text;
		newline.TimeOut = gametic + int(con_notifytime * TICRATE);
		newline.PrintLevel = printlevel;
		if (AddType == NEWLINE || Text.Size() == 0)
		{
			if (con_notifylines > 0)
			{
				Shift(con_notifylines - 1);
			}
			Text.Push(newline);
		}
		else
		{
			Text[Text.Size() - 1] = newline;
		}
		AddType = NEWLINE;
	}

	V_FreeBrokenLines (lines);
	lines = NULL;

	switch (source[source.Len()-1])
	{
	case '\r':	AddType = REPLACELINE;	break;
	case '\n':	AddType = NEWLINE;		break;
	default:	AddType = APPENDLINE;	break;
	}

	TopGoal = 0;
}

//*****************************************************************************
//
void CONSOLE_SetRCONPlayer( ULONG ulPlayer )
{
	g_ulRCONPlayer = ulPlayer;
}

void AddToConsole (int printlevel, const char *text)
{
	// [BB] The server doesn't have a conbuffer.
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
	{
		PrintString ( printlevel, text );
		return;
	}

	conbuffer->AddText(printlevel, text, Logfile);
}

void	SERVERCONSOLE_Print( char *pszString );

/* Adds a string to the console and also to the notify buffer */
int PrintString (int printlevel, const char *outline)
{
	if (printlevel < msglevel || *outline == '\0')
	{
		return 0;
	}

	// [TP] Possibly capture it instead
	if ( C_IsCapturing() )
	{
		g_CaptureBuffer += outline;
		return 0;
	}

	// [BB]: outline is const, it may NOT be altered! Since some of the functions below 
	// alter the output string, we have to make a copy of it and only alter this.

	char *outlinecopy = new char[strlen(outline)+1];
	strcpy (outlinecopy,outline);

	// For servers, dump message to console window.
	// [BB] If we are coming from I_Quit, Args is possibly already invalid.
	if ( Args && Args->CheckParm( "-host" ))
	{
		// [BB] Since the server doesn't have a console buffer, it logs the output immediately.
		FConsoleBuffer::LogLine ( Logfile, outlinecopy );
		if ( printlevel != PRINT_LOW )
		{			
			// [RC] Send this to any connected RCON clients.
			SERVER_RCON_Print( outlinecopy );
			if ( g_ulRCONPlayer != MAXPLAYERS )
				SERVER_PrintfPlayer( printlevel, g_ulRCONPlayer, "%s", outlinecopy );

			SERVERCONSOLE_Print( outlinecopy );
		}
		
		const int length = static_cast<int>(strlen (outlinecopy));
		delete [] outlinecopy;
		return length;
	}

	// [RC] Send this to the G15 LCD, if enabled.
	if ( G15_IsReady() )
		G15_Printf( outlinecopy );

	// User wishes to remove color from all messages.
	if ( con_colorinmessages == 0 )
		V_RemoveColorCodes( outlinecopy );

	if (printlevel != PRINT_LOG)
	{
		I_PrintStr (outlinecopy);

		if ( NETWORK_GetState( ) != NETSTATE_SERVER )
		{
			AddToConsole (printlevel, outlinecopy);

			if (vidactive && screen && SmallFont)
			{
				NotifyStrings.AddString(printlevel, outline);
				maybedrawnow (false, false);
			}
		}
	}
	else if (Logfile != NULL)
	{
		fputs (outlinecopy, Logfile);
		fflush (Logfile);
	}
	const int length = static_cast<int>(strlen (outlinecopy));
	delete [] outlinecopy;
	return length;

}

extern bool gameisdead;

int VPrintf (int printlevel, const char *format, va_list parms)
{
	if (gameisdead)
		return 0;

	FString outline;
	outline.VFormat (format, parms);
	return PrintString (printlevel, outline.GetChars());
}

int Printf (int printlevel, const char *format, ...)
{
	va_list argptr;
	int count;

	va_start (argptr, format);
	count = VPrintf (printlevel, format, argptr);
	va_end (argptr);

	return count;
}

int Printf (const char *format, ...)
{
	va_list argptr;
	int count;

	va_start (argptr, format);
	count = VPrintf (PRINT_HIGH, format, argptr);
	va_end (argptr);

	return count;
}

int DPrintf (int level, const char *format, ...)
{
	va_list argptr;
	int count;

	if (developer >= level)
	{
		va_start (argptr, format);
		count = VPrintf (PRINT_HIGH, format, argptr);
		va_end (argptr);
		return count;
	}
	else
	{
		return 0;
	}
}

void C_FlushDisplay ()
{
	NotifyStrings.Clear();
}

void C_AdjustBottom ()
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	// [Leo] Keep the fullconsole if we are requesting/receiving a snapshot.
	if ( gamestate == GS_FULLCONSOLE || gamestate == GS_STARTUP || 
		( NETWORK_InClientMode() && ( CLIENT_GetConnectionState() != CTS_ACTIVE )) )
		ConBottom = SCREENHEIGHT;
	else if (ConBottom > SCREENHEIGHT / 2 || ConsoleState == c_down)
		ConBottom = SCREENHEIGHT / 2;
}

void C_NewModeAdjust ()
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	C_InitConsole (SCREENWIDTH, SCREENHEIGHT, true);
	C_FlushDisplay ();
	C_AdjustBottom ();
}

int consoletic = 0;
void C_Ticker()
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	static int lasttic = 0;
	consoletic++;

	if (lasttic == 0)
		lasttic = consoletic - 1;

	if (con_buffersize > 0)
	{
		conbuffer->ResizeBuffer(con_buffersize);
	}

	if (ConsoleState != c_up)
	{
		if (ConsoleState == c_falling)
		{
			ConBottom += (consoletic - lasttic) * (SCREENHEIGHT * 2 / 25);
			if (ConBottom >= SCREENHEIGHT / 2)
			{
				ConBottom = SCREENHEIGHT / 2;
				ConsoleState = c_down;
			}
		}
		else if (ConsoleState == c_rising)
		{
			ConBottom -= (consoletic - lasttic) * (SCREENHEIGHT * 2 / 25);
			if (ConBottom <= 0)
			{
				ConsoleState = c_up;
				ConBottom = 0;
			}
		}
	}

	if (--CursorTicker <= 0)
	{
		cursoron ^= 1;
		CursorTicker = C_BLINKRATE;
	}

	lasttic = consoletic;
	NotifyStrings.Tick();
}

void FNotifyBuffer::Tick()
{
	if (TopGoal > Top)
	{
		Top++;
	}
	else if (TopGoal < Top)
	{
		Top--;
	}

	// Remove lines from the beginning that have expired.
	unsigned i;
	for (i = 0; i < Text.Size(); ++i)
	{
		if (Text[i].TimeOut != 0 && Text[i].TimeOut > gametic)
			break;
	}
	if (i > 0)
	{
		Text.Delete(0, i);
	}
}

void FNotifyBuffer::Draw()
{
	// [BC] Some variables for text scaling.
	bool		bScale;
	UCVarValue	ValWidth;
	UCVarValue	ValHeight;

	// [BC] Initialization.
	ValWidth = con_virtualwidth.GetGenericRep( CVAR_Int );
	ValHeight = con_virtualheight.GetGenericRep( CVAR_Int );
	if (( con_scaletext ) && ( con_virtualwidth > 0 ) && ( con_virtualheight > 0 ))
		bScale = true;
	else
		bScale = false;

	// [BC] We have no need to do this in server mode.
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	bool center = (con_centernotify != 0.f);
	int line, lineadv, color, j;
	bool canskip;
	
	if (gamestate == GS_FULLCONSOLE || gamestate == GS_DEMOSCREEN/* || menuactive != MENU_Off*/)
		return;

	line = Top;
	canskip = true;

	lineadv = SmallFont->GetHeight ();
	// [BC] We no longer need to scale lineadv since we specify virtual screen coordinates.
/*
	if (active_con_scaletext() == 0)
	{
		lineadv *= CleanYfac;
	}
*/

	BorderTopRefresh = screen->GetPageCount ();

	for (unsigned i = 0; i < Text.Size(); ++ i)
	{
		FNotifyText &notify = Text[i];

		if (notify.TimeOut == 0)
			continue;

		j = notify.TimeOut - gametic;
		if (j > 0)
		{
			if (!show_messages && notify.PrintLevel != 128)
				continue;

			double alpha = (j < NOTIFYFADETIME) ? 1. * j / NOTIFYFADETIME : 1;

			if (notify.PrintLevel >= PRINTLEVELS)
				color = CR_UNTRANSLATED;
			else
				color = PrintColors[notify.PrintLevel];

			/* [BB] Zandronum handles con_scaletext differently
			if (active_con_scaletext() == 0)
			{
				if (!center)
					screen->DrawText (SmallFont, color, 0, line, notify.Text,
						DTA_CleanNoMove, true, DTA_AlphaF, alpha, TAG_DONE);
				else
					screen->DrawText (SmallFont, color, (SCREENWIDTH -
						SmallFont->StringWidth (notify.Text)*CleanXfac)/2,
						line, notify.Text, DTA_CleanNoMove, true,
						DTA_AlphaF, alpha, TAG_DONE);
			}
			else if (active_con_scaletext() == 1)
			*/
			{
			// [BC] If we want scaling, handle that here.
				if (!center)
					screen->DrawText (SmallFont, color, 0, line, notify.Text,
						DTA_UseVirtualScreen, bScale, // [BB]
						DTA_AlphaF, alpha, TAG_DONE);
				else
					screen->DrawText (SmallFont, color, ( ( bScale ? ValWidth.Int : SCREENWIDTH ) -
						SmallFont->StringWidth (notify.Text))/2,
						line, notify.Text,
						DTA_UseVirtualScreen, bScale, // [BB]
						DTA_AlphaF, alpha, TAG_DONE);
			}
			/* [BB] Zandronum handles con_scaletext differently
			else
			{
				if (!center)
					screen->DrawText (SmallFont, color, 0, line, notify.Text,
						DTA_VirtualWidth, screen->GetWidth() / active_con_scaletext(),
						DTA_VirtualHeight, screen->GetHeight() / active_con_scaletext(),
						DTA_KeepRatio, true,
						DTA_AlphaF, alpha, TAG_DONE);
				else
					screen->DrawText (SmallFont, color, (screen->GetWidth() -
						SmallFont->StringWidth (notify.Text) * active_con_scaletext()) / 2 / active_con_scaletext(),
						line, notify.Text,
						DTA_VirtualWidth, screen->GetWidth() / active_con_scaletext(),
						DTA_VirtualHeight, screen->GetHeight() / active_con_scaletext(),
						DTA_KeepRatio, true,
						DTA_AlphaF, alpha, TAG_DONE);
			}
			*/
			line += lineadv;
			canskip = false;
		}
		else
		{
			if (canskip)
			{
				Top += lineadv;
				line += lineadv;
			}
			notify.TimeOut = 0;
		}
	}
	if (canskip)
	{
		Top = TopGoal;
	}
}

void C_InitTicker (const char *label, unsigned int max, bool showpercent)
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	TickerPercent = showpercent;
	TickerMax = max;
	TickerLabel = label;
	TickerAt = 0;
	maybedrawnow (true, false);
}

void C_SetTicker (unsigned int at, bool forceUpdate)
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	TickerAt = at > TickerMax ? TickerMax : at;
	maybedrawnow (true, TickerVisible ? forceUpdate : false);
}

void C_DrawConsole (bool hw2d)
{
	static int oldbottom = 0;
	int lines, left, offset;
	// [BC] String for drawing the version.
	char	szString[64];

	// [BC] No need to draw the console in server mode.
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	int textScale = active_con_scale();

	left = LEFTMARGIN;
	lines = (ConBottom/textScale-ConFont->GetHeight()*2)/ConFont->GetHeight();
	if (-ConFont->GetHeight() + lines*ConFont->GetHeight() > ConBottom/textScale - ConFont->GetHeight()*7/2)
	{
		offset = -ConFont->GetHeight()/2;
		lines--;
	}
	else
	{
		offset = -ConFont->GetHeight();
	}

	if ((ConBottom < oldbottom) &&
		(gamestate == GS_LEVEL || gamestate == GS_TITLELEVEL) &&
		(viewwindowx || viewwindowy) &&
		viewactive)
	{
		V_SetBorderNeedRefresh();
	}

	oldbottom = ConBottom;

	if (ConsoleState == c_up)
	{
		NotifyStrings.Draw();
		return;
	}
	else if (ConBottom)
	{
		int visheight;
		FTexture *conpic = TexMan[conback];

		visheight = ConBottom;

		screen->DrawTexture (conpic, 0, visheight - screen->GetHeight(),
			DTA_DestWidth, screen->GetWidth(),
			DTA_DestHeight, screen->GetHeight(),
			DTA_ColorOverlay, conshade,
			DTA_AlphaF, (hw2d && gamestate != GS_FULLCONSOLE) ? (double)con_alpha : 1.,
			DTA_Masked, false,
			TAG_DONE);
		if (conline && visheight < screen->GetHeight())
		{
			screen->Clear (0, visheight, screen->GetWidth(), visheight+1, 0, 0);
		}

		if (ConBottom >= 12)
		{
			// [BC] In addition to drawing the program version, draw the ZDoom version as well.
			// [RC] Also draw revision number, but break these up so it's legible.
			sprintf( szString, "\\cIv%s (\\cD%s\\cI) \\ch%s", GetVersionString(), ZDOOMVERSIONSTR, GetGitTime() );
			V_ColorizeString( szString );

			if (textScale == 1)
				screen->DrawText (ConFont, CR_ORANGE, SCREENWIDTH - 8 -
					ConFont->StringWidth( szString ),
					ConBottom / textScale - ConFont->GetHeight() - 4,
					szString, TAG_DONE );
			else
				screen->DrawText(ConFont, CR_ORANGE, SCREENWIDTH / textScale - 8 -
					ConFont->StringWidth( szString ),
					ConBottom / textScale - ConFont->GetHeight() - 4,
					 szString,
					DTA_VirtualWidth, screen->GetWidth() / textScale,
					DTA_VirtualHeight, screen->GetHeight() / textScale,
					DTA_KeepRatio, true, TAG_DONE);

			if (TickerMax)
			{
				char tickstr[256];
				const int tickerY = ConBottom / textScale - ConFont->GetHeight() - 4;
				size_t i;
				int tickend = ConCols / textScale - SCREENWIDTH / textScale / 90 - 6;
				int tickbegin = 0;

				if (TickerLabel)
				{
					tickbegin = (int)strlen (TickerLabel) + 2;
					mysnprintf (tickstr, countof(tickstr), "%s: ", TickerLabel);
				}
				if (tickend > 256 - ConFont->GetCharWidth(0x12))
					tickend = 256 - ConFont->GetCharWidth(0x12);
				tickstr[tickbegin] = 0x10;
				memset (tickstr + tickbegin + 1, 0x11, tickend - tickbegin);
				tickstr[tickend + 1] = 0x12;
				tickstr[tickend + 2] = ' ';
				if (TickerPercent)
				{
					mysnprintf (tickstr + tickend + 3, countof(tickstr) - tickend - 3,
						"%d%%", Scale (TickerAt, 100, TickerMax));
				}
				else
				{
					tickstr[tickend+3] = 0;
				}
				if (textScale == 1)
					screen->DrawText (ConFont, CR_BROWN, LEFTMARGIN, tickerY, tickstr, TAG_DONE);
				else
					screen->DrawText (ConFont, CR_BROWN, LEFTMARGIN, tickerY, tickstr,
						DTA_VirtualWidth, screen->GetWidth() / textScale,
						DTA_VirtualHeight, screen->GetHeight() / textScale,
						DTA_KeepRatio, true, TAG_DONE);

				// Draw the marker
				i = LEFTMARGIN+5+tickbegin*8 + Scale (TickerAt, (SDWORD)(tickend - tickbegin)*8, TickerMax);
				if (textScale == 1)
					screen->DrawChar (ConFont, CR_ORANGE, (int)i, tickerY, 0x13, TAG_DONE);
				else
					screen->DrawChar(ConFont, CR_ORANGE, (int)i, tickerY, 0x13,
						DTA_VirtualWidth, screen->GetWidth() / textScale,
						DTA_VirtualHeight, screen->GetHeight() / textScale,
						DTA_KeepRatio, true, TAG_DONE);

				TickerVisible = true;
			}
			else
			{
				TickerVisible = false;
			}
		}

		// Apply palette blend effects
		if (StatusBar != NULL && !hw2d)
		{
			player_t *player = StatusBar->CPlayer;
			if (player->camera != NULL && player->camera->player != NULL)
			{
				player = player->camera->player;
			}
			if (player->BlendA != 0 && (gamestate == GS_LEVEL || gamestate == GS_TITLELEVEL))
			{
				screen->Dim (PalEntry ((unsigned char)(player->BlendR*255), (unsigned char)(player->BlendG*255), (unsigned char)(player->BlendB*255)),
					player->BlendA, 0, ConBottom, screen->GetWidth(), screen->GetHeight() - ConBottom);
				ST_SetNeedRefresh();
				V_SetBorderNeedRefresh();
			}
		}
	}

	if (menuactive != MENU_Off)
	{
		return;
	}

	if (lines > 0)
	{
		// No more enqueuing because adding new text to the console won't touch the actual print data.
		conbuffer->FormatText(ConFont, ConWidth / textScale);
		unsigned int consolelines = conbuffer->GetFormattedLineCount();
		FBrokenLines **blines = conbuffer->GetLines();
		FBrokenLines **printline = blines + consolelines - 1 - RowAdjust;

		int bottomline = ConBottom / textScale - ConFont->GetHeight()*2 - 4;

		ConsoleDrawing = true;

		for(FBrokenLines **p = printline; p >= blines && lines > 0; p--, lines--)
		{
			if (textScale == 1)
			{
				screen->DrawText(ConFont, CR_TAN, LEFTMARGIN, offset + lines * ConFont->GetHeight(), (*p)->Text, TAG_DONE);
			}
			else
			{
				screen->DrawText(ConFont, CR_TAN, LEFTMARGIN, offset + lines * ConFont->GetHeight(), (*p)->Text,
					DTA_VirtualWidth, screen->GetWidth() / textScale,
					DTA_VirtualHeight, screen->GetHeight() / textScale,
					DTA_KeepRatio, true, TAG_DONE);
			}
		}

		ConsoleDrawing = false;

		if (ConBottom >= 20)
		{
			if (gamestate != GS_STARTUP)
			{
				// Make a copy of the command line, in case an input event is handled
				// while we draw the console and it changes.
				FCommandBuffer command(CmdLine);
				command.Draw(left, bottomline, textScale, cursoron);
			}
			if (RowAdjust && ConBottom >= ConFont->GetHeight()*7/2)
			{
				// Indicate that the view has been scrolled up (10)
				// and if we can scroll no further (12)
				if (textScale == 1)
					screen->DrawChar (ConFont, CR_GREEN, 0, bottomline, RowAdjust == conbuffer->GetFormattedLineCount() ? 12 : 10, TAG_DONE);
				else
					screen->DrawChar(ConFont, CR_GREEN, 0, bottomline, RowAdjust == conbuffer->GetFormattedLineCount() ? 12 : 10,
						DTA_VirtualWidth, screen->GetWidth() / textScale,
						DTA_VirtualHeight, screen->GetHeight() / textScale,
						DTA_KeepRatio, true, TAG_DONE);
			}
		}
	}
}

void C_FullConsole ()
{
	// [BC] The server doesn't have a console.
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	if (demoplayback)
		G_CheckDemoStatus ();
	D_QuitNetGame ();
	advancedemo = false;
	ConsoleState = c_down;
	HistPos = NULL;
	TabbedLast = false;
	TabbedList = false;
	if (gamestate != GS_STARTUP)
	{
		gamestate = GS_FULLCONSOLE;
		level.Music = "";
		S_Start ();
		P_FreeLevelData ();
		V_SetBlend (0,0,0,0);
	}
	else
	{
		C_AdjustBottom ();
	}
}

void C_ToggleConsole ()
{
	// [Leo] Don't let the console close while requesting/receiving a snapshot.
	if (( NETWORK_GetState( ) == NETSTATE_SERVER ) ||
		( NETWORK_InClientMode() && ( CLIENT_GetConnectionState() != CTS_ACTIVE )))
		return;

	if (gamestate == GS_DEMOSCREEN || demoplayback)
	{
		gameaction = ga_fullconsole;
	}
	else if (( CHAT_GetChatMode( ) == 0 && (ConsoleState == c_up || ConsoleState == c_rising)) && menuactive == MENU_Off)
	{
		ConsoleState = c_falling;
		HistPos = NULL;
		TabbedLast = false;
		TabbedList = false;

		// [BB] Don't change the displayed console status when a demo is played.
		if ( CLIENTDEMO_IsPlaying( ) == false )
			players[consoleplayer].bInConsole = true;

		// [RC] Tell the server so we get an "in console" icon.
		if ( NETWORK_GetState( ) == NETSTATE_CLIENT )
			CLIENTCOMMANDS_EnterConsole( );
	}
	else if (gamestate != GS_FULLCONSOLE && gamestate != GS_STARTUP)
	{
		ConsoleState = c_rising;
		C_FlushDisplay ();

		// [BB] Don't change the displayed console status when a demo is played.
		if ( CLIENTDEMO_IsPlaying( ) == false )
			players[consoleplayer].bInConsole = false;

		// [RC] Tell the server so our "in console" icon is removed.
		if ( NETWORK_GetState( ) == NETSTATE_CLIENT )
			CLIENTCOMMANDS_ExitConsole( );

	}
}

void C_HideConsole ()
{
	// [Leo] Don't let the console close while requesting/receiving a snapshot.
	if (( NETWORK_GetState( ) == NETSTATE_SERVER ) ||
		( NETWORK_InClientMode() && ( CLIENT_GetConnectionState() != CTS_ACTIVE )))
		return;

	if (gamestate != GS_FULLCONSOLE)
	{
		ConsoleState = c_up;
		ConBottom = 0;
		HistPos = NULL;

		// [BB] We are not in console anymore, so set bInConsole if necessary.
		if ( players[consoleplayer].bInConsole )
		{
			// [BB] Don't change the displayed console status when a demo is played.
			if ( CLIENTDEMO_IsPlaying( ) == false )
				players[consoleplayer].bInConsole = false;

			// [RC] Tell the server so our "in console" icon is removed.
			if ( NETWORK_GetState( ) == NETSTATE_CLIENT )
				CLIENTCOMMANDS_ExitConsole( );
		}
	}
}

static bool C_HandleKey (event_t *ev, FCommandBuffer &buffer)
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return ( true );

	int data1 = ev->data1;

	switch (ev->subtype)
	{
	default:
		return false;

	case EV_GUI_Char:
		// Add keypress to command line
		buffer.AddChar(ev->data1);
		HistPos = NULL;
		TabbedLast = false;
		TabbedList = false;
		break;

	case EV_GUI_WheelUp:
	case EV_GUI_WheelDown:
		if (!(ev->data3 & GKM_SHIFT))
		{
			data1 = GK_PGDN + EV_GUI_WheelDown - ev->subtype;
		}
		else
		{
			data1 = GK_DOWN + EV_GUI_WheelDown - ev->subtype;
		}
		// Intentional fallthrough

	case EV_GUI_KeyDown:
	case EV_GUI_KeyRepeat:
		switch (data1)
		{
		case '\t':
			// Try to do tab-completion
			C_TabComplete ((ev->data3 & GKM_SHIFT) ? false : true);
			break;

		case GK_PGUP:
			if (ev->data3 & (GKM_SHIFT|GKM_CTRL))
			{ // Scroll console buffer up one page
				RowAdjust += (SCREENHEIGHT-4)/active_con_scale() /
					((gamestate == GS_FULLCONSOLE || gamestate == GS_STARTUP) ? ConFont->GetHeight() : ConFont->GetHeight()*2) - 3;
			}
			else if (RowAdjust < conbuffer->GetFormattedLineCount())
			{ // Scroll console buffer up
				if (ev->subtype == EV_GUI_WheelUp)
				{
					RowAdjust += 3;
				}
				else
				{
					RowAdjust++;
				}
				if (RowAdjust > conbuffer->GetFormattedLineCount())
				{
					RowAdjust = conbuffer->GetFormattedLineCount();
				}
			}
			break;

		case GK_PGDN:
			if (ev->data3 & (GKM_SHIFT|GKM_CTRL))
			{ // Scroll console buffer down one page
				const int scrollamt = (SCREENHEIGHT-4)/active_con_scale() /
					((gamestate == GS_FULLCONSOLE || gamestate == GS_STARTUP) ? ConFont->GetHeight() : ConFont->GetHeight()*2) - 3;
				if (RowAdjust < scrollamt)
				{
					RowAdjust = 0;
				}
				else
				{
					RowAdjust -= scrollamt;
				}
			}
			else if (RowAdjust > 0)
			{ // Scroll console buffer down
				if (ev->subtype == EV_GUI_WheelDown)
				{
					RowAdjust = MAX (0, RowAdjust - 3);
				}
				else
				{
					RowAdjust--;
				}
			}
			break;

		case GK_HOME:
			if (ev->data3 & GKM_CTRL)
			{ // Move to top of console buffer
				RowAdjust = conbuffer->GetFormattedLineCount();
			}
			else
			{ // Move cursor to start of line
				buffer.CursorStart();
			}
			break;

		case GK_END:
			if (ev->data3 & GKM_CTRL)
			{ // Move to bottom of console buffer
				RowAdjust = 0;
			}
			else
			{ // Move cursor to end of line
				buffer.CursorEnd();
			}
			break;

		case GK_LEFT:
			// Move cursor left one character
			buffer.CursorLeft();
			break;

		case GK_RIGHT:
			// Move cursor right one character
			buffer.CursorRight();
			break;

		case '\b':
			// Erase character to left of cursor
			buffer.DeleteLeft();
			TabbedLast = false;
			TabbedList = false;
			break;

		case GK_DEL:
			// Erase character under cursor
			buffer.DeleteRight();
			TabbedLast = false;
			TabbedList = false;
			break;

		case GK_UP:
			// Move to previous entry in the command history
			if (HistPos == NULL)
			{
				HistPos = HistHead;
			}
			else if (HistPos->Older)
			{
				HistPos = HistPos->Older;
			}

			if (HistPos)
			{
				buffer.SetString(HistPos->String);
			}

			TabbedLast = false;
			TabbedList = false;
			break;

		case GK_DOWN:
			// Move to next entry in the command history
			if (HistPos && HistPos->Newer)
			{
				HistPos = HistPos->Newer;
				buffer.SetString(HistPos->String);
			}
			else
			{
				HistPos = NULL;
				buffer.SetString("");
			}
			TabbedLast = false;
			TabbedList = false;
			break;

		case 'X':
			if (ev->data3 & GKM_CTRL)
			{
				buffer.SetString("");
				TabbedLast = TabbedList = false;
			}
			break;

		case 'D':
			if (ev->data3 & GKM_CTRL && buffer.Text.Len() == 0)
			{ // Control-D pressed on an empty line
				if (strlen(con_ctrl_d) == 0)
				{
					break;	// Replacement is empty, so do nothing
				}
				buffer.SetString(*con_ctrl_d);
			}
			else
			{
				break;
			}
			// Intentional fall-through for command(s) added with Ctrl-D

		case '\r':
			// Execute command line (ENTER)

			buffer.Text.StripLeftRight();
			Printf(127, TEXTCOLOR_WHITE "]%s\n", buffer.Text.GetChars());

			if (buffer.Text.Len() == 0)
			{
				 // Command line is empty, so do nothing to the history
			}
			else if (HistHead && HistHead->String.CompareNoCase(buffer.Text) == 0)
			{
				// Command line was the same as the previous one,
				// so leave the history list alone
			}
			else
			{
				// Command line is different from last command line,
				// or there is nothing in the history list,
				// so add it to the history list.

				History *temp = new History;
				temp->String = buffer.Text;
				temp->Older = HistHead;
				if (HistHead)
				{
					HistHead->Newer = temp;
				}
				temp->Newer = NULL;
				HistHead = temp;

				if (!HistTail)
				{
					HistTail = temp;
				}

				if (HistSize == MAXHISTSIZE)
				{
					HistTail = HistTail->Newer;
					delete HistTail->Older;
					HistTail->Older = NULL;
				}
				else
				{
					HistSize++;
				}
			}
			HistPos = NULL;
			{
				// Work with a copy of command to avoid side effects caused by
				// exception raised during execution, like with 'error' CCMD.
				// It's problematic to maintain FString's lock symmetry.
				static TArray<char> command;
				const size_t length = buffer.Text.Len();

				command.Resize(unsigned(length + 1));
				memcpy(&command[0], buffer.Text.GetChars(), length);
				command[length] = '\0';

				buffer.SetString("");

				AddCommandString(&command[0]);
			}
			TabbedLast = false;
			TabbedList = false;
			break;
		
		case '`':
			// Check to see if we have ` bound to the console before accepting
			// it as a way to close the console.
			if (Bindings.GetBinding(KEY_GRAVE).CompareNoCase("toggleconsole"))
			{
				break;
			}
		case GK_ESCAPE:
			// Close console and clear command line. But if we're in the
			// fullscreen console mode, there's nothing to fall back on
			// if it's closed, so open the main menu instead.
			if (gamestate == GS_STARTUP)
			{
				return false;
			}
			else if ((gamestate == GS_FULLCONSOLE) ||
				( NETWORK_InClientMode() &&
				( CLIENT_GetConnectionState( ) != CTS_ACTIVE )))
			{
				C_DoCommand ("menu_main");
			}
			else
			{
				buffer.SetString("");
				HistPos = NULL;
				C_ToggleConsole ();
			}
			break;

		case 'C':
		case 'V':
			TabbedLast = false;
			TabbedList = false;
#ifdef __APPLE__
			if (ev->data3 & GKM_META)
#else // !__APPLE__
			if (ev->data3 & GKM_CTRL)
#endif // __APPLE__
			{
				if (data1 == 'C')
				{ // copy to clipboard
					if (buffer.Text.IsNotEmpty())
					{
						I_PutInClipboard(buffer.Text);
					}
				}
				else
				{ // paste from clipboard
					buffer.AddString(I_GetFromClipboard(false));
					HistPos = NULL;
				}
				break;
			}
			break;
		}
		break;

#ifdef __unix__
	case EV_GUI_MButtonDown:
		buffer.AddString(I_GetFromClipboard(true));
		HistPos = NULL;
		break;
#endif
	}
	// Ensure that the cursor is always visible while typing
	CursorTicker = C_BLINKRATE;
	cursoron = 1;
	return true;
}

bool C_Responder (event_t *ev)
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return ( false );

	if (ev->type != EV_GUI_Event ||
		ConsoleState == c_up ||
		ConsoleState == c_rising ||
		menuactive != MENU_Off)
	{
		return false;
	}

	return C_HandleKey(ev, CmdLine);
}

CCMD (history)
{
	struct History *hist = HistTail;

	while (hist)
	{
		Printf ("   %s\n", hist->String.GetChars());
		hist = hist->Newer;
	}
}

CCMD (clear)
{
	if ( NETWORK_GetState( ) == NETSTATE_SERVER )
		return;

	C_FlushDisplay ();
	ClearConsole ();
}

CCMD (echo)
{
	int last = argv.argc()-1;
	for (int i = 1; i <= last; ++i)
	{
		FString formatted = strbin1 (argv[i]);
		Printf ("%s%s", formatted.GetChars(), i!=last ? " " : "\n");
	}
}

/* Printing in the middle of the screen */

CVAR (Float, con_midtime, 3.f, CVAR_ARCHIVE)

static const char bar1[] = TEXTCOLOR_RED "\n\35\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36"
						  "\36\36\36\36\36\36\36\36\36\36\36\36\37" TEXTCOLOR_TAN "\n";
static const char bar2[] = TEXTCOLOR_RED "\n\35\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36"
						  "\36\36\36\36\36\36\36\36\36\36\36\36\37" TEXTCOLOR_GREEN "\n";
static const char bar3[] = TEXTCOLOR_RED "\n\35\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36"
						  "\36\36\36\36\36\36\36\36\36\36\36\36\37" TEXTCOLOR_NORMAL "\n";
static const char logbar[] = "\n<------------------------------->\n";

void C_MidPrint (FFont *font, const char *msg)
{
	if (StatusBar == NULL || screen == NULL)
		return;

	if (msg != NULL)
	{
		AddToConsole (-1, bar1);
		AddToConsole (-1, msg);
		AddToConsole (-1, bar3);

		StatusBar->AttachMessage (new DHUDMessage (font, msg, 1.5f, 0.375f, 0, 0,
			(EColorRange)PrintColors[PRINTLEVELS], con_midtime), MAKE_ID('C','N','T','R'));
	}
	else
	{
		StatusBar->DetachMessage (MAKE_ID('C','N','T','R'));
	}
}

void C_MidPrintBold (FFont *font, const char *msg)
{
	if (StatusBar == NULL)
		return;

	if (msg)
	{
		AddToConsole (-1, bar2);
		AddToConsole (-1, msg);
		AddToConsole (-1, bar3);

		StatusBar->AttachMessage (new DHUDMessage (font, msg, 1.5f, 0.375f, 0, 0,
			(EColorRange)PrintColors[PRINTLEVELS+1], con_midtime), MAKE_ID('C','N','T','R'));
	}
	else
	{
		StatusBar->DetachMessage (MAKE_ID('C','N','T','R'));
	}
}

DEFINE_ACTION_FUNCTION(DObject, C_MidPrint)
{
	PARAM_PROLOGUE;
	PARAM_STRING(font);
	PARAM_STRING(text);
	PARAM_BOOL_DEF(bold);

	FFont *fnt = FFont::FindFont(font);
	const char *txt = text[0] == '$'? GStrings(&text[1]) : text.GetChars();
	if (!bold) C_MidPrint(fnt, txt);
	else C_MidPrintBold(fnt, txt);
	return 0;
}

/****** Tab completion code ******/

struct TabData
{
	int UseCount;
	FName TabName;

	TabData()
	: UseCount(0)
	{
	}

	TabData(const char *name)
	: UseCount(1), TabName(name)
	{
	}

	TabData(const TabData &other)
	: UseCount(other.UseCount), TabName(other.TabName)
	{
	}
};

static TArray<TabData> TabCommands (TArray<TabData>::NoInit);
static int TabPos;				// Last TabCommand tabbed to
static int TabStart;			// First char in CmdLine to use for tab completion
static int TabSize;				// Size of tab string

static bool FindTabCommand (const char *name, int *stoppos, int len)
{
	FName aname(name);
	unsigned int i;
	int cval = 1;

	for (i = 0; i < TabCommands.Size(); i++)
	{
		if (TabCommands[i].TabName == aname)
		{
			*stoppos = i;
			return true;
		}
		cval = strnicmp (TabCommands[i].TabName.GetChars(), name, len);
		if (cval >= 0)
			break;
	}

	*stoppos = i;

	return (cval == 0);
}

void C_AddTabCommand (const char *name)
{
	int pos;

	if (FindTabCommand (name, &pos, INT_MAX))
	{
		TabCommands[pos].UseCount++;
	}
	else
	{
		TabData tab(name);
		TabCommands.Insert (pos, tab);
	}
}

void C_RemoveTabCommand (const char *name)
{
	if (TabCommands.Size() == 0)
	{
		// There are no tab commands that can be removed.
		// This is important to skip construction of aname 
		// in case the NameManager has already been destroyed.
		return;
	}

	FName aname(name, true);

	if (aname == NAME_None)
	{
		return;
	}
	for (unsigned int i = 0; i < TabCommands.Size(); ++i)
	{
		if (TabCommands[i].TabName == aname)
		{
			if (--TabCommands[i].UseCount == 0)
			{
				TabCommands.Delete(i);
			}
			break;
		}
	}
}

void C_ClearTabCommands ()
{
	TabCommands.Clear();
}

static int FindDiffPoint (FName name1, const char *str2)
{
	const char *str1 = name1.GetChars();
	int i;

	for (i = 0; tolower(str1[i]) == tolower(str2[i]); i++)
		if (str1[i] == 0 || str2[i] == 0)
			break;

	return i;
}

static void C_TabComplete (bool goForward)
{
	unsigned i;
	int diffpoint;

	if (!TabbedLast)
	{
		bool cancomplete;

		// Skip any spaces at beginning of command line
		for (i = 0; i < CmdLine.Text.Len(); ++i)
		{
			if (CmdLine.Text[i] != ' ')
				break;
		}
		if (i == CmdLine.Text.Len())
		{ // Line was nothing but spaces
			return;
		}
		TabStart = i;

		TabSize = (int)CmdLine.Text.Len() - TabStart;

		if (!FindTabCommand(&CmdLine.Text[TabStart], &TabPos, TabSize))
		{
			// [RC] Hack to auto-complete RCON commands.
			if ( ( TabSize > 5 ) && ( strstr(&CmdLine.Text[TabStart], "rcon") != NULL ) )
			{
				TabStart += 5;
				TabSize -=5;

				if (!FindTabCommand (&CmdLine.Text[TabStart], &TabPos, TabSize))
					return;		// No matches even without "RCON".
			}
			else
				return;		// No initial matches
		}

		// Show a list of possible completions, if more than one.
		if (TabbedList || con_notablist)
		{
			cancomplete = true;
		}
		else
		{
			cancomplete = C_TabCompleteList ();
			TabbedList = true;
		}

		if (goForward)
		{ // Position just before the list of completions so that when TabPos
		  // gets advanced below, it will be at the first one.
			--TabPos;
		}
		else
		{ // Find the last matching tab, then go one past it.
			while (++TabPos < (int)TabCommands.Size())
			{
				if (FindDiffPoint(TabCommands[TabPos].TabName, &CmdLine.Text[TabStart]) < TabSize)
				{
					break;
				}
			}
		}
		TabbedLast = true;
		if (!cancomplete)
		{
			return;
		}
	}

	if ((goForward && ++TabPos == (int)TabCommands.Size()) ||
		(!goForward && --TabPos < 0))
	{
		TabbedLast = false;
		CmdLine.Text.Truncate(TabSize);
	}
	else
	{
		diffpoint = FindDiffPoint(TabCommands[TabPos].TabName, &CmdLine.Text[TabStart]);

		if (diffpoint < TabSize)
		{
			// No more matches
			TabbedLast = false;
			CmdLine.Text.Truncate(TabSize - TabStart);
		}
		else
		{
			CmdLine.Text.Truncate(TabStart);
			CmdLine.Text << TabCommands[TabPos].TabName << ' ';
		}
	}
	CmdLine.CursorPos = (unsigned)CmdLine.Text.Len();
	CmdLine.MakeStartPosGood();
}

static bool C_TabCompleteList ()
{
	int nummatches, i;
	size_t maxwidth;
	int commonsize = INT_MAX;

	nummatches = 0;
	maxwidth = 0;

	for (i = TabPos; i < (int)TabCommands.Size(); ++i)
	{
		if (FindDiffPoint (TabCommands[i].TabName, &CmdLine.Text[TabStart]) < TabSize)
		{
			break;
		}
		else
		{
			if (i > TabPos)
			{
				// This keeps track of the longest common prefix for all the possible
				// completions, so we can fill in part of the command for the user if
				// the longest common prefix is longer than what the user already typed.
				int diffpt = FindDiffPoint (TabCommands[i-1].TabName, TabCommands[i].TabName.GetChars());
				if (diffpt < commonsize)
				{
					commonsize = diffpt;
				}
			}
			nummatches++;
			maxwidth = MAX (maxwidth, strlen (TabCommands[i].TabName.GetChars()));
		}
	}
	if (nummatches > 1)
	{
		size_t x = 0;
		maxwidth += 3;
		Printf (TEXTCOLOR_BLUE "Completions for %s:\n", CmdLine.Text.GetChars());
		for (i = TabPos; nummatches > 0; ++i, --nummatches)
		{
			// [Dusk] Print console commands blue, CVars green, aliases red.
			const char* colorcode = "";
			FConsoleCommand* ccmd;
			if (FindCVar (TabCommands[i].TabName, NULL))
				colorcode = TEXTCOLOR_GREEN;
			else if ((ccmd = FConsoleCommand::FindByName (TabCommands[i].TabName)) != NULL)
			{
				if (ccmd->IsAlias())
					colorcode = TEXTCOLOR_RED;
				else
					colorcode = TEXTCOLOR_LIGHTBLUE;
			}

			Printf ("%s%-*s", colorcode, int(maxwidth), TabCommands[i].TabName.GetChars());
			x += maxwidth;
			if (x > ConCols / active_con_scale() - maxwidth)
			{
				x = 0;
				Printf ("\n");
			}
		}
		if (x != 0)
		{
			Printf ("\n");
		}
		// Fill in the longest common prefix, if it's longer than what was typed.
		if (TabSize != commonsize)
		{
			TabSize = commonsize;
			CmdLine.Text.Truncate(TabStart);
			CmdLine.Text.AppendCStrPart(TabCommands[TabPos].TabName.GetChars(), commonsize);
			CmdLine.CursorPos = (unsigned)CmdLine.Text.Len();
		}
		return false;
	}
	return true;
}

//
// [TP]
//
TArray<FString> C_GetTabCompletes (const FString& part)
{
	TArray<FString> result;

	for ( unsigned int i = 0; i < TabCommands.Size(); ++i )
	{
		if ( FindDiffPoint( TabCommands[i].TabName, part ) >= int( part.Len() ))
			result.Push( FString( TabCommands[i].TabName.GetChars() ));
	}

	return result;
}

//
// [TP] Begins capture mode
//
void C_StartCapture()
{
	g_IsCapturing = true;
	g_CaptureBuffer = "";
}

//
// [TP] Ends capture mode and returns the result
//
const char* C_EndCapture()
{
	g_IsCapturing = false;
	return g_CaptureBuffer;
}

//
// [TP] Are we currently capturing console output?
//
bool C_IsCapturing()
{
	return g_IsCapturing;
}
