library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.conv_std_logic_vector;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_textio.all;
use ieee.std_logic_unsigned.conv_integer;
 

---for debug....
use std.textio.all;

entity mos6502 is 
    generic (   dsize : integer := 8;
                asize : integer :=16
            );
    port (  input_clk   : in std_logic; --phi0 input pin.
            rdy         : in std_logic;
            rst_n       : in std_logic;
            irq_n       : in std_logic;
            nmi_n       : in std_logic;
            dbe         : in std_logic;
            r_nw        : out std_logic;
            phi1        : out std_logic;
            phi2        : out std_logic;
            addr        : out std_logic_vector ( asize - 1 downto 0);
            d_io        : inout std_logic_vector ( dsize - 1 downto 0)
    );
end mos6502;

architecture rtl of mos6502 is

signal set_clk : std_logic;
signal trigger_clk : std_logic;

signal pc : std_logic_vector (asize - 1 downto 0);
signal instruction : std_logic_vector (dsize - 1 downto 0);

-- SR Flags (bit 7 to bit 0):
--  7   N   ....    Negative
--  6   V   ....    Overflow
--  5   -   ....    ignored
--  4   B   ....    Break
--  3   D   ....    Decimal (use BCD for arithmetics)
--  2   I   ....    Interrupt (IRQ disable)
--  1   Z   ....    Zero
--  0   C   ....    Carry
constant st_N : integer := 7;
constant st_V : integer := 6;
constant st_B : integer := 4;
constant st_D : integer := 3;
constant st_I : integer := 2;
constant st_Z : integer := 1;
constant st_C : integer := 0;

signal status_reg : std_logic_vector (dsize - 1 downto 0);

signal dbus_buffer : std_logic_vector (dsize - 1 downto 0);
signal reg_x : std_logic_vector (dsize - 1 downto 0);
signal reg_y : std_logic_vector (dsize - 1 downto 0);
signal sp : std_logic_vector (dsize - 1 downto 0);
signal input_data_latch : std_logic_vector (dsize - 1 downto 0);

type dec_status is (reset0, reset1, reset2, reset3, reset4, reset5, 
                    fetch, decode, 
                    exec0, exec1, exec2, exec3, exec4, exec5, 
                    unknown_stat);

type addr_mode is ( ad_imp,
                    ad_imm,
                    ad_acc, 
                    ad_zp0, ad_zp1,
                    ad_zpx0, ad_zpx1,
                    ad_zpy0, ad_zpy1,
                    ad_abs0, ad_abs1, ad_abs2, 
                    ad_absx0, ad_absx1, ad_absx2, 
                    ad_absy0, ad_absy1, ad_absy2,
                    ad_indx_indir0, ad_indx_indir1, 
                        ad_indx_indir2, ad_indx_indir3, ad_indx_indir4, 
                    ad_indir_indx0, ad_indir_indx1, ad_indir_indx2, 
                        ad_indir_indx3, ad_indir_indx4,
                    ad_unknown);

signal cur_status : dec_status;
signal cur_mode : addr_mode;

procedure d_print(msg : string) is
variable out_l : line;
begin
    write(out_l, msg);
    writeline(output, out_l);
end  procedure;

procedure d_print(msg : string; sig : std_logic_vector) is
variable out_l : line;
begin
    write(out_l, msg);
    write(out_l, sig);
    writeline(output, out_l);
end  procedure;

procedure d_print(msg : string; ival : integer) is
variable out_l : line;
begin
    write(out_l, msg);
    write(out_l, ival);
    writeline(output, out_l);
end  procedure;

---ival : 0x0000 - 0xffff
function conv_hex16(ival : integer) return string is
variable tmp1, tmp2, tmp3, tmp4 : integer;
--variable ret : string (1 to 4) := "0000";
variable hex_chr: string (1 to 16) := "0123456789abcdef";
begin
    tmp4 := ival / 16 ** 3;
    tmp3 := (ival mod 16 ** 3) / 16 ** 2;
    tmp2 := (ival mod 16 ** 2) / 16 ** 1;
    tmp1 := ival mod 16 ** 1;
--    d_print("hex_chr");
--    d_print("ival: ", ival);
--    d_print("tmp4: ", tmp4);
--    d_print("tmp3: ", tmp3);
--    d_print("tmp2: ", tmp2);
--    d_print("tmp1: ", tmp1);

    return hex_chr(tmp4 + 1) & hex_chr(tmp3 + 1) 
        & hex_chr(tmp2 + 1) & hex_chr(tmp1 + 1);
end;

function conv_hex8(ival : integer) return string is
variable tmp1, tmp2 : integer;
variable hex_chr: string (1 to 16) := "0123456789abcdef";
begin
    tmp2 := (ival mod 16 ** 2) / 16 ** 1;
    tmp1 := ival mod 16 ** 1;
    return hex_chr(tmp2 + 1) & hex_chr(tmp1 + 1);
end;

begin

    -- clock generate.
    phi1 <= input_clk;
    phi2 <= not input_clk;
    set_clk <= input_clk;
    trigger_clk <= not input_clk;
    dbus_buffer <= d_io;

    --addr(asize - 1 downto dsize) <= internal_abus_h;
    --addr(dsize - 1 downto 0) <= internal_abus_l;

    main_p : process (rst_n, set_clk, trigger_clk)
    variable single_inst : boolean;
    begin
        if (rst_n'event and rst_n = '0') then
            d_print("reset negated.");
            cur_status <= reset0;
            status_reg <= (others => '0');
            pc <= conv_std_logic_vector(16#8000#, asize);
        end if;

        ---micro code part.
        if (set_clk'event and set_clk = '1') then
            d_print("-");

            case cur_status is
                when reset0 => 
                    cur_status <= reset1;
                when reset1 => 
                    cur_status <= reset2;
                when reset2 => 
                    cur_status <= reset3;
                when reset3 => 
                    cur_status <= reset4;
                when reset4 => 
                    cur_status <= reset5;
                when reset5 => 
                    cur_status <= fetch;
                when fetch => 
                    d_print("fetch " & conv_hex16(conv_integer(pc)));
                    addr <= pc;
                    r_nw <= '1';
                    pc <= pc + 1;
                    cur_status <= decode;
                when unknown_stat => 
                    assert false 
                        report ("unknow status") severity failure;
                when others => 
                    null;
            end case;

            if cur_status = decode then
                d_print("decode inst: " & conv_hex8(conv_integer(instruction)));

                ---single byte instruction.
                single_inst := false;

                if instruction = conv_std_logic_vector(16#8a#, dsize) then
                    single_inst := true;
                    d_print("txa");
                elsif instruction = conv_std_logic_vector(16#9a#, dsize) then
                    single_inst := true;
                    d_print("txs");
                    sp <= reg_x;
                elsif instruction = conv_std_logic_vector(16#aa#, dsize) then
                    single_inst := true;
                    d_print("tax");
                elsif instruction = conv_std_logic_vector(16#ba#, dsize) then
                    single_inst := true;
                    d_print("tsx");
                elsif instruction = conv_std_logic_vector(16#ca#, dsize) then
                    single_inst := true;
                    d_print("dex");
                elsif instruction = conv_std_logic_vector(16#ea#, dsize) then
                    single_inst := true;
                    d_print("nop");
                elsif instruction = conv_std_logic_vector(16#08#, dsize) then
                    single_inst := true;
                    d_print("php");
                elsif instruction = conv_std_logic_vector(16#28#, dsize) then
                    single_inst := true;
                    d_print("plp");
                elsif instruction = conv_std_logic_vector(16#48#, dsize) then
                    single_inst := true;
                    d_print("pha");
                elsif instruction = conv_std_logic_vector(16#68#, dsize) then
                    single_inst := true;
                    d_print("pla");
                elsif instruction = conv_std_logic_vector(16#88#, dsize) then
                    single_inst := true;
                    d_print("dey");
                elsif instruction = conv_std_logic_vector(16#a8#, dsize) then
                    single_inst := true;
                    d_print("tay");
                elsif instruction = conv_std_logic_vector(16#c8#, dsize) then
                    single_inst := true;
                    d_print("iny");
                elsif instruction = conv_std_logic_vector(16#e8#, dsize) then
                    single_inst := true;
                    d_print("inx");
                elsif instruction = conv_std_logic_vector(16#18#, dsize) then
                    single_inst := true;
                    d_print("clc");
                elsif instruction = conv_std_logic_vector(16#38#, dsize) then
                    single_inst := true;
                    d_print("sec");
                elsif instruction = conv_std_logic_vector(16#58#, dsize) then
                    single_inst := true;
                    d_print("cli");
                elsif instruction = conv_std_logic_vector(16#78#, dsize) then
                    single_inst := true;
                    d_print("sei");
                    status_reg(st_I) <= '1';
                elsif instruction = conv_std_logic_vector(16#98#, dsize) then
                    single_inst := true;
                    d_print("tya");
                elsif instruction = conv_std_logic_vector(16#b8#, dsize) then
                    single_inst := true;
                    d_print("clv");
                elsif instruction = conv_std_logic_vector(16#d8#, dsize) then
                    single_inst := true;
                    d_print("cld");
                elsif instruction = conv_std_logic_vector(16#f8#, dsize) then
                    single_inst := true;
                    d_print("sed");
                end if;

                if single_inst then
                    cur_status <= fetch;
                    cur_mode <= ad_imp;
                    addr <= (others => 'Z');
                else

                    ---instruction consists of aaabbbcc form.
                    if instruction (1 downto 0) = "01" then
                        --d_print("cc=01");

                        ---bbb part format
                        if instruction (4 downto 2) = "000" or 
                            instruction (4 downto 2) = "001" or 
                            instruction (4 downto 2) = "010" or 
                            instruction (4 downto 2) = "011" or 
                            instruction (4 downto 2) = "100" or 
                            instruction (4 downto 2) = "101" or 
                            instruction (4 downto 2) = "110" or 
                            instruction (4 downto 2) = "111" then

                            if instruction (7 downto 5) = "000" then
                                d_print("ora");
                            elsif instruction (7 downto 5) = "001" then
                                d_print("and");
                            elsif instruction (7 downto 5) = "010" then
                                d_print("eor");
                            elsif instruction (7 downto 5) = "011" then
                                d_print("adc");
                            elsif instruction (7 downto 5) = "100" then
                                d_print("sta");
                            elsif instruction (7 downto 5) = "101" then
                                d_print("lda");
                            elsif instruction (7 downto 5) = "110" then
                                d_print("cmp");
                            elsif instruction (7 downto 5) = "111" then
                                d_print("sbc");
                            else
                                assert false 
                                    report ("unknow instruction") severity failure;
                            end if;
                        end if;
                    elsif instruction (1 downto 0) = "10" then
                        --d_print("cc=10");

                        if instruction (4 downto 2) = "000" then
                            cur_mode <= ad_imm;
                            d_print("immediate");
                            addr <= pc;
                            pc <= pc + 1;
                            cur_status <= exec0;
                        elsif instruction (4 downto 2) = "001" then
                            cur_mode <= ad_zp0;
                        elsif instruction (4 downto 2) = "010" then
                            cur_mode <= ad_acc;
                        elsif instruction (4 downto 2) = "011" then
                            cur_mode <= ad_abs0;
                        elsif instruction (4 downto 2) = "101" then
                            cur_mode <= ad_zpx0;
                        elsif instruction (4 downto 2) = "111" then
                            cur_mode <= ad_absx0;
                        else
                            cur_mode <= ad_unknown;
                            assert false 
                                report ("unknow addr mode") severity failure;
                        end if;

                        if instruction (7 downto 5) = "000" then
                            d_print("asl");
                        elsif instruction (7 downto 5) = "001" then
                            d_print("rol");
                        elsif instruction (7 downto 5) = "010" then
                            d_print("lsr");
                        elsif instruction (7 downto 5) = "011" then
                            d_print("ror");
                        elsif instruction (7 downto 5) = "100" then
                            d_print("stx");
                        elsif instruction (7 downto 5) = "101" then
                            d_print("ldx");
                        elsif instruction (7 downto 5) = "110" then
                            d_print("dec");
                        elsif instruction (7 downto 5) = "111" then
                            d_print("inc");
                        else
                            assert false 
                                report ("unknow instruction") severity failure;
                        end if;

                    elsif instruction (1 downto 0) = "00" then
                        --d_print("cc=00 group...");

                        if instruction (4 downto 0) = "10000" then
                            ---conditional branch instruction..
                        elsif instruction = conv_std_logic_vector(16#00#, dsize) then
                            d_print("brk");
                        elsif instruction = conv_std_logic_vector(16#20#, dsize) then
                            d_print("jsr abs");
                        elsif instruction = conv_std_logic_vector(16#40#, dsize) then
                            d_print("40");
                        elsif instruction = conv_std_logic_vector(16#60#, dsize) then
                            d_print("60");
                        
                            
                        ---bbb part format
                        else
                            if instruction (4 downto 2) = "000" then
                                cur_mode <= ad_imm;
                                d_print("immediate");
                                addr <= pc;
                                pc <= pc + 1;
                                cur_status <= exec0;
                            elsif instruction (4 downto 2) = "001" then
                                cur_mode <= ad_zp0;
                            elsif instruction (4 downto 2) = "011" then
                                cur_mode <= ad_abs0;
                                d_print("abs");
                                addr <= pc;
                                pc <= pc + 1;
                                cur_status <= exec0;
                            elsif instruction (4 downto 2) = "101" then
                                cur_mode <= ad_zpx0;
                            elsif instruction (4 downto 2) = "111" then
                                cur_mode <= ad_absx0;
                            else
                                cur_mode <= ad_unknown;
                                assert false 
                                    report ("unknow addr mode") severity failure;
                            end if;

                            if instruction (7 downto 5) = "001" then
                                d_print("bit");
                            elsif instruction (7 downto 5) = "010" then
                                d_print("jmp");
                            elsif instruction (7 downto 5) = "011" then
                                d_print("jmp (abs)");
                            elsif instruction (7 downto 5) = "100" then
                                d_print("sty");
                            elsif instruction (7 downto 5) = "101" then
                                d_print("ldy");
                            elsif instruction (7 downto 5) = "110" then
                                d_print("cpy");
                            elsif instruction (7 downto 5) = "111" then
                                d_print("cpx");
                            else
                                assert false 
                                    report ("unknow instruction") severity failure;
                            end if;
                        end if; --if instruction (4 downto 0) = "10000"
                    end if; --if instruction (1 downto 0) = "01"

                end if; --if single_inst

            elsif cur_status = exec0 then
                if instruction (1 downto 0) = "00" then
                    if instruction (4 downto 2) = "011" then
                    --abs0
                        addr <= pc;
                        pc <= pc + 1;
                        input_data_latch <= dbus_buffer;
                        cur_status <= exec1;
                    end if;
                end if;
            end if; --if cur_status = decode 
        end if; --if (set_clk'event and set_clk = '1') 

        if (trigger_clk'event and trigger_clk = '1') then
            --d_print("_");
            case cur_status is
                when decode => 
                    instruction <= d_io;
                when exec0 => 
                    if instruction (1 downto 0) = "01" then
                    elsif instruction (1 downto 0) = "10" then
                        if instruction (4 downto 2) = "000" then
                        --immediate
                            if instruction (7 downto 5) = "101" then
                            --ldx
                                reg_x <= dbus_buffer;
                                cur_status <= fetch;
                            end if;
                        end if;
                    end if; --if instruction (1 downto 0) = "01"
                when exec1 => 
                    if instruction (1 downto 0) = "00" then
                        if instruction (4 downto 2) = "011" then
                        --abs
                            if instruction (7 downto 5) = "010" then
                            --jmp
                                pc <= dbus_buffer & input_data_latch;
                                addr <= dbus_buffer & input_data_latch;
                                cur_status <= fetch;
                            end if;
                        end if;
                    end if;
                when others => 
                    null;
            end case; --case cur_status
        end if;
    end process;

end rtl;

