#pragma once

#include <math.h>
#include <algorithm>
#include "buffer2d.h"
#include "arrayutil.h"

#include "ILineDrawer.h"

namespace gl
{

template <typename NumericT, size_t slopeCorrectionTableSize = 256>
class LineDrawer8_Shift : public IBufferLineDrawer<NumericT, uint8_t>
{
private:
	Buffer2D<uint8_t>*	pBuff_;
	
	const NumericT* slopeCorrectionTable_;
	
	inline
	void blend(
		uint8_t* p,
		NumericT v
		)
	{
		uint8_t old = *p;
		if (old != 0xff) {
			NumericT nv = NumericT(old) + v;
			*p = (nv >= NumericT(255.0)) ? 255 : halfAdjust(nv);
		}
	}

public:

	LineDrawer8_Shift()
		:
		slopeCorrectionTable_(NULL),
		pBuff_(NULL)
	{
	}
	
	void SetSlopeCorrectionTable(const NumericT* slopeCorrectionTable)		{	slopeCorrectionTable_ = slopeCorrectionTable;	}
	
	void SetBuffer(Buffer2D<uint8_t>* pBuff)
	{
		pBuff_ = pBuff;
	}
	
	void shift(const uint8_t* src, uint8_t* dst, size_t len, float offset)
	{
		assert(offset >= 0.0f);
		float inversed = 1.0f - offset;
		uint8_t part;
		uint8_t remain = 0;
		for (size_t i=0; i<len; ++i) {
			part = src[i] * inversed;
			dst[i] = part + remain;
			remain = src[i] - part;
		}
		dst[len] = remain;
	}
	
	virtual
	void DrawVerticalLine(uint8_t col, NumericT x, NumericT y1, NumericT y2)
	{
	}

	void DrawLine(uint8_t intensity, NumericT x1, NumericT y1, NumericT x2, NumericT y2)
	{
		NumericT width = abs(x2 - x1);
		NumericT height = abs(y2 - y1);
		if (width + height == NumericT(0))
			return;
		assert(2 <= slopeCorrectionTableSize);
		const int lineOffset = pBuff_->GetLineOffset();
		
		NumericT alpha = NumericT(intensity);
		
//		if (height < width) {
		if (1) {
			if (x2 < x1) { // left;
				std::swap(x2, x1);
				std::swap(y2, y1);
			}
			const NumericT dy = y2 - y1;
			const NumericT gradient = dy / width;
			float rotation = abs(width / dy);
			
			{
				float r2 = width < height ? (width/height) : (height/width);
				alpha *= slopeCorrectionTable_[ ToInt(mul<slopeCorrectionTableSize - 1>(r2)) ];
			}
			uint8_t pat1[512] = {0};
			uint8_t pat2[512+1];
			
			float patLen = rotation * 1.4;
			patLen = std::min(256.0f, patLen);
			patLen = std::max(1.50001f, patLen);
			int iPatLen = ceil(patLen);
			for (size_t i=0; i<iPatLen; ++i) {
				float x = (i * 2) / patLen;
				float v = exp(- x * x);
				pat1[iPatLen-i] = pat1[iPatLen+i] = (v * 255 + 0.5);
			}
			
			uint8_t* ptr = (uint8_t*)pBuff_->GetPixelPtr(ToInt(x1)-2, ToInt(y1));
			
			const float offset = rotation;
			float curOffset = 0;
			float ylen = height;
			for (size_t y=0; y<ylen; ++y) {
				int io = (int) curOffset;
				bool isMinus = curOffset < 0.0f;
				float fracOffset = curOffset - io;
				if (isMinus) {
					fracOffset += 1.0f;
					--io;
				}
				const size_t w = iPatLen * 2;
				shift(pat1, pat2, w, fracOffset);
				for (size_t x=0; x<w; ++x) {
					ptr[io+x] = pat2[x] * (alpha / 255.0);
				}
				OffsetPtr(ptr, lineOffset);
				curOffset += offset;
			}



		}
	}

};

} // namespace gl

