﻿using System;
using System.Xml.Serialization;
using System.Collections.ObjectModel;
using System.IO;
using Windows.ApplicationModel.DataTransfer;
using Windows.Storage;
using Windows.UI.ViewManagement;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using System.Threading.Tasks;

// ユーザー コントロールのアイテム テンプレートについては、http://go.microsoft.com/fwlink/?LinkId=234236 を参照してください

namespace FooEditor
{
    public sealed partial class DocumentControl : UserControl, IXmlSerializable,IDisposable
    {
        DispatcherTimer timer = new DispatcherTimer();
        public DocumentControl()
        {
            this.InitializeComponent();
            
            this.timer.Interval = new TimeSpan(0, 0, 0, 1);
            this.timer.Tick += timer_Tick;
            this.timer.Start();

            AppSettings setting = AppSettings.Current;
            setting.ChangedSetting += setting_ChangedSetting;

            InputPane currentView = InputPane.GetForCurrentView();
            currentView.Showing += currentView_Showing;
            currentView.Hiding += currentView_Hiding;
        }

        public void Dispose()
        {
            AppSettings setting = AppSettings.Current;
            setting.ChangedSetting -= setting_ChangedSetting;

            InputPane currentView = InputPane.GetForCurrentView();
            currentView.Showing -= currentView_Showing;
            currentView.Hiding -= currentView_Hiding;
        }

        public string Title
        {
            get;
            set;
        }

        public FileType DocumentType
        {
            get;
            private set;
        }

        void setting_ChangedSetting(object sender, EventArgs e)
        {
            this.ApplySetting((AppSettings)sender);
        }

        public void GetData(DataRequestedEventArgs args)
        {
            if (this.Textbox.SelectionLength != 0)
            {
                var loader = new Windows.ApplicationModel.Resources.ResourceLoader();
                args.Request.Data.Properties.Title = loader.GetString("SharedDataTitle");
                args.Request.Data.SetText(this.Textbox.SelectedText);
            }
        }

        void currentView_Hiding(InputPane sender, InputPaneVisibilityEventArgs args)
        {
            this.Textbox.Margin = new Thickness(0);
            args.EnsuredFocusedElementInView = true;
        }

        void currentView_Showing(InputPane sender, InputPaneVisibilityEventArgs args)
        {
            this.Textbox.Margin = new Thickness(0, 0, 0, args.OccludedRect.Height);
            args.EnsuredFocusedElementInView = true;
        }

        void timer_Tick(object sender, object e)
        {
            if (this.Textbox.LayoutLineCollection.GenerateFolding())
                this.Textbox.Refresh();
        }

        public void ApplySetting(AppSettings setting)
        {
            FlowDirection flowdir = setting.IsRTL ? FlowDirection.RightToLeft : FlowDirection.LeftToRight;
            if (this.Textbox.FlowDirection != flowdir)
                this.Textbox.FlowDirection = flowdir;
            if (this.Textbox.DrawCaretLine != setting.ShowLineMarker)
                this.Textbox.DrawCaretLine = setting.ShowLineMarker;
            if (this.Textbox.DrawLineNumber != setting.ShowLineNumber)
                this.Textbox.DrawLineNumber = setting.ShowLineNumber;
            if (this.Textbox.DrawRuler != setting.ShowRuler)
                this.Textbox.DrawRuler = setting.ShowRuler;
            if (this.Textbox.TabChars != setting.TabChar)
                this.Textbox.TabChars = setting.TabChar;
            if (this.Textbox.FontSize != setting.FontSize)
                this.Textbox.FontSize = setting.FontSize;
            if (this.Textbox.FontFamily.Source != setting.FontFamily)
                this.Textbox.FontFamily = new Windows.UI.Xaml.Media.FontFamily(setting.FontFamily);
            if (this.Textbox.ShowFullSpace != setting.ShowFullSpace)
                this.Textbox.ShowFullSpace = setting.ShowFullSpace;
            if (this.Textbox.ShowTab != setting.ShowTab)
                this.Textbox.ShowTab = setting.ShowTab;
            bool rebuildLayout = false;
            if (this.Textbox.LineBreakMethod != setting.LineBreakMethod)
            {
                this.Textbox.LineBreakMethod = setting.LineBreakMethod;
                rebuildLayout = true;
            }
            if (this.Textbox.LineBreakCharCount != setting.LineBreakCount)
            {
                this.Textbox.LineBreakCharCount = setting.LineBreakCount;
                rebuildLayout = true;
            }
            if (rebuildLayout)
                this.Textbox.PerfomLayouts();
            this.Textbox.Refresh();
        }

        public async Task SetDocumentType(FileType type)
        {
            this.DocumentType = type;

            if (type == null || string.IsNullOrEmpty(type.DocumentType))
            {
                this.Textbox.Hilighter = null;
                this.Textbox.FoldingStrategy = null;
                this.Textbox.LayoutLineCollection.ClearHilight();
                this.Textbox.LayoutLineCollection.ClearFolding();
                return;
            }

            StorageFile file;
            try
            {
                var uri = new System.Uri("ms-appx:///Keywords/" + type.DocumentType);
                file = await Windows.Storage.StorageFile.GetFileFromApplicationUriAsync(uri);
            }
            catch (FileNotFoundException)
            {
                this.Textbox.Hilighter = null;
                this.Textbox.FoldingStrategy = null;
                this.Textbox.LayoutLineCollection.ClearHilight();
                this.Textbox.LayoutLineCollection.ClearFolding();
                return;
            }

            SyntaxDefnition SynataxDefnition = new SyntaxDefnition();
            SynataxDefnition.generateKeywordList(file.Path);

            if (SynataxDefnition.Hilighter == FooEditor.SyntaxDefnition.XmlHilighter)
            {
                this.Textbox.Hilighter = new XmlHilighter();
            }
            else
            {
                GenericHilighter Hilighter = new GenericHilighter();
                Hilighter.KeywordManager = SynataxDefnition;
                this.Textbox.Hilighter = Hilighter;
            }

            if (!string.IsNullOrEmpty(SynataxDefnition.FoldingBegin) && !string.IsNullOrEmpty(SynataxDefnition.FoldingEnd))
            {
                if (SynataxDefnition.FoldingMethod == FooEditor.SyntaxDefnition.CLangFolding)
                    this.Textbox.FoldingStrategy = new CLangFoldingGenerator(SynataxDefnition.FoldingBegin, SynataxDefnition.FoldingEnd, '{', '}');
                else
                    this.Textbox.FoldingStrategy = new RegexFoldingGenerator(SynataxDefnition.FoldingBegin, SynataxDefnition.FoldingEnd);
            }
            else
            {
                this.Textbox.FoldingStrategy = null;
            }

            this.Textbox.LayoutLineCollection.HilightAll();
            this.Textbox.LayoutLineCollection.ClearFolding();
            this.Textbox.LayoutLineCollection.GenerateFolding();
            this.Textbox.Refresh();
        }
        
        public async Task LoadFile(StorageFile file)
        {
            await this.SetDocumentType(GetFileType(file.Name));
            using (Stream stream = await file.OpenStreamForReadAsync())
            using (StreamReader reader = new StreamReader(stream))
            {
                await this.Textbox.Document.LoadAsync(reader);
                this.Textbox.Refresh();
            }
        }

        public Popup CreatePopup(Type t)
        {
            if (t == typeof(FindFlyout))
            {
                var flyout = new FindFlyout(this.Textbox);
                return FlyoutUtils.CreateFlyoutUnderTopAppBar(flyout);
            }
            if (t == typeof(GoToFlyout))
            {
                var flyout = new GoToFlyout(this.Textbox);
                return FlyoutUtils.CreateFlyoutUnderTopAppBar(flyout);
            }
            throw new ArgumentOutOfRangeException();
        }

        private FileType GetFileType(string file)
        {
            ObservableCollection<FileType> collection = AppSettings.Current.FileTypeCollection;
            foreach (FileType type in collection)
                foreach (string ext in type.ExtensionCollection)
                    if (Path.GetExtension(file) == ext)
                        return type;
            return null;
        }

        public async Task SaveFile(StorageFile file)
        {
            using (Stream stream = await file.OpenStreamForWriteAsync())
            {
                stream.SetLength(0);    //上書き時にゴミが残らないようにする
                using (StreamWriter writer = new StreamWriter(stream))
                {
                    await this.Textbox.Document.SaveAsync(writer);
                }
            }
        }

        public void Undo()
        {
            this.Textbox.Document.UndoManager.undo();
            this.Textbox.Refresh();
        }

        public void Redo()
        {
            this.Textbox.Document.UndoManager.redo();
            this.Textbox.Refresh();
        }

        public System.Xml.Schema.XmlSchema GetSchema()
        {
            return null;
        }

        const string prefixFileName = "save_";

        public async void ReadXml(System.Xml.XmlReader reader)
        {
            reader.ReadStartElement("DocumentControl");

            reader.ReadStartElement("Title");
            this.Title = reader.ReadContentAsString();
            reader.ReadEndElement();

            reader.ReadStartElement("CaretPostionRow");
            int row = reader.ReadContentAsInt();
            reader.ReadEndElement();

            reader.ReadStartElement("CaretPostionCol");
            int col = reader.ReadContentAsInt();
            reader.ReadEndElement();

            reader.ReadStartElement("DocumentType");
            string documentType = reader.ReadContentAsString();
            reader.ReadEndElement();

            if (documentType != string.Empty)
            {
                foreach (FileType type in AppSettings.Current.FileTypeCollection)
                    if (type.DocumentType == documentType)
                    {
                        await this.SetDocumentType(type);
                        break;
                    }
            }

            StorageFile file = await ApplicationData.Current.LocalFolder.GetFileAsync(prefixFileName + this.Title);
            await this.LoadFile(file);
            this.Textbox.JumpCaret(row, col);
            await file.DeleteAsync();

        }

        public async void WriteXml(System.Xml.XmlWriter writer)
        {
            writer.WriteStartElement("Title");
            writer.WriteValue(this.Title);
            writer.WriteEndElement();

            writer.WriteStartElement("CaretPostionRow");
            writer.WriteValue(this.Textbox.CaretPostion.row);
            writer.WriteEndElement();

            writer.WriteStartElement("CaretPostionCol");
            writer.WriteValue(this.Textbox.CaretPostion.col);
            writer.WriteEndElement();

            writer.WriteStartElement("DocumentType");
            writer.WriteValue(this.DocumentType == null ? string.Empty : this.DocumentType.DocumentType);
            writer.WriteEndElement();

            StorageFile file = await ApplicationData.Current.LocalFolder.CreateFileAsync(prefixFileName + this.Title, CreationCollisionOption.ReplaceExisting);
            using (Stream stream = await file.OpenStreamForWriteAsync())
            using (StreamWriter sw = new StreamWriter(stream))
            {
                await this.Textbox.Document.SaveAsync(sw);
            }
        }
    }
}
