﻿/*@@@*
The MIT License (MIT)

Copyright (c) 2013 GMA0BN

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
**@@@*/
#pragma once

#define USE_BARRIER     1       /* 1: Memory barrierを明示的に行う。 */
#define USE_CANONNAME   1       /* 1: Canonical nameを取得する。 */
#define USE_NUMERICHOST 0       /* 1: Numeric host addressで取得する(DNSを呼ばない)。 */
#define VERBOSE         1       /* 1: addrinfo構造体の内容をダンプする。 */
#define THREAD_STKSZ    1024

#if USE_BARRIER
#include <intrin.h>
#pragma intrinsic(_WriteBarrier, _ReadBarrier, _ReadWriteBarrier)   /* 効果あるのか? */
#endif

#if !USE_BARRIER
typedef struct tagSessionRes {
    volatile SOCKET socket;
    HANDLE hThread;
} SessionRes;
#else
typedef struct tagSessionRes {
    SOCKET socket;
    HANDLE hThread;
} SessionRes;
#endif
/*= SessionRes構造体は、本モジュールを呼び出すmain threadと、
 *  セッションを実現するsub threadの両方からaccessされるため、
 *  memory barrierを考慮する必要がある。
 *  これを次のように判断する。
 *   1. Data passage from main thread to sub thread:
 *      Win32 APIのしくみを使ってパラメータとして渡すので、
 *      WIN32 APIが適切に解決する。
 *      よってmemory barrier不要(NOT necessary)。
 *   2. The field "socket":
 *      Sub threadによって書き換えられ、その後main threadに参照される。
 *      (スレッド終了時にソケットが閉じられ、その印として
 *       socketフィールドにINVALID_SOCKETが書かれる。)
 *      よって、sub threadが書いた直後に_WriteBarrier()、
 *      それをmain threadが参照する直前に_ReadBarrier()をそれぞれ要する(necessary)。
 *   3. The field "hThread":
 *      Main threadのみによって書き換えられ、かつ、1.の通り、
 *      sub threadに参照される際のmemory barrierは暗黙のうちに解決されることから、
 *      結局、明示的なmemory barrierは一切必要無い(NOT necessary)。
 */

/// セッションのリソースを初期化します。
extern void init_session_res(void* pRes, size_t size, size_t cnt);

/// セッションのリソースを開放します。
extern BOOL release_session_res(void* pRes, size_t size, size_t cnt, BOOL bFull);

/// 空いているCBを取得します。
extern int get_new_session_cb(void* pRes, size_t size, size_t cnt, size_t sta);

/// アドレフファミリをデコードします。
extern BOOL decode_addr_family(FILE* fp, int family);

/// addrinfo構造体の中身をダンプします。
extern BOOL dump_addrinfo(FILE* fp, struct addrinfo* p, const char* indent);

/// sockaddr構造体の中身をダンプします。
extern BOOL dump_sockaddr(FILE* fp, struct sockaddr* p, size_t addrlen, const char* indent);
