//
//  data.c
//  chncpu
//
//  Created by 西田　耀 on 2014/07/08.
//  Copyright (c) 2014年 CHNOSProject. All rights reserved.
//

#include "chncpu.h"

CHNCPU_Data *CHNCPU_DATA_CreateEmptyData(void)
{
	CHNCPU_Data *data;
	
	data = malloc(sizeof(CHNCPU_Data));
	if(!data){
		return NULL;
	}
	
	data->type = CHNCPU_PType_Undefined;
	data->count = 0;
	data->rawData = NULL;
	data->rawDataByteSize = 0;
	data->filePath = NULL;
	
	return data;
}

void CHNCPU_DATA_FreeData(CHNCPU_Data *data)
{
	if(data){
		if(data->rawData){
			free(data->rawData);
		}
		free(data);
	}
}

int CHNCPU_DATA_AllocateRawDataMemory(CHNCPU_Data *data, ch4_sint dataType, ch4_uint dataCount, unsigned int *errFlags)
{
	// 生データのメモリを割り当てる。
	if(!data){
		if(errFlags){
			*errFlags |= CHNCPU_ERR_INTERNAL;
		}
        return -1;
	}
	
	data->rawDataByteSize = CHNCPU_DATA_CalculateByteSizeOfRawData(dataType, dataCount);
	if(data->rawDataByteSize <= 0){
		if(errFlags){
			*errFlags |= CHNCPU_ERR_INVALID_PTYPE;
		}
		return -1;
	}
	if(data->rawData){
		puts("CHNCPU_Op_DATA_AllocateRawDataMemory: Double allocation. abort.");
		exit(EXIT_FAILURE);
	}
	data->rawData = malloc(data->rawDataByteSize);
	if(!data->rawData){
		data->rawDataByteSize = 0;
		return -1;
	}
	
	data->type = dataType;
	data->count = dataCount;
	
	return 0;
}

CHNCPU_Data *CHNCPU_DATA_CreateDataFromFileAsPType(const char dataPath[], ch4_sint pType)
{
	// 現状ではUINT8のみ
	FILE *fp;
	CHNCPU_Data *data;
	int i;
	
	data = CHNCPU_DATA_CreateEmptyData();
	if(!data){
		return NULL;
	}
	
	fp = fopen(dataPath, "rb");
	if(!fp){
		CHNCPU_DATA_FreeData(data);
		return NULL;
	}
	fseek(fp, 0, SEEK_END);
	data->rawDataByteSize = (int)ftell(fp);
	fseek(fp, 0, SEEK_SET);
	
	// 現状はUINT8前提なのでopCache->rawDataByteSize==dataCount
	if(CHNCPU_DATA_AllocateRawDataMemory(data, pType, data->rawDataByteSize, NULL)){
		CHNCPU_DATA_FreeData(data);
		return NULL;
	}
	
	// データの中身をファイルからコピー
	if(data->type == CHNCPU_PType_UINT8){
	 	for(i = 0; i < data->count; i++){
			((unsigned char *)data->rawData)[i] = (unsigned char)fgetc(fp);
		}
	} else{
		fclose(fp);
		CHNCPU_DATA_FreeData(data);
        return NULL;
	}
	fclose(fp);
	
	data->filePath = dataPath;
	
	return data;
}

int CHNCPU_DATA_AssignDataTagToPReg(CHNCPU_RuntimeEnvironment *env, ch4_uint p, CHNCPU_Data *data)
{
	if(!data || p >= CHNCPU_NUMBER_OF_PREG){
        return -1;
	}
	
	// データタグを指定されたPRegに格納
	env->pReg[p].type = data->type;
	env->pReg[p].mindex = CHNCPU_MemoryIndex_INVALID;
	env->pReg[p].pindex = 0;
	env->pReg[p].data = data;

	return 0;
}


int CHNCPU_DATA_WriteRawDataToFile(CHNCPU_Data *data, const char dataPath[], int dataCount, int dataType)
{
	// retv: written bytes
	FILE *fp;
	int i = 0;
	
	if(!data || !dataPath){
        return -1;
	}
	if(dataType == CHNCPU_PType_Undefined || data->type != dataType){
		// タイプが一致しなければバインドしない
		return -1;
	}
	if(dataCount > data->count){
		// ポインタに格納されているデータの個数を超えてファイルを書き込むことはしない
		return -1;
	}
	fp = fopen(dataPath, "wb");
	if(!fp){
		return -1;
	}
	if(data->type == CHNCPU_PType_UINT8){
	 	for(i = 0; i < data->rawDataByteSize; i++){
			fputc(((unsigned char *)data->rawData)[i], fp);
		}
	} else{
		i = -1;
	}
	fclose(fp);
	
	return i;
}

int CHNCPU_DATA_ReadAtIndex(CHNCPU_Data *data, int index, unsigned int *errFlags)
{
	int retv;
	
	if(!data || index < 0 || index >= data->count){
		if(errFlags){
			*errFlags |= CHNCPU_ERR_BAD_ACCESS;
		}
        return 0;
	}
	
	if(data->type == CHNCPU_PType_UINT8){
	 	retv = ((unsigned char *)data->rawData)[index];
	} else{
		if(errFlags){
			*errFlags |= CHNCPU_ERR_INVALID_PTYPE;
		}
        return 0;
	}
	
	return retv;
}

int CHNCPU_DATA_CalculateByteSizeOfRawData(ch4_sint dataType, ch4_uint dataCount)
{
	if(dataType == CHNCPU_PType_UINT8){
		return dataCount;
	}
	return 0;
}

int CHNCPU_DATA_PrintData(CHNCPU_Data *data, FILE *file)
{
	int i = 0;
	
	if(data){
		fputc('{', file);
		if(data->type == CHNCPU_PType_UINT8){
			for(i = 0; i < data->count; i++){
				if((i & 7) == 0){
					fprintf(file, "\n\t");
				}
				fprintf(file, "%d, ", ((unsigned char *)data->rawData)[i]);
			}
		}
		if(((i - 1) & 7) != 0){
			fputc('\n', file);
		}
		fputc('}', file);
		fputc('\n', file);
	}
	
    return 0;
}
