class NaviVoice < ActiveRecord::Base
  has_many :ratings
  has_many :dl_log_entries

  default_scope -> { where(disabled: false).order('id desc') }

  has_attached_file :archive,
    url: "/system/:class/:attachment/:id_partition/:voicedata_filename",
    path: ":rails_root/public/system/:class/:attachment/:id_partition/:voicedata_filename",
    styles: {preview_ogg: {format: 'ogg'}, preview_mp3: {format: 'mp3'}},
    :processors => [:voice_preview]
  attr_accessor :raw_delete_key
  attr_accessor :tos_agree

  validates_attachment :archive, presence: true,
    size: {less_than: 5120.kilobytes},
    content_type: { content_type: %w(application/zip) }
  validates :archive, voice_archive: { on: :create }
  validates :title, presence: true, length: {maximum: 80}
  validates :description, length: {maximum: 280}
  validates :unit, inclusion: {in: %w(metric imperial)}
  validates :archive_fingerprint, uniqueness: { on: :create }
  validates :lang, length: {is: 2, allow_nil: true}
  validates :tos_agree, acceptance: {if: Proc.new{|o| o.new_record? }}

  before_save :hash_delete_key
  after_validation :dup_errormsg

  # Add stub method of old style paperclip for rails_admin backward compatibility
  # https://github.com/sferik/rails_admin/issues/1674
  def self.attachment_definitions
    {archive: true}
  end

  def delete_key_valid?(key)
    delete_key? or return false
    salt, hash = delete_key.split('/', 2)
    hash == Digest::SHA1.hexdigest("#{salt}#{key}")
  end

  def to_ini
    ini = {
      id: id,
      title: title,
      description: description,
      archive_md5: archive_fingerprint,
      author: author,
      rating: rating,
      lang: lang,
      unit: unit,
      version: 1,
      created_at: created_at,
      updated_at: updated_at,
    }.map { |k, v|
      "#{k.to_s}=#{v.to_s.chomp.gsub(/[\\\n\r]/, {"\n" => '\\n', "\r" => '\\r', "\\" => '\\\\'})}"
    }.join("\n")
  end

  def update_rating
    rating_store = {}
    ratings.each do |r|
      rating_store[r.ident] = r.value
    end
    update_attribute :rating, rating_store.values.inject(:+) / rating_store.count.to_f
  end

  def update_dlcount
    update_attribute :dlcount, dl_log_entries.count
  end

  private

  def hash_delete_key
    @raw_delete_key.blank? and return true
    salt = SecureRandom.hex
    self[:delete_key] = "#{salt}/" + Digest::SHA1.hexdigest("#{salt}#{@raw_delete_key}")
    true
  end

  def dup_errormsg
    self.errors[:archive_fingerprint].blank? and return true
    self.errors[:archive_fingerprint].each do |e|
      self.errors[:archive] << e
    end
  end
end
