# git-mq-setup.sh
# ------------------------------------------------------------------------------
#
# Universal Git-MQ initialization hook.  All Git-MQ command scripts should
# source this; it performs git specific initialization, before delegating to
# the generic mingw-pkg MQ plugin initialization code.
#
# ------------------------------------------------------------------------------
#
# $Id$
#
# Written by Keith Marshall <keith@users.osdn.me>
# Copyright (C) 2018, 2022, Keith Marshall
#
# This file is part of the Git-MQ program suite.
#
# The Git-MQ program suite is free software: you can redistribute it
# and/or modify it under the terms of the GNU General Public Licence
# as published by the Free Software Foundation, either version 3 of
# the Licence, or (at your option) any later version.
#
# The Git-MQ program suite is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public Licence for more details.
#
# You should have received a copy of the GNU General Public Licence
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# ------------------------------------------------------------------------------
#
# Establish the location of the patch files, and associated control files,
# for the active (possibly default) patch series; (this is expressed as a
# path name relative to GIT_DIR for the current working directory).
#
  GIT_ROOT=`git rev-parse --show-toplevel`
  GIT_DIR=${GIT_DIR-"$GIT_ROOT/`git -C "$GIT_ROOT" rev-parse --git-dir`"}

  git_mq_map_queue(){ local suffix=""
    test -f "$GIT_DIR/queue" && suffix=`sed 1q "$GIT_DIR/queue"`
    test "x$suffix" = x || suffix="-$suffix"
    echo "$1$suffix"
  }
  mq_patchdir=`git_mq_map_queue "$GIT_DIR/patches"`

# All Git-MQ commands should call the following function to process
# command options, using a while loop, thus:
#
#    while git_mq_getopt "$@"
#    do case $1 in -*) # handle command specific options
#	esac; shift
#    done
#
# Note: the command specific loop should interpret command specific
# options; standard Git-MQ global options, such as "--colour", will
# be interpreted within this function itself.
#
  git_mq_getopt(){
    if test "x$mq_optarg" = x
    then case $1 in
	   -*) case $1 in
		 --no-colour) mq_colour=never ;;
		 --colour) mq_enable_colour "${2-always}" ;;
	       esac; true ;;
	    *) false ;;
	 esac
    else eval $mq_optarg='"$1"' mq_optarg=""; true
    fi
  }

# The following function handles assignments for the mode control
# argument, if any, to the "--colour" option.
#
# FIXME: Although the "git qseries" output back-end will validate any
# such assignment, at point of use, it would be prudent to check such
# assignments here, at point of definition.
#
  mq_enable_colour(){
    case $1 in -*) mq_colour=always ;; *) mq_optarg=mq_colour ;; esac
  }

# The following function may be used to wrap any Git-MQ action, so
# that its default output will be in colour, and paged, when stdout
# is directed to a terminal device.
#
  git_mq_enable_colour_pager(){
    test "${mq_colour+set}" = set || { test -t 1 && mq_colour=always; }
    "$@" | git_pager
  }

# Set up shell variable "fatal" as a hook to insert "fatal: " as a prefix
# to termination messages dispatched via "mq_abort", (which we adopt as our
# preferred replacement for git's "die" function).
#
  fatal="mq_abort 2 fatal:"
  mq_abort(){ local err=$1; shift; mq_complain "$*"; exit $err; }
  mq_complain(){ echo >&2 ${mq_facility+"$mq_facility: "}"$*"; }

# We provide the following helper function, to facilitate collection of
# optional argument lists.
#
  mq_collect(){ eval $1='${'$1'+"$'$1' "}"$2"'; }

# Delegate further initialization requirements to the generic (mingw-pkg)
# mq-setup.sh plugin module.
#
  mq_require "$libexecdir/mq-setup.sh"

# Finally, ensure that a git pre-commit hook is in place, to protect the
# integrity of the Git-MQ patch queue against the potentially destructive
# effect of an unmanaged commit over applied Git-MQ patches.
#
  test -f "$GIT_DIR/hooks/pre-commit" && mq_require git-pre-commit-verify ||
    mq_require git-pre-commit-setup
#
# ------------------------------------------------------------------------------
# $RCSfile$: end of file
