# git-mq-finish-all.sh
# ------------------------------------------------------------------------------
#
# Supplementary shell script fragment; invoked via "mq_require", it performs
# the clean up actions required when all currently applied Git-MQ patches are
# converted from managed patches to regular commits, removed from the Git-MQ
# patch series, and released from Git-MQ control.
#
# ------------------------------------------------------------------------------
#
# $Id$
#
# Written by Keith Marshall <keith@users.osdn.me>
# Copyright (C) 2019, Keith Marshall
#
#
# This file is part of the Git-MQ program suite.
#
# The Git-MQ program suite is free software: you can redistribute it
# and/or modify it under the terms of the GNU General Public Licence
# as published by the Free Software Foundation, either version 3 of
# the Licence, or (at your option) any later version.
#
# The Git-MQ program suite is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public Licence for more details.
#
# You should have received a copy of the GNU General Public Licence
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# ------------------------------------------------------------------------------
#
# When the "--applied" option is specified, explicit patch, or patch
# range, references are not allowed; check, diagnose, and abort when
# such anomalous usage is identified.
#
  test $# -gt 0 && {
    mq_complain "error: option '--applied' implies all applied patches"
    for mq_ref
    do mq_complain "error: explicit reference to '$mq_ref' not allowed"
    done
    exit 2
  }

# After performing the "git qfinish --applied" operation, there will
# be no remaining applied patches, so we can simply erase all content
# from the status file, when we are done; however, there is likely to
# still be content relating to unapplied patches, within the series
# file, so allocate temporary storage into which to rewrite this.
#
  mq_require mq-tmpfile
  mq_tmpfile mq_series_file_tmp '`mktemp --tmpdir="$mq_patchdir"` ||
    $fatal "cannot establish series file update cache"'

# Regardless of how we eventually update them, we still need to read
# the initial content of both the series, and the status files.
#
  mq_map_control_file_refs "$mq_patchdir" series status

# The actual "git qfinish --applied" operation is accomplished by the
# following awk script; it simply reads the status file, compiling a
# list of all patches to which it refers, deletes all associated patch
# tags, and patch files, and writes a replacement copy of the series
# file, excluding the records associated with these patches.
#
  awk -F"[ :]" > "$mq_series_file_tmp" '
    FILENAME ~ /series$/ { if( ! patch[$1] ) print; }
    FILENAME ~ /status$/ { patch[$2] = 1; qtags = qtags" "$2; }
    END {
      if( ! qtags ) exit 2;
      system( "git tag --delete qparent qbase qtip" qtags "> /dev/null 2>&1" );
      system( "cd \42'"$mq_patchdir"'\42 && rm" qtags );
    }
  ' "$mq_status_file" "$mq_series_file" || {

# The only error condition, which will cause the preceding script to
# terminate abnormally, arises if the status file initially contained
# no patch references; diagnose it.
#
    mq_abort 2 "error: no patches have been applied"
  }

# Provided the "git qfinish" operation did not terminate abnormally,
# we now truncate the status file to zero length, (thus effectively
# deleting all applied patch references), and update the series file
# to reflect the removal of the associated patches from the queue.
#
  > $mq_status_file; mq_update mq_series_file
#
# ------------------------------------------------------------------------------
# $RCSfile$: end of file
