#!/bin/sh
#
# mingw32-ldd.sh
#
# Utility script to plot DLL dependency graphs for specified MS-Windows
# executable files and/or dynamic link libraries.
#
#
# $Id$
#
# Written by Keith Marshall <keithmarshall@users.sourceforge.net>
# Copyright (C) 2016, 2017, MinGW.org Project
#
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice (including the next
# paragraph) shall be included in all copies or substantial portions of the
# Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#
#
  CMD=`basename $0`
  usage(){ echo >&2 "Usage: $CMD {EXENAME | DLLNAME} ..."; STATUS=2; echo; }
#
  PROGRAM_FILES="$SYSTEM_DRIVE/program files"
  SYSTEM_DLLPATH="$SYSTEM_DRIVE/windows/system32"
  EXEPATH=`dirname "$1"` STATUS=0
#
  set_dllpath(){
  # Usage: set_dllpath "colon:separated:path:list"
  #
  # Convert the specified colon separated path list to an equivalent
  # space separated list of individually quoted path names.
  #
    DLLPATH=`echo "$*" | awk -F':' '{
      for (i = 0; i < NF; print "\""$++i"\"");
    }'`
  }
#
  depends(){
  # Usage: depends {EXENAME | DLLNAME}
  #
  # Extract a list of DLL dependencies from the specified executable or
  # dynamic link library file; if EXENAME or DLLNAME explicitly names a
  # file, it will be processed directly, otherwise $DLLPATH is searched,
  # in an attempt to locate it.
  #
    test -f "$1" && list_depends "$1" || (
      exename="$1"; eval set -- $DLLPATH; for dir in "$EXEPATH" "$@"
      do test -f "$dir/$exename" && { list_depends "$dir/$exename"; break; }
      done
    )
  }
#
  list_depends(){
  # Usage: list_depends {EXENAME | DLLNAME}
  #
  # Helper function, called by depends(), to perform dependency list
  # extraction from the file explicitly named by EXENAME or DLLNAME.
  #
    objdump -x "$1" | awk 'BEGIN{IGNORECASE=1}/dll name/{print $NF}'
  }
#
  plot_depends(){
  # Usage: plot_depends {EXENAME | DLLNAME} [count] [prefix]
  #
  # Recursively plot the DLL dependency graph for the executable or
  # dynamic link library file named by EXENAME or DLLNAME; the count
  # and prefix arguments are not specified at top-level entry, but are
  # appended on recursive calls, to specify the number of identified
  # dependencies remaining to be added to the graph display, at the
  # current level of recursion, and the prefix string to be used to
  # indent them within the graph.
  #
    echo "$3"${2+" +- "}"$1"
    local ref count references="`depends $1`"
    for ref in $references
      do references=`ref_shift $references` count=`ref_count $references`
	 plot_depends $ref $count ${2+"$3 `plot_linkage $2`  "}
      done
  }
#
# Helper functions, called by plot_depends(), to assist in formatting
# the plot of the DLL dependency graph; ref_count() simply reports the
# number of entries in the reference list, (passed as an argument list),
# while ref_shift() removes the first such reference from the same list,
# (after it has been scheduled for processing).
#
  ref_count(){ echo $#; }
  ref_shift(){ test $# -gt 0 && shift; echo "$@"; }
#
  plot_linkage(){
  # Usage: plot_linkage count
  #
  # Select the linkage glyph, for connection of the current node in
  # the DLL dependency graph to those below, at the same indentation
  # level; it is a solid vertical bar, when the count argument shows
  # one or more following nodes at the same level, or space when all
  # nodes at this level have already been displayed, (indicated by a
  # count of zero).
  #
    test $1 -gt 0 && echo "|" || echo " "
  }
#
  show_depends(){
  # Usage: show_depends {EXENAME | DLLNAME}
  #
  # Wrapper to invoke plot_depends() for EXENAME or DLLNAME, provided
  # it exists, otherwise display a diagnostic message
  #
    test -f "$1" && plot_depends "$1" || {
      echo >&2 "$CMD: ***ERROR*** $1: file not found"
      STATUS=2
    }
  }
#
# Effective program entry point: set up the DLLPATH (in space separated,
# quoted path list format), then plot the DLL dependency graph for each
# EXENAME or DLLNAME specified on the command line.
#
  test $# -gt 0 || { usage; exit 2; }
  set_dllpath "${DLLPATH-$PROGRAM_FILES:/mingw/bin:$SYSTEM_DLLPATH}"
  while test $# -gt 0; do show_depends "$1"; shift; test $# -gt 0 && echo; done
  exit $STATUS
#
# $RCSfile$: end of file
