// vim: foldmethod=marker commentstring=//%s
package mn.jp.kekkouyakan.collision;
import mn.jp.kekkouyakan.geom.KyVector2f;
import mn.jp.kekkouyakan.util.KyLinkedElement;
import mn.jp.kekkouyakan.util.KyFloatComparator;

public class ColSpace
{//{{{
	//http://marupeke296.com/COL_2D_No8_QuadTree.html

	public static class Node extends KyLinkedElement<IShapeOwner>
	{//{{{
		int minOrder = -1;
		int maxOrder = -1;
		Object userObject = null;

		public Node( IShapeOwner so_ )
		{//{{{
			super( so_ );
		}//}}}
		public ColShape getShape()
		{//{{{
			return getValue().getShape();
		}//}}}
		public boolean isDead()
		{//{{{
			return getValue().isDead();
		}//}}}
	}//}}}
	public static interface IShapeOwner
	{//{{{
		ColShape getShape();
		boolean isDead();
	}//}}}
	public static interface IVisitor
	{//{{{
		boolean visit( Node node_ );
	}//}}}
	public static interface IVisitorEx extends IVisitor
	{//{{{
		Object getUserObject();
	}//}}}

	KyFloatComparator floatComparator;
	Node[][] headNodeLists;
	float[] xOffsetList;
	float[] yOffsetList;

	public ColSpace( int maxLevel_, float x0_, float y0_, float x1_, float y1_, KyFloatComparator cmp_ )
	{//{{{
		if( maxLevel_ < 0 ){
			throw new AssertionError();
		}
		if( maxLevel_ > 3 ){
			throw new AssertionError();
		}
		floatComparator = cmp_;
		headNodeLists = new Node[ maxLevel_ + 1 ][];
		for( int lv_ = 0; lv_ < headNodeLists.length; ++lv_ ){
			Node[] nodeLs_ = headNodeLists[lv_] = new Node[ 1 << (2*lv_) ];
			for( int i_ = 0; i_ < nodeLs_.length; ++i_ ){
				Node node_ = nodeLs_[i_] = new Node(null);
				node_.minOrder = lv_;
				node_.maxOrder = i_;
			}
		}
		final int maxDiv_ = 1 << maxLevel_;
		xOffsetList = new float[ maxDiv_ + 1 ];
		yOffsetList = new float[ maxDiv_ + 1 ];
		float xmin_ = Math.min( x0_, x1_ );
		float xmax_ = Math.max( x0_, x1_ );
		float xsz_ = xmax_ - xmin_;
		float ymin_ = Math.min( y0_, y1_ );
		float ymax_ = Math.max( y0_, y1_ );
		float ysz_ = ymax_ - ymin_;

		for( int i_ = 0; i_ < maxDiv_; ++i_ ){
			xOffsetList[i_] = xmin_ + (float)(i_*xsz_)/(float)maxDiv_;
			yOffsetList[i_] = ymin_ + (float)(i_*ysz_)/(float)maxDiv_;
		}
		xOffsetList[maxDiv_] = xmax_;
		yOffsetList[maxDiv_] = ymax_;
	}//}}}

	static int bitSeparate( int n_ )
	{//{{{
		n_ = (n_|(n_<<8)) & 0x00ff00ff;
		n_ = (n_|(n_<<4)) & 0x0f0f0f0f;
		n_ = (n_|(n_<<2)) & 0x33333333;
		return (n_|(n_<<1)) & 0x55555555;
	}//}}}
	static int getOrder( int x_, int y_ )
	{//{{{
		return bitSeparate( x_ ) | (bitSeparate( y_ ) << 1);
	}//}}}
	static int findIndex( float v_, float[] offsets_ )
	{//{{{
		if( v_ < offsets_[0] ){
			return -1;
		}
		for( int i_ = 1; i_ < offsets_.length; ++i_ ){
			if( v_ < offsets_[i_] ){
				return i_-1;
			}
		}
		return -1;
	}//}}}

	public int getLevelCount()
	{//{{{
		return headNodeLists.length;
	}//}}}
	public int getDivCount()
	{//{{{
		return 1 << (headNodeLists.length - 1);
	}//}}}
	public int getOrderBits( int order_, int lv_ )
	{//{{{
		return (order_ >> 2*(headNodeLists.length - lv_ - 1))&3;
	}//}}}
	public int getOrderIndex( int order_, int lv_ )
	{//{{{
		return (order_ >> 2*(headNodeLists.length - lv_ - 1));
	}//}}}
	public int getShareLevel( int order1_, int order2_ )
	{//{{{
		int xor_ = order1_ ^ order2_;
		int sz_ = headNodeLists.length;
		for( int i_ = 1; i_ < sz_; ++i_ ){
			if( getOrderBits( xor_, i_ ) != 0 ){
				return i_ - 1;
			}
		}
		return sz_-1;
	}//}}}
	public int getShareLevel( Node node_ )
	{//{{{
		return getShareLevel( node_.minOrder, node_.maxOrder );
	}//}}}
	public int getShareIndex( int order1_, int order2_ )
	{//{{{
		int lv_ = getShareLevel( order1_, order2_ );
		return getOrderBits( order1_, lv_ );
	}//}}}
	public int getShareIndex( Node node_ )
	{//{{{
		return getShareIndex( node_.minOrder, node_.maxOrder );
	}//}}}
	public float getMinX()
	{//{{{
		return xOffsetList[0];
	}//}}}
	public float getMaxX()
	{//{{{
		return xOffsetList[xOffsetList.length - 1];
	}//}}}
	public float getMinY()
	{//{{{
		return yOffsetList[0];
	}//}}}
	public float getMaxY()
	{//{{{
		return yOffsetList[yOffsetList.length - 1];
	}//}}}
	public float getWidth()
	{//{{{
		return getMaxX() - getMinX();
	}//}}}
	public float getHeight()
	{//{{{
		return getMaxY() - getMinY();
	}//}}}

	public void register( Node node_ )
	{//{{{
		ColShape shape_ = node_.getShape();
		float xf0_ = shape_.getMinX();
		float yf0_ = shape_.getMinY();
		float xf1_ = shape_.getMaxX();
		float yf1_ = shape_.getMaxY();
		int xi0_ = findIndex( xf0_, xOffsetList );
		int yi0_ = findIndex( yf0_, yOffsetList );
		int xi1_ = findIndex( xf1_, xOffsetList );
		int yi1_ = findIndex( yf1_, yOffsetList );
		int o0_ = getOrder( xi0_, yi0_ );
		int o1_ = getOrder( xi1_, yi1_ );
		node_.minOrder = o0_;
		node_.maxOrder = o1_;
		int lv_ = getShareLevel( o0_, o1_ );
		int index_ = getOrderIndex( o0_, lv_ );
		headNodeLists[lv_][index_].connectNext( node_ );
	}//}}}
	public Node register( IShapeOwner shape_ )
	{//{{{
		Node node_ = new Node( shape_ );
		register( node_ );
		return node_;
	}//}}}
	public Node register( final ColShape shape_ )
	{//{{{
		return register( new IShapeOwner(){
			public ColShape getShape()
			{
				return shape_;
			}
			public boolean isDead()
			{//{{{
				return false;
			}//}}}
		});
	}//}}}

	static boolean visits( Node node_, IVisitor vis_ )
	{//{{{
		for( node_ = (Node)node_.getNext(); node_ != null; node_ = (Node)node_.getNext() ){
			if( !vis_.visit( node_ ) ){
				return false;
			}
		}
		return true;
	}//}}}

	public boolean isOutOfRange( ColShape shape_ )
	{//{{{
		if( shape_.getMinX() < getMinX() ){
			return true;
		}
		if( shape_.getMaxX() > getMaxX() ){
			return true;
		}
		if( shape_.getMinY() < getMinY() ){
			return true;
		}
		if( shape_.getMaxY() > getMaxY() ){
			return true;
		}
		return false;
	}//}}}

	void visitAllNodes( IVisitor vi_ )
	{//{{{
		for( Node[] nodeLs_ : headNodeLists ){
			for( Node node_ : nodeLs_ ){
				if( !visits( node_, vi_ ) ){
					return;
				}
			}
		}
	}//}}}
	void visitCollisionNodes( int order1_, int order2_, IVisitor vi_ )
	{//{{{
		int tgtLv_ = getShareLevel( order1_, order2_ );
		int index_ = getOrderIndex( order1_, tgtLv_ );
		if( !visits(headNodeLists[tgtLv_][index_], vi_) ){
			return;
		}
		int wt_ = 4;
		for( int lv_ = tgtLv_ + 1; lv_ < headNodeLists.length; ++lv_ ){
			int index1_ = index_ * wt_;
			int index2_ = (index_+1) * wt_;
			for( int i_ = index1_; i_ < index2_; ++i_ ){
				if( !visits( headNodeLists[lv_][i_],  vi_ ) ){
					return;
				}
			}
			wt_ *= 4;
		}
		for( int lv_ = tgtLv_ - 1; lv_ > 0; --lv_ ){
			int index0_ = getOrderIndex( order1_, lv_ );
			if( !visits( headNodeLists[lv_][index0_], vi_ ) ){
				return;
			}
		}
		if( tgtLv_ != 0 ){
			if( !visits( headNodeLists[0][0], vi_ ) ){
				return;
			}
		}
	}//}}}
	boolean visitCollisionNodes( ColShape shape_, IVisitor vi_ )
	{//{{{
		float xf0_ = shape_.getMinX();
		float yf0_ = shape_.getMinY();
		float xf1_ = shape_.getMaxX();
		float yf1_ = shape_.getMaxY();
		if( floatComparator.less( xf0_, getMinX() ) ){
			return false;
		}
		if( floatComparator.moreEquals( xf1_, getMaxX() ) ){
			return false;
		}
		if( floatComparator.less( yf0_, getMinY() ) ){
			return false;
		}
		if( floatComparator.moreEquals( yf1_, getMaxY() ) ){
			return false;
		}
		int xi0_ = findIndex( xf0_, xOffsetList );
		int yi0_ = findIndex( yf0_, yOffsetList );
		int xi1_ = findIndex( xf1_, xOffsetList );
		int yi1_ = findIndex( yf1_, yOffsetList );
		int o0_ = getOrder( xi0_, yi0_ );
		int o1_ = getOrder( xi1_, yi1_ );
		visitCollisionNodes( o0_, o1_, vi_ );
		return true;
	}//}}}
	boolean visitCollisionNodes( ColShape shape1_, ColShape shape2_, IVisitor vi_ )
	{//{{{
		float xf0_ = Math.min( shape1_.getMinX(), shape2_.getMinX() );
		float yf0_ = Math.min( shape1_.getMinY(), shape2_.getMinY() );
		float xf1_ = Math.max( shape1_.getMaxX(), shape2_.getMaxX() );
		float yf1_ = Math.max( shape1_.getMaxY(), shape2_.getMaxY() );
		if( floatComparator.less( xf0_, getMinX() ) ){
			return false;
		}
		if( floatComparator.moreEquals( xf1_, getMaxX() ) ){
			return false;
		}
		if( floatComparator.less( yf0_, getMinY() ) ){
			return false;
		}
		if( floatComparator.moreEquals( yf1_, getMaxY() ) ){
			return false;
		}
		int xi0_ = findIndex( xf0_, xOffsetList );
		int yi0_ = findIndex( yf0_, yOffsetList );
		int xi1_ = findIndex( xf1_, xOffsetList );
		int yi1_ = findIndex( yf1_, yOffsetList );
		int o0_ = getOrder( xi0_, yi0_ );
		int o1_ = getOrder( xi1_, yi1_ );
		visitCollisionNodes( o0_, o1_, vi_ );
		return true;
	}//}}}

}//}}}
