/***********************************
 *                                 *
 * Template of CLIB for MachiKania *
 *                                 *
 ***********************************/

// Include the main header
#include "./clib.h"
// CLIBINIT is placed at specific address
#define CLIBINIT __attribute__((section(".machikania_clib")))

/*
 * g_data[] contains the data from MachiKania compiler for this library
 * See the comment in clib.h
 */
void*** g_data;

/*
 * clibdata[] contains the data from this library for MachiKania compiler
 * clibdata[0] : Version of CLIB (This library is for MachiKania ver 1.4)
 * clibdata[1] : Size of data memory
 * clibdata[2] : Public function array
 * clibdata[2-]: Reserved for higher verion of CLIB
 */
const void* const clibdata[]={
	(void*) 0x0140,             // clibdata[0]
	(void*) CLIB_DATA_MEM_SIZE, // clibdata[1] 
	&functions[0],              // clibdata[2]
	0                           // Finally, end with 0
};

/*
 * This function will be called when initializing library,
 * and used to exchange data between MachiKania BASIC and this library.
 * This is only a function placed in .machikania_clib section for providing
 * static start address at 0xA0008000. When editing this function is required,
 * the size of kseg2_program_mem and address of kseg0_program_mem 
 * in linker script must be changed.
 * This function is called twice, for exchanging data and for calling init().
 */
void* CLIBINIT clib_init(void*** data){
	// Store pointer to data
	if (data) g_data=data;
	// Call user initialization routine
	else init();
	// Return pointer to clibdata
	return (void*)&clibdata[0];
}

// call lib_calloc_memory() function
void* __attribute__((section("clib_calloc"))) clib_calloc(int size, void*** g_data){
	asm volatile("lw $v1,0($a1)"); // g_data[0]
	asm volatile("lw $v0,4($a1)"); // g_data[1]
	asm volatile("lw $v0,0($v0)"); // g_data[1][0]
	asm volatile("lw $gp,0($v1)"); // g_data[0][0]
	asm volatile("jr $v0");
}

// call lib_delete() function
void __attribute__((section("clib_free"))) clib_free(void* addr, void*** g_data){
	asm volatile("lw $v1,0($a1)"); // g_data[0]
	asm volatile("lw $v0,4($a1)"); // g_data[1]
	asm volatile("lw $v0,4($v0)"); // g_data[1][1]
	asm volatile("lw $gp,0($v1)"); // g_data[0][0]
	asm volatile("jr $v0");
}

// Function to get g_data in $v0
void*** __attribute__((section("clib_g_data"))) clib_g_data(){ return g_data; }

// Video functions below use following macro
// $v0=g_data;
// lw $v1,0($v0)   // g_data[0]
// lw $v0,12($v0)  // g_data[3]
// lw $v0,"x"($v0) // g_data[3][x/4]
// lw $gp,0($v1)   // g_data[0][0]
// jalr $v0
#define machikania_video(x) \
	clib_g_data();\
	asm volatile("lw $v1,0($v0)");\
	asm volatile("lw $v0,12($v0)");\
	asm volatile("lw $v0,"x"($v0)");\
	asm volatile("lw $gp,0($v1)");\
	asm volatile("jalr $v0")

void __attribute__((section("start_composite"))) start_composite(void)
{ machikania_video("0"); }
void __attribute__((section("stop_composite"))) stop_composite(void)
{ machikania_video("4"); }
void __attribute__((section("printchar"))) printchar(unsigned char n)
{ machikania_video("8"); }
void __attribute__((section("printstr"))) printstr(unsigned char *s)
{ machikania_video("12"); }
void __attribute__((section("printnum"))) printnum(unsigned int n)
{ machikania_video("16"); }
void __attribute__((section("printnum2"))) printnum2(unsigned int n,unsigned char e)
{ machikania_video("20"); }
void __attribute__((section("cls"))) cls(void)
{ machikania_video("24"); }
void __attribute__((section("vramscroll"))) vramscroll(void)
{ machikania_video("28"); }
void __attribute__((section("setcursorcolor"))) setcursorcolor(unsigned char c)
{ machikania_video("32"); }
void __attribute__((section("setcursor"))) setcursor(unsigned char x,unsigned char y,unsigned char c)
{ machikania_video("36"); }
void __attribute__((section("set_palette"))) set_palette(unsigned char n,unsigned char b,unsigned char r,unsigned char g)
{ machikania_video("40"); }
void __attribute__((section("set_bgcolor"))) set_bgcolor(unsigned char b,unsigned char r,unsigned char g)
{ machikania_video("44"); }

// Graphic functions below use following macro
// $v0=g_data;
// lw $v1,0($v0)   // g_data[0]
// lw $v0,16($v0)  // g_data[4]
// lw $v0,"x"($v0) // g_data[4][x/4]
// lw $gp,0($v1)   // g_data[0][0]
// jalr $v0
#define machikania_graphic(x) \
	clib_g_data();\
	asm volatile("lw $v1,0($v0)");\
	asm volatile("lw $v0,16($v0)");\
	asm volatile("lw $v0,"x"($v0)");\
	asm volatile("lw $gp,0($v1)");\
	asm volatile("jalr $v0")

void __attribute__((section("g_pset"))) g_pset(int x,int y,unsigned int c)
{ machikania_graphic("0"); }
void __attribute__((section("g_putbmpmn"))) g_putbmpmn(int x,int y,char m,char n,const unsigned char bmp[])
{ machikania_graphic("4"); }
void __attribute__((section("g_clrbmpmn"))) g_clrbmpmn(int x,int y,char m,char n)
{ machikania_graphic("8"); }
void __attribute__((section("g_gline"))) g_gline(int x1,int y1,int x2,int y2,unsigned int c)
{ machikania_graphic("12"); }
void __attribute__((section("g_hline"))) g_hline(int x1,int x2,int y,unsigned int c)
{ machikania_graphic("16"); }
void __attribute__((section("g_circle"))) g_circle(int x0,int y0,unsigned int r,unsigned int c)
{ machikania_graphic("20"); }
void __attribute__((section("g_circlefill"))) g_circlefill(int x0,int y0,unsigned int r,unsigned int c)
{ machikania_graphic("24"); }
void __attribute__((section("g_boxfill"))) g_boxfill(int x1,int y1,int x2,int y2,unsigned int c)
{ machikania_graphic("28"); }
void __attribute__((section("g_putfont"))) g_putfont(int x,int y,unsigned int c,int bc,unsigned char n)
{ machikania_graphic("32"); }
void __attribute__((section("g_printstr"))) g_printstr(int x,int y,unsigned int c,int bc,unsigned char *s)
{ machikania_graphic("36"); }
void __attribute__((section("g_printnum"))) g_printnum(int x,int y,unsigned char c,int bc,unsigned int n)
{ machikania_graphic("40"); }
void __attribute__((section("g_printnum2"))) g_printnum2(int x,int y,unsigned char c,int bc,unsigned int n,unsigned char e)
{ machikania_graphic("44"); }
unsigned int __attribute__((section("g_color"))) g_color(int x,int y)
{ machikania_graphic("48"); }

// Keyboard functions below use following macro
// $v0=g_data;
// lw $v1,0($v0)   // g_data[0]
// lw $v0,20($v0)  // g_data[5]
// lw $v0,"x"($v0) // g_data[5][x/4]
// lw $gp,0($v1)   // g_data[0][0]
// jalr $v0
#define machikania_keyboard(x) \
	clib_g_data();\
	asm volatile("lw $v1,0($v0)");\
	asm volatile("lw $v0,20($v0)");\
	asm volatile("lw $v0,"x"($v0)");\
	asm volatile("lw $gp,0($v1)");\
	asm volatile("jalr $v0")

unsigned char __attribute__((section("shiftkeys"))) shiftkeys()
{ machikania_keyboard("0"); }
unsigned char __attribute__((section("ps2readkey"))) ps2readkey()
{ machikania_keyboard("4"); }

//*/