﻿using System;
using System.Text;
using System.Linq;
using System.Collections.Generic;
using System.IO;


using Vintagestory.API.Common;
using Vintagestory.API.Server;
using Vintagestory.API.Config;
using Vintagestory.API.Datastructures;

namespace AdminToolkit
{
	/// <summary>
	/// Admininstrators toolkit Mod system
	/// </summary>
	public class AdminToolkit : ModSystem
	{
		/* Things it should do:
		 * ==============
		 * [DONE] Server RULES Printout {richtext / multi-linugual }
		 * [DONE] Optional: ACCEPTANCE / REJECT of server rules {changes players role}
		 * [DONE] List all Admins (online or offline) & track last logoff date
		 * [DONE] Keeps track of Elapsed game calendar days for players
		 * [DONE] Alter ADMIN text to look 'different'
		 * [DONE] Cyclic automatic Backups
		 * ?????? Broadcast messages, on a schedule ??????
		 * Status/MOTD/Server name : custom formats to indicate server state/time/things
		 */

		private ICoreAPI API { get; set; }
		private ICoreServerAPI ServerAPI { get; set; }

		internal readonly string[] adminPriviledges = {
				Privilege.buildblockseverywhere,
				Privilege.useblockseverywhere,
				Privilege.gamemode,
				Privilege.pickingrange,
				Privilege.kick,
				Privilege.ban,
				Privilege.whitelist,
				Privilege.setwelcome,
				Privilege.announce,
				Privilege.readlists,
				Privilege.give,
				Privilege.setspawn,
				Privilege.controlserver,
				Privilege.tp,
				Privilege.time,
				Privilege.grantrevoke,
				Privilege.root,
				Privilege.commandplayer,
			};

		internal const string _lastLoginKey = @"LastLogin";
		internal const string _lastGameDayCountKey = @"LastDayCount";
		internal const string _configFilename = @"admintoolkit.json";

		internal static List<string> AdminRoles;	

		public AdminToolkit( )
		{
			
		}

		/// <summary>
		/// Gets or sets the cached configuration.
		/// </summary>
		/// <value>The cached configuration.</value>
		internal AdminModConfig CachedConfiguration {
			get {				
				return ( AdminModConfig )ServerAPI.ObjectCache[_configFilename];
			}
			set {
				ServerAPI.ObjectCache.Add(_configFilename, value);
			}
		}

		public override bool ShouldLoad(EnumAppSide forSide)
		{
			return forSide.IsServer( );
		}

		public override void StartServerSide(ICoreServerAPI api)
		{
			base.StartServerSide(api);

			Mod.Logger.Notification("Starting Administrators Toolkit mod");

			this.API = api;
			this.ServerAPI = API as ICoreServerAPI;

			PopulateAdminRoleTable( );
			PrepareServersideConfig( );

			this.ServerAPI.RegisterCommand(new RulesCommand(this.ServerAPI) );
			this.ServerAPI.RegisterCommand(new AdminListingCommand(this.ServerAPI) );
			this.ServerAPI.RegisterCommand(new BackupCycleCommand(this.ServerAPI) );

			if (CachedConfiguration.BoomingVoice) {
				this.ServerAPI.Event.PlayerChat += BoomingVoiceOfAuthority;
			}
			//TODO: Server MOTD / Name, DYNAMIC status info [season, moon-phase, events... ect]

			this.ServerAPI.Event.ServerRunPhase(EnumServerRunPhase.Shutdown, SaveConfigSettings);
		}

		private void PopulateAdminRoleTable( )
		{
			AdminRoles = new List<string>( );
			foreach (var role in ServerAPI.Server.Config.Roles) {
				
				if (role.PrivilegeLevel >= 200 && role.Privileges.Any(adminPriviledges.Contains )) {
					
					AdminRoles.Add(role.Code);

					#if DEBUG
					Mod.Logger.VerboseDebug("Role {0} =[{1}] considerd Admin-like", role.Name, role.Code);
					#endif
				}
			}
		}

		private void PrepareServersideConfig( )
		{
			AdminModConfig config = ServerAPI.LoadModConfig<AdminModConfig>(_configFilename);

			if (config == null) {
				//Regen default
				Mod.Logger.Warning("Regenerating default config as it was missing / unparsable...");
				ServerAPI.StoreModConfig<AdminModConfig>(new AdminModConfig( ), _configFilename);
				config = ServerAPI.LoadModConfig<AdminModConfig>(_configFilename);
			}

			this.CachedConfiguration = config;

			if (this.CachedConfiguration.RuleRoleChangerEnabled) { Mod.Logger.Notification("Admin toolkit; Role change on Rule accept: * ENABLED *"); }
		}

		/// <summary>
		/// Makes Administrator text messages appear 'different'
		/// </summary>
		/// <returns>The voice of authority.</returns>
		/// <param name="byPlayer">By player.</param>
		/// <param name="channelId">Channel identifier.</param>
		/// <param name="message">Message.</param>
		/// <param name="consumed">Consumed.</param>		                                      
		private void BoomingVoiceOfAuthority(IServerPlayer byPlayer, int channelId, ref string message, ref string data, BoolRef consumed)
		{
			if (AdminRoles.Contains(byPlayer.Role.Code)) {
				//Make text lined
				consumed.value = false;
				StringBuilder adminMessage = new StringBuilder( );
				foreach (char letter in message) {
					
					adminMessage.Append(letter);
					if (!Char.IsWhiteSpace(letter)) {
						//adminMessage.Append(@"̅̅̅");
						adminMessage.Append(@"̲");
					}
				}

				message = adminMessage.ToString( );
			}
		}

		private void SaveConfigSettings( )
		{
			if (this.CachedConfiguration != null) {				
				Mod.Logger.Notification("Persisting configuration.");
				ServerAPI.StoreModConfig<AdminModConfig>(this.CachedConfiguration, _configFilename);
			}
		}

}
}

