﻿using System;
using System.Text;
using System.Linq;
using System.Collections.Generic;
using System.IO;


using Vintagestory.API.Common.Entities;
using Vintagestory.API.Server;
using Vintagestory.API.Common;
using Vintagestory.API.Config;

using Vintagestory.Client.NoObf;//Perhaps this 'LanguageConfig' moves to a more general place? ...API.Config...

namespace AdminToolkit
{
	public class RulesCommand : AdminModCommand
	{
		
		private const string _rulesPathKey = @"Rules";
		private const string _acceptKeyword = @"accept";
		private const string _ruleAcceptKey = @"RulesAccepted";

		private string RulesPath;
		private Dictionary<string, List<string>> Rules;

		private bool rolesAreValid;


		public RulesCommand(ICoreServerAPI _serverAPI) : base(_serverAPI)
		{			
			PreloadRulesText( );
			ValidateAllRolesExist( );

			this.Command = "rules";
			this.Description = "Display server Rules";
			this.handler += ServerRulesCommand;

			this.ServerAPI.Event.PlayerJoin += UponJoin;
		}

		private void PreloadRulesText( )
		{
			Rules = new Dictionary<string, List<string>>( );
			RulesPath = ServerAPI.GetOrCreateDataPath(_rulesPathKey);
			GuiCompositeSettings.LanguageConfig[] supportedLangs = ServerAPI.Assets.Get<GuiCompositeSettings.LanguageConfig[]>(new AssetLocation("lang/languages.json"));
			string[ ] lang_codes = supportedLangs.Select(sl => sl.Code).ToArray( );//{ "en", "de", "it", "fr",  "pt-br", "ru" };

			foreach (var languageCode in lang_codes) {
				//For each locale that is configured... get rules_?.txt
				string rulesFilePath = Path.Combine(RulesPath, $"rules_{languageCode}.txt");
				if (File.Exists(rulesFilePath)) {
					using (var ruleFileStream = File.OpenRead(rulesFilePath)) {
						StreamReader ruleReader = new StreamReader(ruleFileStream);

						#if DEBUG
						Logger.VerboseDebug($"found 'rules_{languageCode}.txt' of size {ruleFileStream.Length} bytes");
						#endif

						string line;
						while ((line = ruleReader.ReadLine( )) != null) {
							if (Rules.ContainsKey(languageCode)) {
								Rules[languageCode].Add(line);
							} else {
								Rules.Add(languageCode, new List<string>( ));
								Rules[languageCode].Add(line);
							}
						}

						ruleReader.Close( );
					}
				} else {
					Logger.Warning("Non existant '{0}' file for Rules-text", $"rules_{languageCode}.txt");
				}
			}
		}

		private void ValidateAllRolesExist( )
		{
			if (!ServerAPI.Server.Config.Roles.Any(pr => pr.Code == this.CachedConfiguration.PlayerRoleNormal)) {
				Logger.Error("Role code defined for Normal missing: [{0}]", this.CachedConfiguration.PlayerRoleNormal);

			} else if (!ServerAPI.Server.Config.Roles.Any(pr => pr.Code == this.CachedConfiguration.PlayerRoleRestrain)) {
				Logger.Error("Role code defined for Restrained missing: [{0}]", this.CachedConfiguration.PlayerRoleRestrain);

			} else {
				Logger.VerboseDebug("Both role codes confgured are present.");

				this.rolesAreValid = true;
			}
		}

		internal static bool CheckRuleAccepted(IServerPlayer byPlayer)
		{
			if (byPlayer.ServerData.CustomPlayerData.ContainsKey(_ruleAcceptKey)) {
				bool ruleAccept;
				if (bool.TryParse(byPlayer.ServerData.CustomPlayerData[_ruleAcceptKey], out ruleAccept)) {
					return ruleAccept;
				}
			}
			return false;
		}

		/// <summary>
		/// rules command
		/// </summary>
		/// <returns>The rules command.</returns>
		/// <param name="player">Player.</param>
		/// <param name="groupId">Group identifier.</param>
		/// <param name="args">Arguments.</param>
		private void ServerRulesCommand(IServerPlayer player, int groupId, CmdArgs args)
		{
			//Full printout of RULES file.
			//also Used for formally accepting the rules... should be logged - when accepting

			if (args.Length == 1) {
				string word = args.PopWord( );

				if (string.Compare(_acceptKeyword, word, StringComparison.InvariantCultureIgnoreCase) == 0) {

					if (CheckRuleAccepted(player) == false) {
						
						PlayerAcceptsRulesAction( player);
					} else {
						player.SendMessage(groupId, MultiLang.Get(player, "atk:already-accept",player.PlayerName), EnumChatType.CommandSuccess);
					}
				} else {
					player.SendMessage(groupId, MultiLang.Get(player, "atk:type-chat-rules", player.PlayerName ,_acceptKeyword), EnumChatType.CommandError);
				}

			} else {
				var targetLocale = player.LanguageCode;

				if (Rules.ContainsKey(targetLocale)) {

					//TODO: rate limit for large files?
					foreach (var line in Rules[targetLocale]) {
						player.SendMessage(groupId, line, EnumChatType.OthersMessage);
					}
				} else {
					//Subsitute rules?
					player.SendMessage(groupId, MultiLang.Get(player, "atk:missing-rules", targetLocale), EnumChatType.CommandError);
				}
			}
		}

		/// <summary>
		/// Send Rule info text or Admin welcome, plus other info
		/// </summary>
		/// <param name="byPlayer">By player.</param>
		private void UponJoin(IServerPlayer byPlayer)
		{
			if (!AdminToolkit.AdminRoles.Contains(byPlayer.Role.Code)) {				
				//For regular players
				if (RulesCommand.CheckRuleAccepted(byPlayer) == false) {
					
					byPlayer.SendMessage(GlobalConstants.CurrentChatGroup, MultiLang.GetUnformatted(byPlayer,"atk:type-rules"), EnumChatType.Notification);

					if (this.CachedConfiguration.RuleRoleChangerEnabled) {
						PlayerNeverAcceptedRulesAction(byPlayer);
					}
				}
			}
		}

		private void PlayerAcceptsRulesAction(IServerPlayer byPlayer )
		{
			ServerAPI.BroadcastMessageToAllGroups(MultiLang.Get(byPlayer,"atk:accepted-rules", byPlayer.PlayerName), EnumChatType.CommandSuccess);
			byPlayer.ServerData.CustomPlayerData[_ruleAcceptKey] = true.ToString( );
			ServerAPI.Server.LogNotification($"Player:{byPlayer.PlayerName} UID: {byPlayer.PlayerUID} HAS ACCEPTED RULES @ {DateTime.Now.ToShortDateString( )}");

			if (byPlayer.Role.Code == this.CachedConfiguration.PlayerRoleRestrain && this.rolesAreValid && this.CachedConfiguration.RuleRoleChangerEnabled) {
				//Joined as 'restrained' player - will have their role changed to normal

				IPlayerRole normalRole = ServerAPI.Server.Config.Roles.Single(rl => string.Compare(rl.Code, this.CachedConfiguration.PlayerRoleNormal, StringComparison.InvariantCultureIgnoreCase) == 0);

				ServerAPI.Permissions.SetRole(byPlayer, normalRole.Code);
				Logger.Notification("Player {0} accepted the rules, and was promoted to role: {1}", byPlayer.PlayerName, byPlayer.Role.Name);
				byPlayer.SendMessage(GlobalConstants.GeneralChatGroup, MultiLang.Get(byPlayer, "atk:promoted-to", byPlayer.PlayerName, byPlayer.Role.Name), EnumChatType.OthersMessage);
			} else {
				Logger.Debug("Role change process did not complete; not enabled or mis-configured");
			}
		}

		private void PlayerNeverAcceptedRulesAction(IServerPlayer byPlayer )
		{			
			if (byPlayer.Role.Code == this.CachedConfiguration.PlayerRoleNormal && this.rolesAreValid && this.CachedConfiguration.RuleRoleChangerEnabled ) {
				//Joined as 'normal' player - will have their role changed to restricted 

				IPlayerRole restrainRole = ServerAPI.Server.Config.Roles.Single(rl => string.Compare( rl.Code,  this.CachedConfiguration.PlayerRoleRestrain, StringComparison.InvariantCultureIgnoreCase) == 0);

				ServerAPI.Permissions.SetRole(byPlayer, restrainRole.Code);
				Logger.Notification("Player {0} never accepted rules, and was demoted to role: {1}", byPlayer.PlayerName, byPlayer.Role.Name);
				byPlayer.SendMessage(GlobalConstants.GeneralChatGroup, MultiLang.Get(byPlayer, "atk:demoted-to", byPlayer.PlayerName, byPlayer.Role.Name), EnumChatType.OthersMessage);
			} 
		}
	}
}

