//
//  MSFindPanelController.m
//  Fumizuki
//
//  Created by 二鏡 on 11/09/08.
//  Copyright 2011年 二鏡庵. All rights reserved.
//

#import "MSFindPanelController.h"
#import "MSTextView.h"
#import "MSPager.h"
#import "MSLayoutBlockManager.h"

NSString *MSFindPanelPatternUpdateNotifiation = @"FindPanelRegrexUpdate";
static NSString *searchStringContext = @"searchString";
static NSString *replaceStringContext = @"replaceString";

@interface MSFindPanelController ()
- (void)_abandonTarget;
- (void)_getTarget;
- (void)_abandonMatch;
- (void)_makeMatch;
- (void)_abandonExpression;
- (void)_compileExpression;
- (void)_moveToPreviousMatch;
- (void)_moveToNextMatch;
@property (readwrite) BOOL hasMatch;
@end

static id _sid;
@implementation MSFindPanelController
@synthesize searchString, replaceString, matchStatus, targetMode, useRegularExpression, ignoreCase, hasMatch;

+ (id)sharedFindPanel
{
    if(_sid)
        return _sid;
    
    _sid = [[self alloc] initWithWindowNibName: @"FindPanel"];
    return _sid;
}

- (id)initWithWindow:(NSWindow *)window
{
    self = [super initWithWindow:window];
    if (self) {
        // Initialization code here.
        [self addObserver: self
               forKeyPath: searchStringContext
                  options: 0
                  context: searchStringContext];
        [self addObserver: self
               forKeyPath: replaceStringContext
                  options: 0
                  context: replaceStringContext];
        id center = [NSNotificationCenter defaultCenter];
        [center addObserver: self
                   selector: @selector(_regexpDidUpdate:)
                       name: MSFindPanelPatternUpdateNotifiation
                     object: self];
        invalidIndexes = [NSMutableIndexSet indexSet];
    }
    
    return self;
}

- (void)windowDidLoad
{
    [super windowDidLoad];
    
    [[self window] center];
    // Implement this method to handle any nitialization after your window controller's window has been loaded from its nib file.
}

- (void)observeValueForKeyPath:(NSString *)keyPath
                      ofObject:(id)object
                        change:(NSDictionary *)change
                       context:(void *)context
{
    if(context == searchStringContext)
    {
        isPatternTouched = YES;
        return;
    }

    if(context == replaceStringContext)
    {
        return;
    }
    
    [super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
}

- (void)_regexpDidUpdate:(id)notif
{
}

- (void)_setMatchStatus:(NSInteger)count
{
    NSString *string, *format;
    switch(count)
    {
        case -1:
            string = @""; // reset status
            break;
        case 0:
            string = NSLocalizedString(@"no matchs", @"Find Panel no match string");
            break;
        default:
            format = NSLocalizedString(@"%d matchs",@"match pattern");
            string = [NSString stringWithFormat: format, count];
    }
    
    [self willChangeValueForKey: @"matchStatus"];
    matchStatus = string;
    [self didChangeValueForKey: @"matchStatus"];
}

- (void)windowDidResignKey:(id)obj
{
    [self _abandonTarget];
    [self _abandonMatch];
}


#pragma mark -
#pragma mark primitive operations
- (void)_abandonTarget
{
    _target.textView = nil;
    _target.storage = nil;
}

- (void)_getTarget
{
    if(_target.textView == nil)
    {
        id view = [[NSApp mainWindow] firstResponder];
        _target.textView = view;
        _target.storage = _target.textView.pager.content.textStorage;
    }
}

- (void)_abandonMatch
{
    matches = nil;
    currentMatch = -1;
    [self _setMatchStatus: -1];
    self.hasMatch = NO;
}

- (NSRange)_targetRange
{
    NSRange range = _target.textView.selectedRange;
    if(range.length == 0 || targetMode == eAllTextMode)
        range = NSMakeRange(0,[_target.storage length]);
    return range;
}

- (void)_makeMatch
{
    NSRange range = [self _targetRange];
    matches = [[regexp matchesInString: _target.storage.mutableString
                               options: 0
                                 range: range] mutableCopy];
    currentMatch = -1;
    [invalidIndexes removeAllIndexes];
    [self _setMatchStatus: [matches count]];
    self.hasMatch = ([matches count] != 0);
}

- (void)_abandonExpression
{
    regexp = nil;
}

- (void)_compileExpression
{
    regexp = nil;
    isPatternTouched = NO;
    id pattern;
    
    if(searchString != nil)
    {
        NSError *error;
        NSRegularExpressionOptions options;
        options = NSRegularExpressionAnchorsMatchLines; 
        if(ignoreCase)
            options |= NSRegularExpressionCaseInsensitive;
        if(useRegularExpression == NO)
            pattern = [NSRegularExpression escapedPatternForString: searchString];
        else
            pattern = searchString;
        
        regexp = [NSRegularExpression regularExpressionWithPattern: pattern
                                                           options: options
                                                             error: &error];
        if(regexp == nil)
        {
            [NSApp presentError: error];
            return;
        }
    }
}

- (void)_compileTemplate
{
    // 置き換え文字列の更新。必要ならencupslate
    replaceTemplate = nil;
    if(replaceString != nil)
    {
        if(useRegularExpression == NO)
            replaceTemplate = [NSRegularExpression escapedTemplateForString: replaceString];
        else
            replaceTemplate = replaceString;
    }
}

- (NSString*)_applyTemplate:(NSString*)templateString
                   withRange:(NSRange)targetRange
{
    id substring = [[_target.storage mutableString] substringWithRange: targetRange];
    NSMatchingOptions options = 0;
    if(templateString == nil)
        templateString = @"";
    return [regexp stringByReplacingMatchesInString: substring
                                            options: options
                                              range: NSMakeRange(0, [substring length])
                                       withTemplate: templateString];
}

- (CFIndex)_replaceMatchAtIndex:(CFIndex)i
                   withTemplate:(NSString*)templateString
{
    // 一応ブロックしておく
    if([invalidIndexes containsIndex: i] || i == -1)
        return 0;
    
    NSTextCheckingResult *result = [matches objectAtIndex: i];
    NSRange range = result.range;
    id newString = [self _applyTemplate: templateString
                              withRange: range];
    CFIndex delta = [newString length] - range.length;
    [_target.textView insertText: newString
                replacementRange: range];

    [invalidIndexes addIndex: i];
    
    return delta;
}

- (NSRange)_activeMatchingRange
{
    CFIndex limit = [matches count];
    if([invalidIndexes count] == limit)
        return NSMakeRange(NSNotFound,0);
    
    CFIndex i;
    CFIndex first = 0,last = 0;
    for(i=0;i<limit;i++)
    {
        if([invalidIndexes containsIndex: i] == NO)
        {
            first = i;
            break;
        }
    }
            
    for(i=limit-1;i>=0;i--)
    {
        if([invalidIndexes containsIndex: i] == NO)
        {
            last = i;
            break;
        }
    }
    
    // first ~ last
    NSTextCheckingResult *firstMatch = [matches objectAtIndex: first];
    NSTextCheckingResult *lastMatch  = [matches objectAtIndex: last];
    NSRange firstRange = firstMatch.range;
    NSRange lastRange  = lastMatch.range;
    return NSUnionRange(firstRange,lastRange);
}

- (void)_shiftMatches:(CFIndex)delta
           fromIndex:(CFIndex)index
{
    CFIndex limit = [matches count];
    NSAssert( index >= 0 && index < limit, @"shift index over bounds");
    
    NSRange targetRange = NSMakeRange(index,limit-index);
    NSArray *subMatches = [matches subarrayWithRange: targetRange];
    NSMutableArray *shiftedArray = [NSMutableArray array];
    for(NSTextCheckingResult *result in subMatches)
    {
        id newResult = [result resultByAdjustingRangesWithOffset: delta];
        [shiftedArray addObject: newResult];
    }
    
    [matches replaceObjectsInRange: targetRange
              withObjectsFromArray: shiftedArray];
}

- (void)_replaceAll
{
    // 1. active firstとactive last matchから置換の全対象レンジを発見
    NSRange range = [self _activeMatchingRange];
    if(range.location == NSNotFound)
        return;
    
    // 2. applyで置換文字列を生成
    id newString = [self _applyTemplate: replaceTemplate
                              withRange: range];
    
    // 3. 後は同じ
    CFIndex delta = [newString length] - range.length;
    [_target.textView insertText: newString
                replacementRange: range];
    
    range.length += delta;
    [_target.textView setSelectedRange: NSMakeRange(NSMaxRange(range),0)];

    // 4. 全てのマッチをinvalidにする
    [invalidIndexes addIndexesInRange: NSMakeRange(0,[matches count])];
}

- (BOOL)_isCurrentMatchActive
{
    return (currentMatch != -1) && ([invalidIndexes containsIndex: currentMatch] == NO);
}

- (void)_shiftMatchesIfNeed:(CFIndex)delta
{
    if(delta == 0)
        return;
    
    if(currentMatch != [matches count] -1)
        [self _shiftMatches:delta fromIndex: currentMatch+1];
}

- (void)_moveToNextMatch
{
    if(matches == nil)
        [self _makeMatch];

    // count == 0の場合を含む
    // 全てのmatchが含まれているなら次の行き先は見つからない
    CFIndex count = [matches count];
    if(count == [invalidIndexes count]) 
    {
        currentMatch = -1;
        return;
    }
    
    // 
    for(;;)
    {
        currentMatch = (currentMatch+1)%count;
        if([invalidIndexes containsIndex: currentMatch] == NO)
            break;
    }
    NSTextCheckingResult *result = [matches objectAtIndex: currentMatch];
    NSRange range = result.range;
    _target.textView.selectedRange = range;
    [_target.textView scrollToReferenceIndex: range.location];
    [_target.textView setNeedsDisplay: YES];
}

- (void)_moveToPreviousMatch
{
    if(matches == nil)
        [self _makeMatch];
    
    CFIndex count = [matches count];
    if(count == [invalidIndexes count])
    {
        currentMatch = -1;
        return;
    }
    
    for(;;)
    {
        currentMatch = (currentMatch-1+count)%count;
        if([invalidIndexes containsIndex: currentMatch] == NO)
            break;
    }
    
    NSTextCheckingResult *result = [matches objectAtIndex: currentMatch];
    NSRange range = result.range;
    _target.textView.selectedRange = range;
    [_target.textView scrollToReferenceIndex: range.location];
    [_target.textView setNeedsDisplay: YES];
}

#pragma mark actions
- (IBAction)updateSearchString:(id)sender
{
    isPatternTouched = YES;
    [self _setMatchStatus: -1];
}

- (IBAction)updateReplaceString:(id)sender
{
}

- (IBAction)changeSearchRange:(id)sender
{
    isPatternTouched = YES;
    [self _setMatchStatus: -1];
}

- (IBAction)search:(id)sender
{
    if(isPatternTouched)
    {
        [self _abandonMatch];
        [self _compileExpression];   
    }
    [self _getTarget];
    [self _makeMatch];
}

- (IBAction)replaceAll:(id)sender
{
    // 現在のマッチを全てtemplateで置き換える
    // 問題あり。一部を動かしてから前置感が来るかも
    if(isPatternTouched)
    {
        [self _abandonMatch];     
        [self _compileExpression];
    }
    [self _compileTemplate];
    [self _replaceAll];
}

- (IBAction)replace:(id)sender
{
    // currentMatchを入れ替える
    // selectedRangeは入れ替えた文字列にする
    if([self _isCurrentMatchActive] == NO)
        return;
    
    [self _compileTemplate];
    CFIndex delta = [self _replaceMatchAtIndex: currentMatch
                                  withTemplate: replaceTemplate];
    [self _shiftMatchesIfNeed: delta];
    
    NSTextCheckingResult *aMatch = [matches objectAtIndex: (NSUInteger)currentMatch];
    NSRange matchRange = [aMatch range];
    matchRange.length += delta;
    _target.textView.selectedRange = matchRange;
    [_target.textView setNeedsDisplay: YES];
}

- (IBAction)replaceAndFindNext:(id)sender
{
    // currentMatchを置換文字列で入れ替え、次のマッチへ移動
    if([self _isCurrentMatchActive])
    {
        [self _compileTemplate];
        CFIndex delta = [self _replaceMatchAtIndex: currentMatch
                                      withTemplate: replaceTemplate];
        [self _shiftMatchesIfNeed: delta];
    }
    
    // 次へ移動. replaceがない場合でも移動はしうる
    [self _moveToNextMatch];
}

- (IBAction)previous:(id)sender
{
    if(isPatternTouched)
    {
        [self _abandonMatch];     
        [self _compileExpression];
    }
    [self _getTarget];
    [self _moveToPreviousMatch];
    
}

- (IBAction)next:(id)sender
{
    if(isPatternTouched)
    {
        [self _abandonMatch];     
        [self _compileExpression];
    }
    [self _getTarget];
    [self _moveToNextMatch];
}

@end
