//
//  MSTextView+NSTextInputClient.m
//  Manuscript
//
//  Created by 二鏡 on 12/03/20.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSTextView.h"
#import "MSTextView_Private.h"

@implementation MSTextView (NSTextInputClient)
- (NSAttributedString*)attributedString
{
    return pager_.content.textStorage;
}

- (NSAttributedString *)attributedSubstringForProposedRange:(NSRange)aRange
                                                actualRange:(NSRangePointer)actualRange
{
    NSTextStorage *storage = pager_.content.textStorage;
    NSRange textRange = NSMakeRange(0,[storage length]);
    NSRange targetRange = NSIntersectionRange(textRange, aRange);
    if(actualRange)
        *actualRange = targetRange;
    return [storage attributedSubstringFromRange: targetRange];
}

- (NSUInteger)characterIndexForPoint:(NSPoint)aPoint
{
    NSRect rect;
    rect.origin = aPoint;
    rect.size = NSZeroSize;
    
    // screen -> window
    rect = [[self window] convertRectFromScreen: rect];
    
    // window -> local
    aPoint = [self convertPoint: aPoint fromView: nil];
    NSUInteger idx = [pager_ indexOfCharacterAtPoint: aPoint];
    return idx;
}

- (NSRect)firstRectForCharacterRange:(NSRange)aRange
                         actualRange:(NSRangePointer)actualRange
{
    // locationの行を探し、内部の相対index値から目指す文字を特定し、
    // その範囲でmake_bounding_rectを使う
    // 最後にそれをscreenに変換して出来上がり
    NSRect rect = [pager_ firstRectForCharacterRange: aRange
                                         actualRange: actualRange];
    rect = [self convertRect: rect toView: nil];
    rect = [[self window] convertRectToScreen: rect];
    rect.size.width *= -1;
    return rect;
}

- (BOOL)drawsVerticallyForCharacterAtIndex:(NSUInteger)charIndex
{
    return YES;
}

- (BOOL)hasMarkedText
{
    return inputState.has_marked_text();
}

- (NSAttributedString*)formatInsertionText:(id)aString
                          replacementRange:(NSRange)aRange
{
    NSMutableAttributedString *ret;
    NSRange textRange = NSMakeRange(0,[aString length]);
    if([aString isKindOfClass: [NSAttributedString class]])
    {
        ret = [aString mutableCopy];
        [ret addAttribute: NSFontAttributeName 
                    value: [pager_.format font]
                    range: textRange];
        // ルビ補正なし、フォントのみ補正
        return ret;
    }
    
    // 単独コントロールコードという謎の入力が来ることがあるので対処
    if([aString length] == 1)
    {
        id set = [NSCharacterSet controlCharacterSet];
        if([set characterIsMember: [aString characterAtIndex: 0]])
        {
            aString = @"";
            textRange.length = 0;
        }
    }
    ret = [[NSMutableAttributedString alloc]
           initWithString: aString];
    
    // 基本属性処理
    [ret addAttribute: NSFontAttributeName 
                value: [pager_.format font]
                range: textRange];
    [ret addAttribute: NSForegroundColorAttributeName
                value: [NSColor blackColor]
                range: textRange];
    
    NSTextStorage *storage = pager_.content.textStorage;
    NSUInteger length = [storage length];
    if(length == aRange.location)
        return [ret copy];
    
    //    // ルビ補正
    //    NSRange activeRange;
    //    NSRange limitRange;
    //    limitRange.location = MAX(aRange.location-1,0);
    //    limitRange.length = length - limitRange.location;
    //    id ruby = [storage attribute: FZRubyAttributeName
    //                         atIndex: aRange.location
    //           longestEffectiveRange: &activeRange
    //                         inRange: limitRange];
    //    
    //    if(ruby)
    //    {
    //        NSUInteger head = activeRange.location+1;
    //        NSUInteger tail = NSMaxRange(activeRange);
    //        NSUInteger insertion = aRange.location;
    //        
    //        if(insertion >= head && insertion < tail)
    //        {
    //            [ret addAttribute: FZRubyAttributeName
    //                        value: ruby
    //                        range: textRange];
    //            [ret addAttribute: NSForegroundColorAttributeName
    //                        value: gRubyStringColor
    //                        range: textRange];
    //        }
    //    }
    //    
    return [ret copy];
}


- (NSAttributedString*)formatMarkedText:(id)aString
                    forReplacementRange:(NSRange)aRange
{
    // 特定のレンジに対するmarkedTextをつくる。ルビの拡張などが行われる
    NSMutableAttributedString *ret;
    if([aString isKindOfClass: [NSAttributedString class]])
    {
        ret = [aString mutableCopy];
    }
    else
    {
        ret = [[NSMutableAttributedString alloc]
               initWithString: aString];
    }
    // フォントの強制上書きはどの道行うので問答無用でmutableとしてよいと思われ。
    NSRange textRange = NSMakeRange(0,[aString length]);
    [ret addAttribute: NSFontAttributeName 
                value: [pager_.format font]
                range: textRange];
    
    // ルビ解析
    NSTextStorage *storage = pager_.content.textStorage;
    NSUInteger length = [storage length];
    if(length == aRange.location)
        return [ret copy];
    
    //    NSRange activeRange;
    //    NSRange limitRange;
    //    limitRange.location = MAX(aRange.location-1,0);
    //    limitRange.length = length - limitRange.location;
    //    id ruby = [storage attribute: FZRubyAttributeName
    //                         atIndex: aRange.location
    //           longestEffectiveRange: &activeRange
    //                         inRange: limitRange];
    //    
    //    if(ruby)
    //    {
    //        NSUInteger head = activeRange.location+1;
    //        NSUInteger tail = NSMaxRange(activeRange);
    //        NSUInteger insertion = aRange.location;
    //        
    //        if(insertion >= head && insertion < tail)
    //        {
    //            [ret addAttribute: FZRubyAttributeName
    //                        value: ruby
    //                        range: textRange];
    //            [ret addAttribute: NSForegroundColorAttributeName
    //                        value: gRubyStringColor
    //                        range: textRange];
    //        }
    //    }
    return [ret copy];
}


- (void)insertText:(id)aString
  replacementRange:(NSRange)replacementRange
{
    replacementRange = inputState.replacement_range_for_request(replacementRange);
    
    NSUInteger length = [aString length];
    NSTextStorage *storage = pager_.content.textStorage;
    NSRange updatedRange = NSMakeRange(replacementRange.location,length);
    
    // undoの優先順位は
    // 1. 変換前のセレクション
    // 2. 直打ちのセレクション
    // 3. 選択なし(つまり新規挿入)
    id oldString = nil;
    if(inputState.has_marked_text())
    {
        oldString = inputState.get_undo_buffer(); // nilも含む
    }
    else
    {
        if(replacementRange.length != 0)
        {
            oldString = [storage attributedSubstringFromRange: replacementRange];
        }
    }
    
    id newString = [self formatInsertionText: aString
                            replacementRange: replacementRange];
    
    // 逆演算登録
    {
        id undoTarget = [[delegate_ undoManager] prepareWithInvocationTarget: self];
        NSRange undoRange = NSMakeRange(replacementRange.location, [newString length]);
        // oldString == nil <-> 新規挿入なのでdeleteでバック
        if(oldString != nil)
            [undoTarget insertText: oldString
                  replacementRange: undoRange];
        else
            [undoTarget deleteCharactersInRange: undoRange];
    }
    
    [self memoryScrollRegion];
    caretState.lock();
    {
        // 改行文字が含まれる場合、contentのパラグラフ生成に指示を出す
        MSLayoutBlockManager *content = pager_.content;
        [content replaceCharactersInRange: replacementRange
                     withAttributedString: newString];
        [self unmarkText];
    }
    caretState.unlock();
    
    // replace baseに
    self.selectedRange = NSMakeRange(NSMaxRange(updatedRange),0);
    inputState.reset_selection_anchor();
    caretState.clear_height();
    [self scrollToReferenceIndex: NSMaxRange(updatedRange)-1];
    [[self inputContext] invalidateCharacterCoordinates];
    [self setNeedsDisplay: YES];
}

- (void)setMarkedText:(id)aString 
        selectedRange:(NSRange)newSelection
     replacementRange:(NSRange)replacementRange
{
    replacementRange = inputState.replacement_range_for_request(replacementRange);
    
    NSTextStorage *storage = pager_.content.textStorage;
    // 変換を開始する際にselectionがあった場合、それをundo用に保存
    if(inputState.has_marked_text() == false)
    {
        if(replacementRange.length == 0)
            inputState.set_undo_buffer(nil);
        else
            inputState.set_undo_buffer([storage attributedSubstringFromRange: replacementRange]);
    }
    
    NSUInteger newLength = [aString length];
    
    [self memoryScrollRegion];
    caretState.lock();
    self.markedRange = NSMakeRange(replacementRange.location, newLength);
    id newString = [self formatMarkedText: aString
                      forReplacementRange: replacementRange];
    [pager_.content replaceCharactersInRange: replacementRange
                        withAttributedString: newString];
    
    if(newLength == 0)
        [self unmarkText];
    caretState.unlock();
    
    self.selectedRange = 
    NSMakeRange(replacementRange.location + newSelection.location,
                newSelection.length);
    inputState.reset_selection_anchor();
    caretState.clear_height();
    
    // scroll
    {
        NSRange range = inputState.markedRange;
        NSUInteger ref;
        if(range.location == NSNotFound)
            ref = NSMaxRange(replacementRange);
        else
            ref = NSMaxRange(range);
        if(ref != 0)
            ref--;
        [self scrollToReferenceIndex: ref];
    }
    [[self inputContext] invalidateCharacterCoordinates];
    [self setNeedsDisplay: YES];
    
}

- (void)unmarkText {
    inputState.set_undo_buffer(nil);
    self.markedRange = NSMakeRange(NSNotFound, 0);
    [[self inputContext] discardMarkedText];
}

- (NSArray *)validAttributesForMarkedText {
    // We only allow these attributes to be set on our marked text (plus standard attributes)
    // NSMarkedClauseSegmentAttributeName is important for CJK input, among other uses
    // NSGlyphInfoAttributeName allows alternate forms of characters
    return [NSArray arrayWithObjects:NSMarkedClauseSegmentAttributeName, nil];
}

- (NSInteger)windowLevel {
    return [[self window] level];
}


@end
