/*
 *  GlyphMetric.h
 *  Manuscript
 *
 *  Created by 二鏡 on 11/05/21.
 *  Copyright 2011 二鏡庵. All rights reserved.
 *
 */
#import <CoreFoundation/CoreFoundation.h>
#import <Quartz/Quartz.h>
#include <map>
#include <string>

/*
 * How to use
 *  GlyphMetricStorage.
 *   Font毎のメトリクスキャッシュ。データは最初にここから検索して使う。見つからない場合はloaderを併用する。
 *  GlyphMetricLoader
 *   キャッシュがミスヒットした時、そのグリフと配列上のindexをpushしておく
 *   で、一括で実データをloadし、write_backとmemoryでデータを取り出す。使い終ったらresetで再初期化
 *   キャパに到達した時点でpushはfalse(空きなし)を通知するので必要なら動的にロードすること
 */


using namespace std;
using namespace __gnu_cxx;

const uint32_t cGlyphMetricLoaderCapacity = 64;

struct GlyphMetrics
{
    CGSize translate;
    CGSize advance;
};

class GlyphMetricsCache
{
    map<CGGlyph, GlyphMetrics> table;
public:
    inline const GlyphMetrics* find(CGGlyph key) const
    {
        map<CGGlyph, GlyphMetrics>::const_iterator p;
        
        p = table.find(key);
        if(p == table.end())
            return NULL;
        else
            return &(p->second);
    }
    inline void insert(CGGlyph key, const GlyphMetrics &metric)
    {
        table.insert(make_pair(key,metric));
    }
};


class GlyphMetricsCacheTable
{
    class FontIdentifier
    {
        string _name;
        CGFloat _size;
    public:
        FontIdentifier() {}
        FontIdentifier(const FontIdentifier &obj) : _name(obj._name), _size(obj._size)
        {}
        FontIdentifier(string name, CGFloat size) : _name(name), _size(size)
        {}
        
        bool operator <(const FontIdentifier &obj) const
        {
            if(_name < obj._name)
                return true;
            return _size < obj._size;
        }
        
        bool operator >(const FontIdentifier &obj) const
        {
            if(_name > obj._name)
                return true;
            return _size > obj._size;
            
        }
        
        bool operator ==(const FontIdentifier &obj) const
        {
            return _name == obj._name && _size == obj._size;
        }
    };
    
    map<FontIdentifier,GlyphMetricsCache*> cacheTable; // font -> storage table
    
public:
    GlyphMetricsCache *getCache(CTFontRef font) 
    {
        map<FontIdentifier,GlyphMetricsCache*>::const_iterator p;
        
        CFStringRef fontName = CTFontCopyPostScriptName(font);
        const char *cName = [(id)fontName UTF8String];
        CGFloat fontSize = CTFontGetSize(font);
        FontIdentifier key = FontIdentifier(string(cName),fontSize);
        p = cacheTable.find(key);
        if(p == cacheTable.end())
        {
            GlyphMetricsCache *cache = new GlyphMetricsCache;
            cacheTable.insert(make_pair(key,cache));
            CFRelease(fontName);
            return cache;
        }
        
        CFRelease(fontName);
        return p->second;
    }
};

// グリフ管理用の共有テーブル
// グリフストレージはここから検索される。
extern GlyphMetricsCacheTable gGlyphMetricsCacheTable;

class GlyphMetricsLoader
{
    CGGlyph  glyphs[cGlyphMetricLoaderCapacity];
    uint32_t  write_back_buf[cGlyphMetricLoaderCapacity];
    CGSize  translate_buf[cGlyphMetricLoaderCapacity];
    CGSize  advance_buf[cGlyphMetricLoaderCapacity];
    uint32_t count;
    
public:
    // parametor
    CTFontRef font;
    CGFloat layoutScale;
    
    GlyphMetricsLoader()
    {
        count = 0;
        layoutScale = 1.0;
    }
    
    inline bool push(CGGlyph glyph, uint32_t write_back_index)
    {
        glyphs[count] = glyph;
        write_back_buf[count] = write_back_index;
        count++;
        return count != cGlyphMetricLoaderCapacity;
    }
    
    inline void load(CTFontOrientation orientation)
    {
        CTFontGetAdvancesForGlyphs(font, orientation, glyphs, advance_buf, count);
        CTFontGetVerticalTranslationsForGlyphs(font, glyphs, translate_buf, count);
    }
    
    inline void write_back(CGSize *__restrict__ translates, CGSize *__restrict__ advances)
    {
        for(int i=0;i<count;i++)
        {
            translates[write_back_buf[i]] = translate_buf[i];
            advances[write_back_buf[i]] = advance_buf[i];
        }
    }
    
    inline void memory(GlyphMetricsCache *cache)
    {
        for(int i=0;i<count;i++)
        {
            GlyphMetrics metric = {translate_buf[i],advance_buf[i]};
            cache->insert(glyphs[i], metric);
        }
    }
    
    inline void reset()
    {
        count = 0;
    }
    
};
