//
//  _GlyphLocator.h
//  CoreTextVerticalTest
//
//  Created by 二鏡 on 11/07/28.
//  Copyright 2011年 二鏡庵. All rights reserved.
//

#ifndef VTKit__GlyphLocator_h
#define VTKit__GlyphLocator_h
#import <ApplicationServices/ApplicationServices.h>
#import <CoreFoundation/CFString.h>
#include <vector>
#include <string>
#include <map>
#import "_GlyphMetricsLoader.h"
#import "CompiledAttributedString.h"
#import "_CharacterAttributePredicate.h"

using namespace std;

@protocol _GlyphBufferSpaceDelegate
@required
- (void)substituteGlyphs:(CGGlyph*)glyphs
              characters:(const UniChar*)chars
                   count:(CFIndex)count
                  inFont:(NSFont*)font;
- (void)parseCharacters:(const UniChar*)characters
           toAttributes:(uint8_t*)attributes
                 length:(CFIndex)length;
@end

class _GlyphLocatorBase;

#pragma mark -

class _GlyphBufferSpace
{
protected:
    // calculation buffer
    vector<CGGlyph> glyphs;
    vector<CGSize> advances;
    vector<CGSize> translations;
    vector<uint8> characterAttribute;
    
    // final layout buffer
    vector<CGPoint> positions;
    vector<CGFloat> caretOffsets; // 必ず0から読み出さなくてはならない
    
    GlyphMetricsLoader glyphLoader;
    __strong id <_GlyphBufferSpaceDelegate> delegate;
    
    // target string
    CompiledAttributedString *target;
    
public:
    CGFloat layoutScale;
    friend class _GlyphLocatorBase;
    
private:
    void reserveBuffer(unsigned long size)
    {
        glyphs.reserve(size);
        positions.reserve(size);
        advances.reserve(size);
        translations.reserve(size);
        characterAttribute.reserve(size);
        caretOffsets.reserve(size+1);
    };
    
    void resizeBuffer(unsigned long size)
    {
        glyphs.resize(size);
        positions.resize(size);
        advances.resize(size);
        translations.resize(size);
        characterAttribute.resize(size);
        caretOffsets.resize(size+1);
    }
    
public:
    _GlyphBufferSpace()
    {
        this->reserveBuffer(2048);
        layoutScale = 1.0;
        glyphLoader.layoutScale = layoutScale;
    }
    
    _GlyphBufferSpace(size_t initial_size)
    {
        this->reserveBuffer(initial_size);
        layoutScale = 1.0;
        glyphLoader.layoutScale = layoutScale;
    }
    
    void setDelegate(id <_GlyphBufferSpaceDelegate> obj)
    {
        delegate = obj;
    }
    
    inline const CompiledAttributedString* getString()
    {
        return target;
    }
    
    inline const CGGlyph* getGlyphs()
    {
        return &glyphs[0];
    }
    
    inline const CGPoint* getPositions()
    {
        return &positions[0];
    }
    
    inline const CGFloat* getCaretOffsets()
    {
        return &caretOffsets[0];
    }
    
    inline const vector<CGFloat>& getCaretOffsetsRef()
    {
        return caretOffsets;
    }
    
    inline const unsigned long getGlyphCount()
    {
        return glyphs.size();
    }
    
    // char -> glyph
    // バッファは自動で拡張される
    void loadString(CompiledAttributedString *value);
    // horizontal glyph -> vertical glyph (by delegate)
    void loadStage1substitute();
    // take advances & translations from glyphs
    // *** stage2は共有キャッシュを叩くのでスレッドセーフではない ***
    void loadStage2metrics(CTFontOrientation orientation);
    void loadStage3attributes();
};

#pragma mark -

class _GlyphLocatorBase
{
    _GlyphBufferSpace *buffer;
protected:
    inline const vector<CGGlyph>& glyphs_() const { return buffer->glyphs; }
    inline const vector<CGSize>& advances_() const { return buffer->advances; }
    inline const vector<CGSize>& translations_() const { return buffer->translations; } 
    inline const vector<uint8>& characterAttribute_() const { return buffer->characterAttribute; } 
    inline const vector<CGPoint>& positions_() const { return buffer->positions; } 
    inline const vector<CGFloat>& caretOffsets_() const { return buffer->caretOffsets; } 
    
    inline void _locateSingleGlyph(CFIndex index, CGFloat x, CGFloat y)
    {
        CGPoint point = CGPointMake(x,y);
        buffer->positions[index] = point;
    }
    
    inline void _locateGlyphsPropotionally(const CFRange range, CGFloat x, CGFloat y, CTFontOrientation orientation)
    {
        const vector<CGSize>& advances = advances_();
        if(orientation == kCTFontVerticalOrientation)
        {
            for(CFIndex i=range.location;i<range.location+range.length;i++)
            {
                _locateSingleGlyph(i, x, y);
                x += advances[i].height;
                y -= advances[i].width;
            }
        }
        else
        {
            for(CFIndex i=range.location;i<range.location+range.length;i++)
            {
                _locateSingleGlyph(i, x, y);
                x += advances[i].width;
                y -= advances[i].height;
            }
        }
    }
    
    inline CFIndex _findLastWeakBreak(CFRange range)
    {
        const vector<uint8>& characterAttribute = characterAttribute_();
        for(CFIndex i=range.location+range.length-1; i>=range.location; i--)
        {
            if(characterAttribute[i] & cWeakBreak)
                return i;
        }
        return kCFNotFound;
    }
    
    inline CFIndex _findHyphenation(CFRange range)
    {
        NSString *string = buffer->target->substring_in_range(range);
        CFRange limit = CFRangeMake(0, range.length-1);
        
        // 英語ベースと仮定しているが、ここはオプションで処理されるべきだろう
        CFLocaleRef locale = (CFLocaleRef)[[NSLocale alloc] initWithLocaleIdentifier: @"en_US"];
        CFIndex result = 
        CFStringGetHyphenationLocationBeforeIndex((CFStringRef)string,
                                                  limit.length,
                                                  limit,
                                                  0,
                                                  locale,
                                                  NULL);
        [(id)locale release];
        if(result != kCFNotFound)
            result += range.location-1; // ハイフン文字の手前でカットしたいので-1
        return result;
    }
    
    inline void fixCaretOffsets(CFRange lineRange)
    {
        const vector<CGPoint>& positions = positions_();
        const vector<CGSize>& advances = advances_();
        vector<CGFloat>& caretOffsets = buffer->caretOffsets;
        CFIndex i;
        CFIndex limit = lineRange.location+lineRange.length;
        for(i=0;i<lineRange.length;i++)
        {
            caretOffsets[i] = -positions[lineRange.location+i].y;
        }
        
        if(limit == 0)
            return;
        
        caretOffsets[i] = caretOffsets[i-1] + advances[limit-1].width;
    }
    
    inline void fixVerticalTranslates(CFRange range)
    {
        for(CFIndex i=range.location;i<range.location+range.length;i++)
        {
            CGPoint point = buffer->positions[i];
            point.x += buffer->translations[i].width;
            point.y += buffer->translations[i].height;
            buffer->positions[i] = point;
        }
    }
    
    inline void finalizeVerticalLine(CFRange lineRange)
    {
        this->fixCaretOffsets(lineRange);
        this->fixVerticalTranslates(lineRange);
    }
    
public:
    _GlyphLocatorBase(_GlyphBufferSpace *buf)
    {
        buffer = buf;
    }
};

#endif
