/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.ListActivity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.widget.SimpleAdapter;
import android.widget.Toast;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.utils.Log;
import org.routine_work.simple_battery_logger.util.SystemResourceUtils;

public class CsvFileViewerActivity extends ListActivity implements DBConstants
{

	/**
	 * Extra date as "yyyy-MM-dd"
	 */
	public static final String EXTRA_DATE = "date";
	private static final String LOG_TAG = "simple-battery-logger";
	private static final String[] MAPPING_FROM = new String[]
	{
		DBConstants.BatteryHistory.Columns.DATETIME,
		DBConstants.BatteryHistory.Columns.STATUS,
		DBConstants.BatteryHistory.Columns.PLUGGED,
		DBConstants.BatteryHistory.Columns.LEVEL,
		DBConstants.BatteryHistory.Columns.VOLTAGE,
		DBConstants.BatteryHistory.Columns.TEMPERATURE,
	};
	private static final int[] MAPPING_TO = new int[]
	{
		R.id.battery_history_datetime,
		R.id.battery_history_status,
		R.id.battery_history_plugged,
		R.id.battery_history_level,
		R.id.battery_history_voltage,
		R.id.battery_history_temperature,
	};
	private SimpleAdapter listAdapter;
	private List<Map<String, String>> dataList;

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();

		menuInflater.inflate(R.menu.csv_files_menu, menu);
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.send_menuitem:
				sendCsvFile();
				break;
			case R.id.preference_menuitem:
				Intent prefIntent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivity(prefIntent);
				break;
			case R.id.quit_menuitem:
				Log.d(LOG_TAG, "quit_menuitem is clicked.");
				DashboardActivity.quitApplication(this);
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}

		return result;
	}

	/**
	 * Called when the activity is first created.
	 */
	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "Hello");
		super.onCreate(savedInstanceState);
		setContentView(R.layout.battery_history_list_activity);

		dataList = new ArrayList<Map<String, String>>();
		listAdapter = new SimpleAdapter(this, dataList, R.layout.battery_history_list_item, MAPPING_FROM, MAPPING_TO);
		setListAdapter(listAdapter);

		loadCsvFile();

		Log.v(LOG_TAG, "Bye");
	}

	private void loadCsvFile()
	{
		Log.v(LOG_TAG, "Hello");

		dataList.clear();

		Uri dataUri = getIntent().getData();
		Log.d(LOG_TAG, "dataUri => " + dataUri);
		if (dataUri != null && "file".equals(dataUri.getScheme()))
		{
			String path = dataUri.getPath();
			Log.d(LOG_TAG, "path => " + path);
			File csvFile = new File(path);
			setTitle(csvFile.getName());

			try
			{
				FileInputStream fileInputStream = new FileInputStream(csvFile);
				try
				{
					InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream, "UTF-8");
					BufferedReader reader = new BufferedReader(inputStreamReader);
					String line = reader.readLine(); // header
					if (line != null)
					{
						while ((line = reader.readLine()) != null)
						{
							String[] split = line.split(",", 8);
							if (split.length == 8)
							{
								int level = Integer.parseInt(split[3]);
								int scale = Integer.parseInt(split[4]);
								int levelPercent = 100 * level / scale;
								Map<String, String> data = new HashMap<String, String>();
								data.put(DBConstants.BatteryHistory.Columns.DATETIME, split[0]);
								data.put(DBConstants.BatteryHistory.Columns.STATUS, split[1]);
								data.put(DBConstants.BatteryHistory.Columns.PLUGGED, split[5]);
								data.put(DBConstants.BatteryHistory.Columns.LEVEL, String.valueOf(levelPercent));
								data.put(DBConstants.BatteryHistory.Columns.VOLTAGE, split[6]);
								data.put(DBConstants.BatteryHistory.Columns.TEMPERATURE, split[7]);
								dataList.add(data);
							}
						}
					}
				}
				finally
				{
					fileInputStream.close();
				}
			}
			catch (UnsupportedEncodingException ex)
			{
				Log.e(LOG_TAG, "UTF-8 is not supported.", ex);
			}
			catch (IOException ex)
			{
				Log.e(LOG_TAG, "Parse CSV file failed. : path => " + path, ex);
			}
		}

		listAdapter.notifyDataSetChanged();

		Log.v(LOG_TAG, "Bye");
	}

	private void sendCsvFile()
	{
		Log.v(LOG_TAG, "Hello");

		Uri csvDataUri = getIntent().getData();
		Log.d(LOG_TAG, "csvFileUri => " + csvDataUri);
		if (csvDataUri != null && "file".equals(csvDataUri.getScheme()))
		{
			try
			{
				Intent intent = new Intent(Intent.ACTION_SEND);
				intent.setType("text/csv");
				intent.putExtra(Intent.EXTRA_STREAM, csvDataUri);
				startActivity(intent);
			}
			catch (ActivityNotFoundException e)
			{
				Log.e(LOG_TAG, "Failed to send a CSV file.", e);

				String message = getString(R.string.activity_not_found) + "\n" + csvDataUri;
				Toast.makeText(this, message, Toast.LENGTH_SHORT).show();
			}
		}

		Log.v(LOG_TAG, "Bye");
	}
}
