package kisscelltopng.kiss.io;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import kisscelltopng.kiss.types.KiSSCell;
import kisscelltopng.kiss.types.KiSSPalette;

import charactermanaj.model.io.ArchivedEntry;
import charactermanaj.model.io.ArchivedFile;

public class KiSSFile {

	protected ArchivedFile<? extends ArchivedEntry> archivedFile;
	
	protected KiSSCellLoader cellLoader;

	protected KiSSPaletteLoader paletteLoader;

	public KiSSFile(ArchivedFile<? extends ArchivedEntry> archivedFile) {
		this(archivedFile, new KiSSCellLoader(), new KiSSPaletteLoader());
	}

	public KiSSFile(ArchivedFile<? extends ArchivedEntry> archivedFile,
			KiSSCellLoader cellLoader, KiSSPaletteLoader paletteLoader) {
		if (archivedFile == null || cellLoader == null || paletteLoader == null) {
			throw new IllegalArgumentException();
		}
		this.archivedFile = archivedFile;
		this.cellLoader = cellLoader;
		this.paletteLoader = paletteLoader;
	}
	
	public ArchivedFile<? extends ArchivedEntry> getArchivedFile() {
		return archivedFile;
	}
	
	public void close() throws IOException {
		archivedFile.close();
	}
	
	public Collection<String> getCellNames() throws IOException {
		ArrayList<String> cellNames = new ArrayList<String>();
		for (ArchivedEntry entry : archivedFile) {
			String cellName = entry.getName();
			if (cellName.toLowerCase().endsWith(".cel")) {
				cellNames.add(cellName);
			}
		}
		Collections.sort(cellNames);
		return cellNames;
	}
	
	public KiSSCell getCell(String cellName, boolean pixelRequired) throws IOException {
		ArchivedEntry entry = archivedFile.getEntry(cellName);
		if (entry == null) {
			throw new FileNotFoundException(cellName);
		}
		KiSSCell cell;
		InputStream is = archivedFile.getInputStream(entry);
		try {
			cell = cellLoader.load(new BufferedInputStream(is), pixelRequired);
			
		} finally {
			is.close();
		}
		return cell;
	}
	
	public Collection<String> getPaletteNames() throws IOException {
		ArrayList<String> paletteNames = new ArrayList<String>();
		for (ArchivedEntry entry : archivedFile) {
			String paletteName = entry.getName();
			if (paletteName.toLowerCase().endsWith(".kcf")) {
				paletteNames.add(paletteName);
			}
		}
		Collections.sort(paletteNames);
		return paletteNames;
	}
	
	public KiSSPalette getPalette(String paletteName) throws IOException {
		ArchivedEntry entry = archivedFile.getEntry(paletteName);
		if (entry == null) {
			throw new FileNotFoundException(paletteName);
		}
		KiSSPalette palette;
		InputStream is = archivedFile.getInputStream(entry);
		try {
			palette = paletteLoader.load(new BufferedInputStream(is));
			
		} finally {
			is.close();
		}
		return palette;
	}
	
}

