package kisscelltopng.ui.model;

import java.awt.Dimension;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.TreeMap;

import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import kisscelltopng.kiss.types.KiSSPalette;
import kisscelltopng.ui.CellsTablePane;

/**
 * セル一覧テーブルモデル.<br> 
 * @author seraphy
 */
public class CellsTablePaneModel extends AbstractTableModel {

	private static final long serialVersionUID = 1958552539633248311L;

	/**
	 * 選択列
	 */
	public static final int COLUMN_SELECT = 0;
	
	/**
	 * ファイル名列(アーカイブ上のファイル識別子)
	 */
	public static final int COLUMN_FILENAME = 1;
	
	/**
	 * 編集可能な表示名
	 */
	public static final int COLUMN_CELLNAME = 2;
	
	/**
	 * 画像サイズ
	 */
	public static final int COLUMN_SIZE = 3;
	
	/**
	 * パレット
	 */
	public static final int COLUMN_PALETTE = 4;
	
	/**
	 * パレットグループ
	 */
	public static final int COLUMN_PALETTE_GROUP = 5;
	
	/**
	 * 出力先
	 */
	public static final int COLUMN_DEST_DIR = 6;

	/**
	 * ソートカラム
	 */
	private int sortColumn = COLUMN_FILENAME;
	
	/**
	 * ソートが昇順であるか?
	 */
	private boolean sortDirection = true;
	
	/**
	 * カラム名の配列
	 */
	private static final String[] COLUMNS;
	
	/**
	 * カラム幅の配列
	 */
	private static final Integer[] COLUMNS_WIDTH;
	
	
	/**
	 * パレット選択セルで使用するコンボボックス.
	 */
	private JComboBox paletteChooseCombobox = new JComboBox();
	
	/**
	 * 出力先ディレクトリ選択セルで使用するコンボボックス(編集可能)
	 */
	private JComboBox dirChooseComboBox = new JComboBox();

	
	/**
	 * パレットのリスト.<br>
	 */
	private List<KiSSPalette> palettes = Collections.emptyList();

	/**
	 * アイテムのリスト.<br>
	 */
	private ArrayList<CellsTableItem> items = new ArrayList<CellsTableItem>();
	
	/**
	 * クラスイニシャライザ.
	 */
	static {
		ResourceBundle res = ResourceBundle.getBundle(CellsTablePane.class.getName());
		
		ArrayList<String> names = new ArrayList<String>();
		for (String name : res.getString("column.names").split(",")) {
			names.add(name.trim());
		}
		
		ArrayList<Integer> widths = new ArrayList<Integer>();
		for (String name : res.getString("column.widths").split(",")) {
			widths.add(Integer.valueOf(name.trim()));
		}
		
		COLUMNS = names.toArray(new String[names.size()]);
		COLUMNS_WIDTH = widths.toArray(new Integer[widths.size()]);
	}
	
	
	public List<KiSSPalette> getPalettes() {
		return palettes;
	}
	
	/**
	 * パレットを設定する.<br>
	 * パレット列のセルエディタも更新される.<br>
	 * @param palettes
	 */
	public void setPalettes(List<KiSSPalette> palettes) {
		if (palettes == null) {
			palettes = Collections.emptyList();
		}
		this.palettes = Collections
				.unmodifiableList(new ArrayList<KiSSPalette>(palettes));

		paletteChooseCombobox.removeAllItems();
		for (KiSSPalette palette : palettes) {
			paletteChooseCombobox.addItem(palette);
		}

		fireTableDataChanged();
	}
	
	/**
	 * 出力先ディレクトリを設定し、
	 * サブフォルダをドロップダウンで選択可能にする.<br>
	 * @param destDir 出力先ディレクトリ
	 */
	public void setDestDir(File destDir) {
		ArrayList<String> dirs = new ArrayList<String>();
		if (destDir != null && destDir.isDirectory()) {
			for (File file : destDir.listFiles()) {
				if (file.isDirectory()) {
					dirs.add(file.getName());
				}
			}
		}
		Collections.sort(dirs);

		dirChooseComboBox.removeAllItems();
		for (String dir : dirs) {
			dirChooseComboBox.addItem(dir);
		}

		updateDestDir(true);
	}

	/**
	 * テーブルの列モデルを設定する.<br>
	 * @param columnModel テーブルのカラムモデル
	 */
	public void setupColumnModel(TableColumnModel columnModel) {
		for (int idx = 0; idx < COLUMNS_WIDTH.length; idx++) {
			TableColumn tc = columnModel.getColumn(idx);
			tc.setPreferredWidth(COLUMNS_WIDTH[idx]);
		}

		TableColumn tcPaletteGroup = columnModel
				.getColumn(CellsTablePaneModel.COLUMN_PALETTE_GROUP);
		tcPaletteGroup.setCellEditor(new DefaultCellEditor(
				new JComboBox(new Integer[] {0, 1, 2, 3, 4, 5, 6, 7, 8, 9})));
		
		TableColumn tcPalette = columnModel
				.getColumn(CellsTablePaneModel.COLUMN_PALETTE);
		tcPalette.setCellEditor(getPaletteCellEditor());
		
		TableColumn tcOutDir = columnModel
				.getColumn(CellsTablePaneModel.COLUMN_DEST_DIR);
		tcOutDir.setCellEditor(getDirCellEditor());
		
		if (palettes.isEmpty()) {
			tcPalette.setMinWidth(0);
			tcPaletteGroup.setMinWidth(0);
			tcPalette.setPreferredWidth(0);
			tcPaletteGroup.setPreferredWidth(0);
		}
	}
	
	public String[] getColumnNames() {
		ArrayList<String> columns = new ArrayList<String>();
		for (int idx = 0; idx < COLUMNS.length; idx++) {
			if (idx == COLUMN_PALETTE || idx == COLUMN_PALETTE_GROUP) {
				if (palettes.isEmpty()) {
					continue;
				}
			}
			String columnName = COLUMNS[idx];
			columns.add(columnName);
		}
		return columns.toArray(new String[columns.size()]);
	}
	
	public int getSortColumn() {
		return sortColumn;
	}
	
	public boolean getSortDirection() {
		return sortDirection;
	}
	
	public void sort(final int sortColumn, final boolean direction) {
		Comparator<CellsTableItem> comparator = new Comparator<CellsTableItem>() {
			public int compare(CellsTableItem o1, CellsTableItem o2) {
				int ret = 0;
				
				// ソートキーに従いソートする.
				switch (sortColumn) {
				case COLUMN_SELECT:
					ret = (o1.isSelected() ? 1 : 0) - (o2.isSelected() ? 1 : 0);
					break;
				
				case COLUMN_FILENAME:
					break;
				
				case COLUMN_CELLNAME:
					ret = o1.getCellName().compareTo(o2.getCellName());
					break;

				case COLUMN_SIZE:
					Dimension s1 = o1.getSize();
					Dimension s2 = o2.getSize();
					ret = (s1.height * s1.width) - (s2.height * s2.width);
					break;
					
				case COLUMN_PALETTE:
					KiSSPalette palette1 = o1.getPalette();
					KiSSPalette palette2 = o2.getPalette();
					ret = (palette1 == null ? "" : palette1.toString()).compareTo(palette2 == null ? "" : palette2.toString());
					if (ret == 0) {
						ret = o1.getPaletteGroup() - o2.getPaletteGroup();
					}
					break;
				
				case COLUMN_PALETTE_GROUP:
					ret = o1.getPaletteGroup() - o2.getPaletteGroup();
					break;
				
				case COLUMN_DEST_DIR:
					ret = o1.getOutdir().compareTo(o2.getOutdir());
					break;
				}

				// 同位の場合はファイル名でソートする.
				if (ret == 0) {
					ret = o1.getFileName().compareTo(o2.getFileName());
				}
				return direction ? ret : -ret;
			}
		};
		Collections.sort(items, comparator);
		this.sortColumn = sortColumn;
		this.sortDirection = direction;
		fireTableDataChanged();
	}
	
	public boolean isAllItemSelected() {
		for (CellsTableItem item : items) {
			if (!item.isSelected()) {
				return false;
			}
		}
		return !items.isEmpty();
	}
	
	public void selectAll(boolean selected) {
		for (CellsTableItem item : items) {
			item.setSelected(selected);
		}
		fireTableDataChanged();
	}
	
	public void clear() {
		items.clear();
		fireTableDataChanged();
	}
	
	public void add(CellsTableItem item) {
		if (item == null) {
			throw new IllegalArgumentException();
		}
		items.add(item);
		int row = items.size() - 1;
		fireTableRowsInserted(row, row);
	}
	
	public void add(Collection<CellsTableItem> items) {
		if (items == null || items.isEmpty()) {
			return;
		}
		int startRow = this.items.size();
		this.items.addAll(items);
		int endRow = this.items.size() - 1;
		fireTableRowsInserted(startRow, endRow);
	}
	
	public CellsTableItem get(int row) {
		return items.get(row);
	}

	public int getColumnCount() {
		return COLUMNS.length;
	}
	
	@Override
	public Class<?> getColumnClass(int columnIndex) {
		if (columnIndex == COLUMN_SELECT) {
			return Boolean.class;
		}
		if (columnIndex == COLUMN_PALETTE){
			return KiSSPalette.class;
		}
		if (columnIndex == COLUMN_PALETTE_GROUP) {
			return Integer.class;
		}
		return String.class;
	}
	
	@Override
	public String getColumnName(int column) {
		return COLUMNS[column];
	}
	
	public int getRowCount() {
		return items.size();
	}
	
	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		if (columnIndex == COLUMN_SELECT ||
			columnIndex == COLUMN_CELLNAME ||
			columnIndex == COLUMN_DEST_DIR) {
			return true;
		}
		if (columnIndex == COLUMN_PALETTE ||
			columnIndex == COLUMN_PALETTE_GROUP) {

			CellsTableItem item = items.get(rowIndex);
			return item.isIndexedColor();
		}
		return false;
	}
	
	public Object getValueAt(int rowIndex, int columnIndex) {
		CellsTableItem item = items.get(rowIndex);
		switch (columnIndex) {
		case COLUMN_SELECT:
			return item.isSelected();
			
		case COLUMN_FILENAME:
			return item.getFileName();
			
		case COLUMN_CELLNAME:
			return item.getCellName();
			
		case COLUMN_SIZE:
			return toString(item.getSize());

		case COLUMN_PALETTE:
			if (item.isIndexedColor()) {
				return item.getPalette();
			}
			return null;
			
		case COLUMN_PALETTE_GROUP:
			if (item.isIndexedColor()) {
				return item.getPaletteGroup();
			}
			return null;
			
		case COLUMN_DEST_DIR:
			return item.getOutdir();
			
		default:
		}
		// フォールバック用 (正常であれば、ここにはこない)
		return "(" + rowIndex + "," + columnIndex + ")";
	}
	
	protected String toString(Dimension siz) {
		return String.format("%d x %d", (int) siz.getWidth(), (int) siz.getHeight());
	}
	
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		boolean notified = false;
		CellsTableItem item = items.get(rowIndex);
		if (columnIndex == COLUMN_SELECT) {
			Boolean sw = (Boolean) aValue;
			item.setSelected(sw != null && sw.booleanValue());

		} else if (columnIndex == COLUMN_CELLNAME) {
			String value = (String) aValue;
			if (value != null && value.trim().length() > 0) {
				item.setCellName(value.trim());
			}
			
		} else if (columnIndex == COLUMN_PALETTE) {
			KiSSPalette palette = (KiSSPalette) aValue;
			if (palette != null) {
				item.setPalette(palette);
			}

		} else if (columnIndex == COLUMN_PALETTE_GROUP) {
			Integer value = (Integer) aValue;
			if (value != null && value.intValue() >= 0 && value.intValue() < 10) {
				item.setPaletteGroup(value.intValue());
			}
		
		} else if (columnIndex == COLUMN_DEST_DIR) {
			String destDir = (String) aValue;
			if (destDir == null || destDir.trim().length() == 0) {
				item.setOutdir(null);

			} else {
				item.setOutdir(destDir.trim());
				notified = updateDestDir(false);
			}
		}
		
		if ( !notified) {
			fireTableCellUpdated(rowIndex, columnIndex);
		}
	}
	
	/**
	 * 出力対象とする全てのアイテムの出力先ディレクトリの一覧をドロップダウンの候補に設定します.<br>
	 * コンボボックスのドロップダウンリストが変更された場合は変更結果がtrueとなり、テーブルモデルの変更通知が行われます.<br>
	 * (変更フラグが初期状態で設定されている場合は常にモデルの通知が行われます。)
	 * @param modified 変更フラグの初期状態
	 * @return 変更結果
	 */
	protected boolean updateDestDir(boolean modified) {
		TreeMap<String, Integer> dirs = new TreeMap<String, Integer>();
		for (CellsTableItem item : items) {
			if ( !item.isSelected()) {
				continue;
			}
			String outdir = item.getOutdir();
			if (outdir != null && outdir.length() > 0) {
				dirs.put(outdir, Integer.valueOf(0));
			}
		}

		int mx = dirChooseComboBox.getItemCount();
		for (int idx = 0; idx < mx; idx++) {
			String outdir = (String) dirChooseComboBox.getItemAt(idx);
			dirs.put(outdir, Integer.valueOf(1));
		}
		
		for (Map.Entry<String, Integer> entry : dirs.entrySet()) {
			String outdir = entry.getKey();
			Integer val = entry.getValue();
			if (val.equals(Integer.valueOf(0))) {
				dirChooseComboBox.addItem(outdir);
				modified = true;
			}
		}
		if (modified) {
			fireTableDataChanged();
		}
		return modified;
	}
	
	/**
	 * パレット列のセルエディタ.<br>
	 * このモデルのパレットリストの状態を反映する.<br>
	 * @return セルエディタ.<br>
	 */
	public TableCellEditor getPaletteCellEditor() {
		return new DefaultCellEditor(paletteChooseCombobox);
	}

	/**
	 * 出力先フォルダ用コンボボックスつきセルエディタ.<br>
	 * @return セルエディタ
	 */
	public TableCellEditor getDirCellEditor() {
		dirChooseComboBox.setEditable(true);
		return new DefaultCellEditor(dirChooseComboBox);
	}
	
}