package charactermanaj.model.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.AbstractCollection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * JARファイルをアーカイブとするアーカイブファクトリ.<br>
 * jarファイルの解析にはjdk標準のものを用いる.<br>
 * @author seraphy
 */
public class JarArchivedFileFactory extends ArchivedFileFactory {

	@Override
	public JarArchivedFile openArchive(File file) throws IOException {
		return new JarArchivedFile(file);
	}
	
}

class JarArchivedFile extends AbstractCollection<JarArchivedEntry> implements ArchivedFile<JarArchivedEntry> {

	protected File archivedFile;
	
	protected JarFile jarFile;
	
	protected HashMap<String, JarArchivedEntry> entries = new HashMap<String, JarArchivedEntry>();
	
	public JarArchivedFile(File archivedFile) throws IOException {
		if (archivedFile == null) {
			throw new IllegalArgumentException();
		}
		this.archivedFile = archivedFile;
		this.jarFile = new JarFile(archivedFile);
		
		enumFiles();
	}
	
	protected void enumFiles() throws IOException {
		Enumeration<? extends JarEntry> enm = jarFile.entries();
		while (enm.hasMoreElements()) {
			JarEntry header = (JarEntry) enm.nextElement();
			JarArchivedEntry entry = new JarArchivedEntry(header);
			entries.put(entry.getName(), entry);
		}
	}
	
	public String getName() {
		return archivedFile.getPath().replace('\\', '/');
	}
	
	public ArchivedEntry getEntry(String name) {
		return entries.get(name);
	}
	
	public InputStream getInputStream(ArchivedEntry entry) throws IOException {
		if (entry == null) {
			throw new IllegalArgumentException();
		}
		JarArchivedEntry lhaEntry = entries.get(entry.getName());
		if (lhaEntry == null) {
			return null;
		}
		return jarFile.getInputStream(lhaEntry.getJarEntry());
	}
	
	public void close() throws IOException {
		jarFile.close();
	}
	
	@Override
	public Iterator<JarArchivedEntry> iterator() {
		return entries.values().iterator();
	}
	
	@Override
	public int size() {
		return entries.size();
	}
}

class JarArchivedEntry implements ArchivedEntry {
	
	protected JarEntry jarEntry;
	
	protected String name;
	
	public JarArchivedEntry(JarEntry zipEntry) {
		if (zipEntry == null) {
			throw new IllegalArgumentException();
		}
		this.jarEntry = zipEntry;
		this.name = zipEntry.getName().replace('\\', '/');
	}
	
	public JarEntry getJarEntry() {
		return jarEntry;
	}

	public String getName() {
		return name;
	}
	
	public long getTime() {
		return jarEntry.getTime();
	}
	
	public boolean isDirectory() {
		return getName().endsWith("/");
	}
}
