package kisscelltopng.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.EventListenerList;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;

import kisscelltopng.ui.model.CellsTableItem;
import kisscelltopng.ui.model.CellsTablePaneModel;


/**
 * セル選択テーブル.<br>
 * セルの選択が変更されるか、選択中のセルにかかわるモデルの変更があった場合、および
 * 保存ボタンが押された場合にアクションイベントが発生する.<br>
 * @author seraphy
 */
public class CellsTablePane extends JPanel {

	private static final long serialVersionUID = -1L;

	/**
	 * 選択変更アクション.<br>
	 */
	public static final String ACTION_SELCHANGE = "selChange";

	/**
	 * 保存アクション.<br>
	 */
	public static final String ACTION_SAVE = "save";


	private EventListenerList eventListeners = new EventListenerList();
	

	public CellsTablePane() {
		this(new CellsTablePaneModel());
	}
	
	private JTable table;
	
	private AbstractAction actSave = new AbstractAction() {
		private static final long serialVersionUID = 1L;

		public void actionPerformed(ActionEvent e) {
			onSave();
		}
	};

	public CellsTablePane(final CellsTablePaneModel model) {
		table = new JTable(model);
		
		setupColumn();

		model.addTableModelListener(new TableModelListener() {
			public void tableChanged(TableModelEvent e) {
				// イベント発生時、モデル上の範囲外を示す場合があるので、その場合はモデル内に限定する.
				int st = Math.max(0, e.getFirstRow());
				int en = Math.min(model.getRowCount() - 1, e.getLastRow());
				int cur = table.getSelectedRow();
				if (cur >= st && cur <= en) {
					onSelChange();
				}
			}
		});

		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent e) {
				if ( !e.getValueIsAdjusting()) {
					onSelChange();
				}
			}
		});
		
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		table.setShowGrid(true);
		table.setGridColor(Color.GRAY);
		table.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);
		table.setShowVerticalLines(true);
		table.setShowHorizontalLines(true);
		table.setPreferredScrollableViewportSize(new Dimension(300, 300));
		
		setLayout(new BorderLayout());
		add(new JScrollPane(table), BorderLayout.CENTER);
		
		JTableHeader tableHeader = table.getTableHeader();
		tableHeader.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				evaluatePopup(e);
			}
			@Override
			public void mouseReleased(MouseEvent e) {
				evaluatePopup(e);
			}
			private void evaluatePopup(MouseEvent e) {
			       if (e.isPopupTrigger()) {
			    	   showHeaderPopupMenu(e);
			    	   e.consume();
			       }
			    }
			}
		);
		
		JPanel btnPanel = new JPanel(new BorderLayout());
		btnPanel.setBorder(BorderFactory.createEmptyBorder(3, 3, 5, 5));
		actSave.putValue(Action.NAME, "出力する");
		btnPanel.add(new JButton(actSave), BorderLayout.EAST);
		add(btnPanel, BorderLayout.SOUTH);
	}
	
	/**
	 * ソート用メニュー項目をJPopupMenuまたはJMenuに追加するための実装側が
	 * ソート項目のJMenuItemを受け取るためのインターフェイス.<br>
	 * @author seraphy
	 */
	public interface MenuItemCallback {
		
		void addMenuItem(JMenuItem menuItem);
		
	}

	/**
	 * ソートメニュー項目を必要な数だけコールバックに返します.<br>
	 * @param callback ソートメニュー項目を受け取るコールバック
	 */
	public void addSortMenuItem(MenuItemCallback callback) {
		if (callback == null) {
			throw new IllegalArgumentException();
		}
		final CellsTablePaneModel model = getModel();
		String[] columnNames = model.getColumnNames();
		final int currentSortColumnIdx = model.getSortColumn();
		final boolean currentSortDirection = model.getSortDirection();
		
		for (int idx = 0; idx < columnNames.length; idx++) {
			final int columnIdx = idx;
			JCheckBoxMenuItem item = new JCheckBoxMenuItem(columnNames[columnIdx]);
			if (columnIdx == currentSortColumnIdx) {
				item.setSelected(true);
			}
			item.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					model.sort(columnIdx, currentSortColumnIdx != columnIdx ? true : !currentSortDirection);
				}
			});
			callback.addMenuItem(item);
		}
	}
	
	/**
	 * ホップアップメニューでソートメニューを表示します.<br>
	 * マウスイベントの発生地点にホップアップメニューが表示されます.<br>
	 * @param e マウスイベント
	 */
	protected void showHeaderPopupMenu(MouseEvent e) {
		final JPopupMenu popupMenu = new JPopupMenu("SORT");
		addSortMenuItem(new MenuItemCallback() {
			public void addMenuItem(JMenuItem menuItem) {
				popupMenu.add(menuItem);
			}
		});
		popupMenu.show(this, e.getX(), e.getY());
	}
	
	public void setupColumn() {
		getModel().setupColumnModel(table.getColumnModel());
	}
	
	public CellsTablePaneModel getModel() {
		return (CellsTablePaneModel) table.getModel();
	}
	
	public void addActionListener(ActionListener l) {
		if (l != null) {
			eventListeners.add(ActionListener.class, l);
		}
	}
	
	public void removeActionListener(ActionListener l) {
		if (l != null) {
			eventListeners.remove(ActionListener.class, l);
		}
	}
	
	protected void fireActionPerformed(ActionEvent e) {
		for (ActionListener l : eventListeners.getListeners(ActionListener.class)) {
			l.actionPerformed(e);
		}
	}
	
	protected void onSelChange() {
		ActionEvent e = new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
				ACTION_SELCHANGE, System.currentTimeMillis(), 0);
		fireActionPerformed(e);
	}
	
	public CellsTableItem getSelectedItem() {
		int selIdx = table.getSelectedRow();
		if (selIdx < 0) {
			return null;
		}
		return getModel().get(selIdx);
	}
	
	protected void onSave() {
		ActionEvent e = new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
				ACTION_SAVE, System.currentTimeMillis(), 0);
		fireActionPerformed(e);
	}
	
	public void setEnabledSave(boolean enabled) {
		actSave.setEnabled(enabled);
	}
	
	public boolean isEnabledSave() {
		return actSave.isEnabled();
	}
}
