package kisscelltopng.ui;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.ResourceBundle;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.event.EventListenerList;
import javax.swing.filechooser.FileFilter;

import kisscelltopng.ui.model.DirChoosePanelModel;
import kisscelltopng.ui.model.DirChoosePanelModel.SourceType;

public class DirChoosePanel extends JPanel implements PropertyChangeListener {
	
	private static final long serialVersionUID = 1L;
	
	public static final String ACTION_LOAD = "load";
	
	public static final String ACTION_RESET = "reset";
	
	private EventListenerList eventListeners = new EventListenerList();

	private DirChoosePanelModel model;
	
	private JTextField txtSourceFile = new JTextField();
	
	private JTextField txtDestinationDir = new JTextField();
	
	private JComboBox cmbSourceType = new JComboBox(DirChoosePanelModel.SourceType.values());
	
	private AbstractAction actLoad = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			ActionEvent ee = new ActionEvent(DirChoosePanel.this,
					ActionEvent.ACTION_PERFORMED, ACTION_LOAD, e.getWhen(),
					e.getModifiers());
			fireActionEvent(ee);
		}
	};
	
	private AbstractAction actReset = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			ActionEvent ee = new ActionEvent(DirChoosePanel.this,
					ActionEvent.ACTION_PERFORMED, ACTION_RESET, e.getWhen(),
					e.getModifiers());
			fireActionEvent(ee);
		}
	};
	
	private AbstractAction actBrowseKiSSFile = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onBrowseSourceFile();
		}
	};

	private AbstractAction actBrowseKiSSDir = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onBrowseSourceDir();
		}
	};

	private AbstractAction actBrowseOutDir = new AbstractAction() {
		private static final long serialVersionUID = 1L;
		public void actionPerformed(ActionEvent e) {
			onBrowseDestinationDir();
		}
	};

	public DirChoosePanel() {
		this(new DirChoosePanelModel());
	}

	public DirChoosePanel(DirChoosePanelModel model) {
		initComponent();
		setModel(model);
	}
	
	public final void setModel(DirChoosePanelModel model) {
		if (model == null) {
			throw new IllegalArgumentException();
		}
		if (this.model != null) {
			this.model.removePropertyChangeListener(this);
		}
		this.model = model;
		this.model.addPropertyChangeListener(this);
		applyModel();
	}
	
	public DirChoosePanelModel getModel() {
		return model;
	}
	
	public void propertyChange(PropertyChangeEvent evt) {
		applyModel();
	}
	
	public void addActionListener(ActionListener listener) {
		if (listener != null) {
			eventListeners.add(ActionListener.class, listener);
		}
	}
	
	public void removeActionListener(ActionListener listener) {
		if (listener != null) {
			eventListeners.remove(ActionListener.class, listener);
		}
	}
	
	public ActionListener[] getActionListeners() {
		return eventListeners.getListeners(ActionListener.class);
	}
	
	protected void fireActionEvent(ActionEvent e) {
		if (e == null) {
			throw new IllegalArgumentException();
		}
		for (ActionListener listener : getActionListeners()) {
			listener.actionPerformed(e);
		}
	}
	
	private void initComponent() {
		final ResourceBundle res = ResourceBundle.getBundle(getClass().getName());

		setLayout(new GridBagLayout());
		GridBagConstraints gbc = new GridBagConstraints();
		
		setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3),
				BorderFactory.createTitledBorder(res.getString("group.caption"))));

		gbc.ipadx = 0;
		gbc.ipady = 0;
		gbc.gridheight = 1;
		gbc.gridwidth = 1;
		gbc.insets = new Insets(0, 0, 0, 0);
		gbc.weightx = 0.;
		gbc.weighty = 0.;
		gbc.anchor = GridBagConstraints.CENTER;
		gbc.fill = GridBagConstraints.BOTH;
		
		int y = 0;
		
		gbc.gridx = 0;
		gbc.gridy = y;
		gbc.insets = new Insets(0, 3, 0, 3);
		add(new JLabel(res.getString("label.sourceType")), gbc);

		gbc.gridx = 1;
		gbc.gridy = y;
		gbc.weightx = 0.;
		gbc.insets = new Insets(0, 0, 0, 0);
		add(cmbSourceType, gbc);

		y++;
		
		gbc.gridx = 0;
		gbc.gridy = y;
		gbc.weightx = 0.;
		gbc.insets = new Insets(0, 3, 0, 3);
		add(new JLabel(res.getString("label.sourceFile")), gbc);

		gbc.gridx = 1;
		gbc.gridy = y;
		gbc.weightx = 1.;
		gbc.insets = new Insets(0, 0, 0, 0);
		add(txtSourceFile, gbc);

		gbc.gridx = 2;
		gbc.gridy = y;
		gbc.weightx = 0.;
		actBrowseKiSSDir.putValue(Action.NAME, res.getString("button.KiSSDir"));
		JButton btnBrowseKiSSDir = new JButton(actBrowseKiSSDir);
		add(btnBrowseKiSSDir, gbc);

		gbc.gridx = 3;
		gbc.gridy = y;
		actBrowseKiSSFile.putValue(Action.NAME, res.getString("button.KiSSFile"));
		JButton btnBrowseKiSSFile = new JButton(actBrowseKiSSFile);
		add(btnBrowseKiSSFile, gbc);
		
		y++;
		
		gbc.gridx = 0;
		gbc.gridy = y;
		gbc.insets = new Insets(0, 3, 0, 3);
		add(new JLabel(res.getString("label.outdir")), gbc);

		gbc.gridx = 1;
		gbc.gridy = y;
		gbc.weightx = 1.;
		gbc.insets = new Insets(0, 0, 0, 0);
		add(txtDestinationDir, gbc);

		gbc.gridx = 2;
		gbc.gridy = y;
		gbc.weightx = 0.;
		actBrowseOutDir.putValue(Action.NAME, res.getString("button.outdir"));
		JButton btnBrowseOutDir = new JButton(actBrowseOutDir);
		add(btnBrowseOutDir, gbc);
		
		y++;
		
		gbc.gridx = 0;
		gbc.gridy = y;
		gbc.gridwidth = 4;
		gbc.insets = new Insets(3, 5, 3, 5);
		add(new JSeparator(JSeparator.HORIZONTAL), gbc);
		
		y++;
		
		gbc.gridx = 2;
		gbc.gridy = y;
		gbc.gridwidth = 1;
		gbc.insets = new Insets(0, 0, 0, 0);
		actLoad.putValue(Action.NAME, res.getString("button.load"));
		JButton btnLoad = new JButton(actLoad);
		add(btnLoad, gbc);

		gbc.gridx = 3;
		gbc.gridy = y;
		actReset.putValue(Action.NAME, res.getString("button.reset"));
		JButton btnReset = new JButton(actReset);
		add(btnReset, gbc);
		
		cmbSourceType.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				model.setSourceType((SourceType) cmbSourceType.getSelectedItem());
			}
		});
		cmbSourceType.setRenderer(new DefaultListCellRenderer() {
			private static final long serialVersionUID = 1L;

			@Override
			public Component getListCellRendererComponent(JList list,
					Object value, int index, boolean isSelected,
					boolean cellHasFocus) {
				SourceType sourceType = (SourceType) value;
				String resourceName = sourceType.getResourceName();
				String displayName = res.getString(resourceName);
				return super.getListCellRendererComponent(list, displayName, index, isSelected,
						cellHasFocus);
			}
		});

		txtDestinationDir.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				onDestDirTextChange();
			}
		});
		txtSourceFile.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				onSourceFileTextChange();
			}
		});
		txtDestinationDir.addFocusListener(new FocusAdapter() {
			@Override
			public void focusLost(FocusEvent e) {
				onDestDirTextChange();
			}
		});
		txtSourceFile.addFocusListener(new FocusAdapter() {
			@Override
			public void focusLost(FocusEvent e) {
				onSourceFileTextChange();
			}
		});
	}
	
	protected void onSourceFileTextChange() {
		String sourceFile = txtSourceFile.getText().trim();
		if (sourceFile.length() > 0) {
			model.setSourceFile(new File(sourceFile));

		} else {
			model.setSourceFile(null);
		}
	}

	protected void onDestDirTextChange() {
		String destinationDir = txtDestinationDir.getText().trim();
		if (destinationDir.length() > 0) {
			model.setDestDir(new File(destinationDir));

		} else {
			model.setDestDir(null);
		}
	}
	
	protected void applyModel() {
		cmbSourceType.setSelectedItem(model.getSourceType());

		File kissFile = model.getSourceFile();
		File outDir = model.getDestDir();
		
		txtSourceFile.setText(kissFile == null ? "" : kissFile.getPath());
		txtDestinationDir.setText(outDir == null ? "" : outDir.getPath());
		
		boolean lock = model.isLock();
		txtSourceFile.setEditable(!lock);
		
		actLoad.setEnabled(!lock);
		actReset.setEnabled(lock);
		
		actBrowseKiSSDir.setEnabled(!lock);
		actBrowseKiSSFile.setEnabled(!lock);
		
		cmbSourceType.setEnabled(!lock);
	}
	
	protected void onBrowseSourceDir() {
		String currentSelection = txtSourceFile.getText();
		JFileChooser fileChooser = new JFileChooser(currentSelection);
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		int ret = fileChooser.showOpenDialog(this);
		if (ret != JFileChooser.APPROVE_OPTION) {
			return;
		}
		txtSourceFile.setText(fileChooser.getSelectedFile().getPath());
		model.setSourceFile(new File(txtSourceFile.getText()));
	}
	
	protected void onBrowseSourceFile() {
		browseSourceFile();
	}
	

	public boolean browseSourceFile() {
		String currentSelection = txtSourceFile.getText();
		JFileChooser fileChooser = new JFileChooser(currentSelection);
		fileChooser.addChoosableFileFilter(new FileFilter() {
			@Override
			public boolean accept(File f) {
				if (f.isDirectory()) {
					return true;
				}
				String name = f.getName().toLowerCase();
				return name.endsWith(".lzh") || name.endsWith(".lha") || name.endsWith(".zip");
			}
			@Override
			public String getDescription() {
				return "KiSS Archive (*.lzh;*.lha;*.zip)";
			}
		});
		int ret = fileChooser.showOpenDialog(this);
		if (ret != JFileChooser.APPROVE_OPTION) {
			return false;
		}
		txtSourceFile.setText(fileChooser.getSelectedFile().getPath());
		model.setSourceFile(new File(txtSourceFile.getText()));
		return true;
	}
	
	protected void onBrowseDestinationDir() {
		String currentSelection = txtDestinationDir.getText();
		JFileChooser fileChooser = new JFileChooser(currentSelection);
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		int ret = fileChooser.showSaveDialog(this);
		if (ret != JFileChooser.APPROVE_OPTION) {
			return;
		}
		txtDestinationDir.setText(fileChooser.getSelectedFile().getPath());
		model.setDestDir(new File(txtDestinationDir.getText()));
	}
	
}
