/*
 * TextKeyMatcherApp.java
 */

package textkeymatcher;

import textkeymatcher.ui.TextKeyMatcherView;
import java.awt.Font;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import org.jdesktop.application.Application;
import org.jdesktop.application.SingleFrameApplication;
import textkeymatcher.util.log.AWTExceptionLoggingHandler;
import textkeymatcher.util.log.ApplicationLoggerConfigurator;

/**
 * The main class of the application.
 */
public class TextKeyMatcherApp extends SingleFrameApplication {

    private static final Logger logger = Logger.getLogger(TextKeyMatcherApp.class.getName());
    
    /**
     * ロガーの初期化.<br>
     * 失敗しても継続する.<br>
     */
    private static void initLogger() {
        // ローカルファイルシステム上のユーザ定義ディレクトリから
        // ログの設定を読み取る.(OSにより、設定ファイルの位置が異なることに注意)
        ApplicationLoggerConfigurator.configure();

        // SwingのEDT内の例外ハンドラの設定 (ロギングするだけ)
        AWTExceptionLoggingHandler.initSystemProperty();
    }

    /**
     * At startup create and show the main frame of the application.
     */
    @Override protected void startup() {
        // ロガーの初期化
        initLogger();

        // UIマネージャの初期化
        try {
            setupUIManager();

        } catch (Exception ex) {
            // 失敗しても実害はないと思われるため継続する.
            logger.log(Level.CONFIG, "setupUIManager Failed.", ex);
        }
        
        // メインウィンドウの構築と表示
        show(new TextKeyMatcherView(this));
    }

    /**
     * This method is to initialize the specified window by injecting resources.
     * Windows shown in our application come fully initialized from the GUI
     * builder, so this additional configuration is not needed.
     */
    @Override protected void configureWindow(java.awt.Window root) {
    }

    @Override
    protected void shutdown() {
        logger.log(Level.INFO, "shutdown.");
        super.shutdown();
    }
    

    /**
     * A convenient static getter for the application instance.
     * @return the instance of TextKeyMatcherApp
     */
    public static TextKeyMatcherApp getApplication() {
        return Application.getInstance(TextKeyMatcherApp.class);
    }

    /**
     * UIをセットアップする.
     * @throws Exception いろいろな失敗
     */
    private static void setupUIManager() throws Exception {
        System.setProperty("apple.laf.useScreenMenuBar", "true");

        // 実行プラットフォームのネイティブな外観にする.
        UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

        // JSpliderのvalueを非表示 (GTKではデフォルトで有効のため)
        UIManager.put("Slider.paintValue", Boolean.FALSE);

        // JTextAreaの既定フォントを固定幅から、標準テキストと同じフォントに変更.
        // (Linuxなどで固定幅フォントでは日本語フォントを持っていないため。)
        Object textFieldFontUI = UIManager.get("TextField.font");
        if (textFieldFontUI == null) {
            // もし無ければダイアログUIフォントを使う.(これは日本語をサポートするであろう。)
            textFieldFontUI = new UIDefaults.ProxyLazyValue(
                    "javax.swing.plaf.FontUIResource", null, new Object[]{
                        "dialog", Integer.valueOf(Font.PLAIN), Integer.valueOf(12)});
        }
        UIManager.put("TextArea.font", textFieldFontUI);
    }

    /**
     * Main method launching the application.
     */
    public static void main(String[] args) {
        launch(TextKeyMatcherApp.class, args);
    }
}
