/*
 * CAN/LIN/UART/PORT Checker for RL78/F14
 *
 * Target: QB-R5F10PPJ-TB (RL78/F14, 100pin, 256KB ROM, 20KB RAM)
 * Author: Yasushi Tanaka
 *
 * [ UARThCo ]
 */

#include <string.h>
#include "common.h"
#include "iodefine.h"
#include "cpu.h"
#include "dtc.h"
#include "log.h"
#include "uart.h"

/*
 * UART
 * |[gE_NV`
 */
#define UART0_PORT_REDUCTION	(1)
									/* 0:P15(TxD0),P16(RxD0) 1:P62(TxD0),P61(RxD0) */
#define UART1_PORT_REDUCTION	(1)
									/* 0:P12(TxD0),P11(RxD0) 1:P74(TxD0),P75(RxD) */

/*
 * UART
 * Mobt@TCY`(VO)
 */
#define UART_RX_BUFFER_SIZE		(0x80)
									/* Mobt@TCY(VOobt@) */
#define UART_RX_MARGIN_SIZE		(2)
									/* DTCI[o[}[W */
#define UART_RX_DTC_LEN			((u1)(UART_RX_BUFFER_SIZE - UART_RX_MARGIN_SIZE))
									/* DTCɐݒ肷M */

/*
 * UART
 * MOobt@TCY
 */
#define UART_RING_BUFFER_SIZE	(0xc0)
									/* MOobt@TCY */
#define UART_RING_BUFFER_CH		(2)
									/* MOobt@`l */

/*
 * SFRݒl(PER0WX^)
 */
#define RL78F_PER0_SAU0EN		((u1)(0x04))
									/* VAEACEjbg0L */
#define RL78F_PER0_SAU1EN		((u1)(0x08))
									/* VAEACEjbg1L */

/*
 * SFRݒl(STmLWX^)
 */
#define RL78F_ST0L_ST00			(u1)((0x01))
									/* AC0,`l0~ */
#define RL78F_ST0L_ST01			((u1)(0x02))
									/* AC0,`l1~ */
#define RL78F_ST1L_ST10			((u1)(0x01))
									/* AC1,`l0~ */
#define RL78F_ST1L_ST11			((u1)(0x02))
									/* AC1,`l1~ */

/*
 * SFRݒl(MKmHWX^)
 */
#define RL78F_MK0H_STMK0		((u1)(0x02))
 									/* INTST0}XN */
#define RL78F_MK0H_SRMK0		((u1)(0x04))
									/* INTSR0}XN */
#define RL78F_MK1H_STMK1		((u1)(0x20))
									/* INTST1}XN */
#define RL78F_MK1H_SRMK1		((u1)(0x40))
									/* INTSR1}XN */

/*
 * SFRݒl(PRmHWX^)
 */
#define RL78F_PR00H_STPR00		((u1)(0x02))
								 	/* INTST0荞ݗD揇(bit0) */
#define RL78F_PR10H_STPR10		((u1)(0x02))
								 	/* INTST0荞ݗD揇(bit1) */
#define RL78F_PR00H_SRPR00		((u1)(0x04))
 									/* INTSR0荞ݗD揇(bit0) */
#define RL78F_PR10H_SRPR10		((u1)(0x04))
 									/* INTSR0荞ݗD揇(bit1) */
#define RL78F_PR01H_STPR01		((u1)(0x20))
								 	/* INTST1荞ݗD揇(bit0) */
#define RL78F_PR11H_STPR11		((u1)(0x20))
								 	/* INTST1荞ݗD揇(bit1) */
#define RL78F_PR01H_SRPR01		((u1)(0x40))
 									/* INTSR1荞ݗD揇(bit0) */
#define RL78F_PR11H_SRPR11		((u1)(0x40))
 									/* INTSR1荞ݗD揇(bit1) */

/*
 * SFRݒl(SPSmLWX^)
 */
#define RL78F_SPSL_PRSM0_MASK	((u1)(0x0f))
									/* PRSm00-PRSm03̃rbg}XN */
#define RL78F_SPSL_PRSM1_FCLK	((u1)(0x00))
									/* PRSm10-PRSm13ݒl(fCLK) */

/*
 * SFRݒl(SMRmnWX^)
 */
#define RL78F_SMR_CKS1			((u2)(0x8000))
									/* NbNƂCKm1I */
#define RL78F_SMR_STS1			((u2)(0x0100))
									/* RXDq[q̗LGbWgKƂ(Mp) */
#define RL78F_SMR_RESERVED		((u2)(0x0020))
									/* \rbg(KZbg) */
#define RL78F_SMR_MD1			((u2)(0x0002))
									/* UART[h */
#define RL78F_SMR_MD0			((u2)(0x0001))
									/* obt@󂫂Ŋ荞(Mp) */

/*
 * SFRݒl(SCRmnWX^)
 */
#define RL78F_SCR_TXE			((u2)(0x8000))
									/* M@\:L */
#define RL78F_SCR_RXE			((u2)(0x4000))
									/* M@\:L */
#define RL78F_SCR_DIR			((u2)(0x0080))
									/* LSBt@[Xg */
#define RL78F_SCR_SLC			((u2)(0x0010))
									/* Xgbvrbg:1rbg */
#define RL78F_SCR_DLS			((u2)(0x0007))
									/* f[^:8rbg */

/*
 * SFRݒl(SDRmnWX^)
 */
#define RL78F_SDR_128K			((u2)(0xf800))
									/* 128000bps */
#define RL78F_SDR_230K			((u2)(0x8800))
									/* 231884bps */
#define RL78F_SDR_460K			((u2)(0x4400))
									/* 457143bps */
#define RL78F_SDR_500K			((u2)(0x3e00))
									/* 500000bps */
#define RL78F_SDR_921K			((u2)(0x2000))
									/* 941177bps */
#define RL78F_SDR_12M			((u2)(0x1800))
									/* 1230769bps */

/*
 * SFRݒl(NFEN0WX^)
 */
#define RL78F_NFEN0_SNFEN00		((u1)(0x01))
									/* jbg0mCYtB^L */
#define RL78F_NFEN0_SNFEN10		((u1)(0x04))
									/* jbg1mCYtB^L */

/*
 * SFRݒl(SIRmnLWX^)
 */
#define RL78F_SIRL_OVCT			((u1)(0x01))
									/* I[o[G[ NA */
#define RL78F_SIRL_PECT			((u1)(0x02))
									/* peBG[ NA */
#define RL78F_SIRL_FECT			((u1)(0x04))
									/* t[~OG[ NA */

/*
 * SFRݒl(SSRmnLWX^)
 */
#define RL78F_SSRL_OVF			((u1)(0x01))
									/* I[o[EG[otO */
#define RL78F_SSRL_PEF			((u1)(0x02))
									/* peBEG[otO */
#define RL78F_SSRL_FEF			((u1)(0x04))
									/* t[~OEG[otO */
#define RL78F_SSRL_BFF			((u1)(0x20))
									/* obt@EWX^ԕ\tO */
#define RL78F_SSRL_TSF			((u1)(0x40))
									/* ʐMԕ\tO */

/*
 * SFRݒl(SOmWX^)
 */
#define RL78F_SO0_SO00			((u2)(0x0001))
									/* SO00Hxo */
#define RL78F_SO0_SO01			((u2)(0x0002))
									/* SO01Hxo */
#define RL78F_SO0_CKO00			((u2)(0x0100))
									/* CKO00Hxo */
#define RL78F_SO0_CKO01			((u2)(0x0200))
									/* CKO01Hxo */
#define RL78F_SO1_SO10			((u2)(0x0001))
									/* SO10Hxo */
#define RL78F_SO1_SO11			((u2)(0x0002))
									/* SO11Hxo */
#define RL78F_SO1_CKO10			((u2)(0x0100))
									/* CKO10Hxo */
#define RL78F_SO1_CKO11			((u2)(0x0200))
									/* CKO11Hxo */

/*
 * SFRݒl(SOEmLWX^)
 */
#define RL78F_SOE0L_DISABLE		((u1)(0x00))
									/* AC0Mo͋֎~ */
#define RL78F_SOE0L_SOE00		((u1)(0x01))
									/* AC0,`l0Mo͋ */
#define RL78F_SOE0L_SOE01		((u1)(0x02))
									/* AC0,`l1Mo͋ */
#define RL78F_SOE1L_DISABLE		((u1)(0x00))
									/* AC1Mo͋֎~ */
#define RL78F_SOE1L_SOE10		((u1)(0x01))
									/* AC1,`l0Mo͋ */
#define RL78F_SOE1L_SOE11		((u1)(0x02))
									/* AC1,`l1Mo͋ */

/*
 * SFRݒl(SOLmLWX^)
 */
#define RL78F_SOLL_DISABLE		((u1)(0x00))
									/* ʏo */

/*
 * SFRݒl(PIOR4WX^)
 */
#define RL78F_PIOR4_PIOR40		((u1)(0x01))
									/* P15(TXD0),P16(RXD0)P62(TXD0),P61(RXD0) */
#define RL78F_PIOR4_PIOR42		((u1)(0x04))
									/* P12(TXD1),P11(RXD1)P74(TXD1),P75(RXD1) */

/*
 * SFRݒl(POM1WX^)
 * |[gE_NV̏ꍇɎgp
 */
#define RL78F_POM1_AND0			((u1)(0x9f))
									/* P15(TXD0),P16(RXD0)Ƃɒʏo̓[h */
#define RL78F_POM1_AND1			((u1)(0xf9))
									/* P11(RXD1),P12(TXD1)Ƃɒʏo̓[h */

/*
 * SFRݒl(PM1WX^)
 * |[gE_NV̏ꍇɎgp
 */
#define RL78F_PM1_AND0			((u1)(0xdf))
									/* P15(TXD0)͏o͕ */
#define RL78F_PM1_OR0			((u1)(0x40))
									/* P16(RXD0)͓͕ */
#define RL78F_PM1_AND1			((u1)(0xfb))
									/* P12(TXD1)͏o͕ */
#define RL78F_PM1_OR1			((u1)(0x02))
									/* P11(RXD1)͓͕ */

/*
 * SFRݒl(POM6WX^)
 * |[gE_NVL̏ꍇɎgp
 */
#define RL78F_POM6_AND0			((u1)(0xf9))
									/* P62(TXD0),P61(RXD0)Ƃɒʏo̓[h */

/*
 * SFRݒl(PM6WX^)
 * |[gE_NVL̏ꍇɎgp
 */
#define RL78F_PM6_AND0			((u1)(0xfb))
									/* P62(TXD0)͏o͕ */
#define RL78F_PM6_OR0			((u1)(0x02))
									/* P61(RXD0)͓͕ */

/*
 * SFRݒl(PM7WX^)
 * |[gE_NVL̏ꍇɎgp
 */
#define RL78F_PM7_AND1			((u1)(0xef))
									/* P74(TXD1)͏o͕ */
#define RL78F_PM7_OR1			((u1)(0x20))
									/* P75(RXD1)͓͕ */

/*
 * SFRݒl(PMC7WX^)
 * |[gE_NVL̏ꍇɎgp(RL78/F14̂ݑ݂SFR)
 */
#ifdef R5F10PPJ
#define RL78F_PMC7_AND1			((u1)(0xef))
									/* P74(TXD1)̓AiO͂ł͂Ȃ */
#endif /* R5F10PPJ */

/*
 * SFRݒl(P1WX^)
 * |[gE_NV̏ꍇɎgp
 */
#define RL78F_P1_OR0			((u1)(0x20))
									/* P15(TXD0)Hx */
#define RL78F_P1_OR1			((u1)(0x04))
									/* P12(TXD1)Hx */

/*
 * SFRݒl(P6WX^)
 * |[gE_NVL̏ꍇɎgp
 */
#define RL78F_P6_OR0			((u1)(0x04))
									/* P62(TXD0)Hx */

/*
 * SFRݒl(P7WX^)
 * |[gE_NVL̏ꍇɎgp
 */
#define RL78F_P7_OR1			((u1)(0x10))
									/* P74(TXD0)Hx */

/*
 * SFRݒl(SSmLWX^)
 */
#define RL78F_SS0L_SS00			(0x01)
									/* AC0,`l0Jn */
#define RL78F_SS0L_SS01			(0x02)
									/* AC0,`l1Jn */
#define RL78F_SS1L_SS10			(0x01)
									/* AC1,`l0Jn */
#define RL78F_SS1L_SS11			(0x02)
									/* AC1,`l1Jn */

/*
 * UART
 * M_uobt@rbg`
 */
#define UART_RX_CH0_BUF0		(0x01)
									/* `l0,obt@0ŎM */
#define UART_RX_CH0_BUF1		(0x02)
									/* `l0,obt@1ŎM */
#define UART_RX_CH1_BUF0		(0x10)
									/* `l1,obt@0ŎM */
#define UART_RX_CH1_BUF1		(0x20)
									/* `l1,obt@1ŎM */

/*
 * UART
 * M_uobt@tO
 */
static SADDR u1 uart_rx_flag;
									/* M_uobt@tO */

/*
 * UART
 * M_uobt@ԍ`
 */
#define UART_RX_BUF0			((u1)(0))
									/* Mobt@0(`l0,obt@0) */
#define UART_RX_BUF1			((u1)(1))
									/* Mobt@1(`l0,obt@1) */
#define UART_RX_BUF2			((u1)(2))
									/* Mobt@2(`l1,obt@0) */
#define UART_RX_BUF3			((u1)(3))
									/* Mobt@3(`l1,obt@1) */
#define UART_RX_BUF_MAX			((u1)(4))
									/* Mobt@̌ */

/*
 * UART
 * Mobt@
 */
static u1 uart_rx_buffer[UART_RX_BUF_MAX][UART_RX_BUFFER_SIZE];
									/* Mobt@ */

/*
 * UART
 * Oobt@
 */
static u1 uart_ring_buffer[UART_RING_BUFFER_CH][UART_RING_BUFFER_SIZE];
									/* Oobt@ */
static SADDR u1 uart_ring_readp[UART_RING_BUFFER_CH];
									/* Oobt@ǂݍ݃|Cg */
static SADDR u1 uart_ring_writep[UART_RING_BUFFER_CH];
									/* Oobt@݃|Cg */
static SADDR u1 uart_ring_num[UART_RING_BUFFER_CH];
									/* Oobt@L */

/*
 * UART
 * MDTCCTobt@
 */
static SADDR u1 uart_rx_dtcct[UART_RING_BUFFER_CH];
									/* MDTCCTobt@ */

/*
 * UART
 * vf[^
 */
static uart_stat uart_count[UART_RING_BUFFER_CH];
									/* JEgf[^ */

/*
 * UART
 * {[[gSDRWX^ݒl擾
 *
 * baudrate: 128 or 230 or 460 or 500 or 921 or 1200
 * 擾łȂꍇ0Ԃ
 */
static u2 uart_get_sdr(u2 baudrate)
{
	u2 sdr;

	/*  */
	sdr = 0;

	switch (baudrate)
	{
		/* 128kbps */
		case 128:
			sdr = RL78F_SDR_128K;
			break;

		/* 230kbps */
		case 230:
			sdr = RL78F_SDR_230K;
			break;

		/* 460kbps */
		case 460:
			sdr = RL78F_SDR_460K;
			break;

		/* 500kbps */
		case 500:
			sdr = RL78F_SDR_500K;
			break;

		/* 921kbps */
		case 921:
			sdr = RL78F_SDR_921K;
			break;

		/* 1.2Mbps */
		case 1200:
			sdr = RL78F_SDR_12M;
			break;

		/* ̑(`̃{[[g) */
		default:
			break;
	}

	return sdr;
}

/*
 * UART
 * 
 */
void uart_init(u1 ch, u2 baudrate)
{
	u2 sdr;

	/* ̃`l̏ꍇAׂĂ̎M_uobt@tO */
	uart_rx_flag = 0;

	/* Oobt@ */
	uart_ring_readp[ch] = 0;
	uart_ring_writep[ch] = 0;
	uart_ring_num[ch] = 0;

	/* MDTCCTobt@ */
	uart_rx_dtcct[ch] = 0;

	/* vf[^ */
	memset(&uart_count[ch], 0, sizeof(uart_count[ch]));

	/* `l */
	if (0 == ch)
	{
		/* ӃCl[uEWX^0ŃVAEACEjbg0Lɂ */
		PER0 |= RL78F_PER0_SAU0EN;

		/* M荞݋yюM荞݂}XN */
		MK0H |= (RL78F_MK0H_STMK0 | RL78F_MK0H_SRMK0);

		/* 荞ݗD揇ʂx1ɐݒ */
		PR00H &= (u1)(~(RL78F_PR00H_STPR00 | RL78F_PR00H_SRPR00));
		PR10H |= (u1)(RL78F_PR10H_STPR10 | RL78F_PR10H_SRPR10);

		/* M`lyюM`l̓֎~ */
		ST0L = (RL78F_ST0L_ST00 | RL78F_ST0L_ST01);

		/* CK1CLKƂfCLKI(PRSm10-PRMSm13gp) */
		SPS0L = (u1)((SPS0L & RL78F_SPSL_PRSM0_MASK) | RL78F_SPSL_PRSM1_FCLK);

		/* 샂[hƂUARTANbNƂCK1CLKI */
		/* MRL78F_SMR_MD0rbg𗧂Ă */
		SMR00 = (   RL78F_SMR_CKS1 |
					RL78F_SMR_RESERVED |
					RL78F_SMR_MD1 |
					RL78F_SMR_MD0 );
		/* MRL78F_SMR_STS1rbg𗧂Ă */
		SMR01 = (   RL78F_SMR_CKS1 |
					RL78F_SMR_STS1 |
					RL78F_SMR_RESERVED |
					RL78F_SMR_MD1 );

		/* f[^8rbgAXgbvrbg1rbgApeBȂ */
		/* MRL78F_SCR_TXELɂ */
		SCR00 = (   RL78F_SCR_TXE |
					RL78F_SCR_DIR |
					RL78F_SCR_SLC |
					RL78F_SCR_DLS );
		/* MRL78F_SCR_RXELɂ */
		SCR01 = (   RL78F_SCR_RXE |
					RL78F_SCR_DIR |
					RL78F_SCR_SLC |
					RL78F_SCR_DLS );

		/* {[[g */
		sdr = uart_get_sdr(baudrate);
		if (0 != sdr)
		{
			/* SDRݒl擾ł */
			SDR00 = sdr;
			SDR01 = sdr;

			/* mCYtB^L */
			NFEN0 |= RL78F_NFEN0_SNFEN00;

			/* MG[NA */
			SIR00L = (  RL78F_SIRL_OVCT |
						RL78F_SIRL_PECT |
						RL78F_SIRL_FECT);
			SIR01L = (  RL78F_SIRL_OVCT |
						RL78F_SIRL_PECT |
						RL78F_SIRL_FECT);

			/* SOóACKOo͂VAʐM~ɕύX */
			SOE0L = RL78F_SOE0L_DISABLE;

			/* o͂𔽓]Ȃ */
			SOL0L = RL78F_SOLL_DISABLE;

			/* SOóACKOo͂Hx */
			SO0 = ( RL78F_SO0_SO00 |
					RL78F_SO0_SO01 |
					RL78F_SO0_CKO00 |
					RL78F_SO0_CKO01 );

			/* SOóACKOo͂VAʐMɕύX */
			SOE0L = ( RL78F_SOE0L_SOE00 | RL78F_SOE0L_SOE01 );

#if UART0_PORT_REDUCTION == 0
			/* |[gE_NVݒ */
			PIOR4 &= (u1)(~RL78F_PIOR4_PIOR40);

			/* p|[gݒ(P15:TXD0, P16:RXD0) */
			P1 |= RL78F_P1_OR0;
			POM1 &= RL78F_POM1_AND0;
			PM1 = (u1)((PM1 & RL78F_PM1_AND0) | RL78F_PM1_OR0);
#else
			/* |[gE_NVݒ */
			PIOR4 |= RL78F_PIOR4_PIOR40;

			/* p|[gݒ(P62:TXD0, P61:RXD0) */
			P6 |= RL78F_P6_OR0;
			POM6 &= RL78F_POM6_AND0;
			PM6 = (u1)((PM6 & RL78F_PM6_AND0) | RL78F_PM6_OR0);
#endif /* UART0_PORT_REDUCTION == 0 */

			/* M`lyюM`l̓Jn */
			SS0L = (RL78F_SS0L_SS00 | RL78F_SS0L_SS01);
		}
	}
	else
	{
		/* ӃCl[uEWX^0ŃVAEACEjbg1Lɂ */
		PER0 |= RL78F_PER0_SAU1EN;

		/* M荞݋yюM荞݂}XN */
		MK1H |= (RL78F_MK1H_STMK1 | RL78F_MK1H_SRMK1);

		/* 荞ݗD揇ʂx1ɐݒ */
		PR01H &= (u1)(~(RL78F_PR01H_STPR01 | RL78F_PR01H_SRPR01));
		PR11H |= (u1)(RL78F_PR11H_STPR11 | RL78F_PR11H_SRPR11);

		/* M`lyюM`l̓֎~ */
		ST1L = (RL78F_ST1L_ST10 | RL78F_ST1L_ST11);

		/* CK1CLKƂfCLKI(PRSm10-PRMSm13gp) */
		SPS1L = (u1)((SPS1L & RL78F_SPSL_PRSM0_MASK) | RL78F_SPSL_PRSM1_FCLK);

		/* 샂[hƂUARTANbNƂCK1CLKI */
		/* MRL78F_SMR_MD0rbg𗧂Ă */
		SMR10 = (   RL78F_SMR_CKS1 |
					RL78F_SMR_RESERVED |
					RL78F_SMR_MD1 |
					RL78F_SMR_MD0 );
		/* MRL78F_SMR_STS1rbg𗧂Ă */
		SMR11 = (   RL78F_SMR_CKS1 |
					RL78F_SMR_STS1 |
					RL78F_SMR_RESERVED |
					RL78F_SMR_MD1 );

		/* MRL78F_SCR_TXELɂ */
		SCR10 = (   RL78F_SCR_TXE |
					RL78F_SCR_DIR |
					RL78F_SCR_SLC |
					RL78F_SCR_DLS );
		/* MRL78F_SCR_RXELɂ */
		SCR11 = (   RL78F_SCR_RXE |
					RL78F_SCR_DIR |
					RL78F_SCR_SLC |
					RL78F_SCR_DLS );

		/* {[[g */
		sdr = uart_get_sdr(baudrate);
		if (0 != sdr)
		{
			/* SDRݒl擾ł */
			SDR10 = sdr;
			SDR11 = sdr;

			/* mCYtB^L */
			NFEN0 |= RL78F_NFEN0_SNFEN10;

			/* MG[NA */
			SIR10L = (  RL78F_SIRL_OVCT |
						RL78F_SIRL_PECT |
						RL78F_SIRL_FECT);
			SIR11L = (  RL78F_SIRL_OVCT |
						RL78F_SIRL_PECT |
						RL78F_SIRL_FECT);

			/* SOóACKOo͂VAʐM~ɕύX */
			SOE1L = RL78F_SOE1L_DISABLE;

			/* o͂𔽓]Ȃ */
			SOL1L = RL78F_SOLL_DISABLE;

			/* SOóACKOo͂Hx */
			SO1 = ( RL78F_SO1_SO10 |
					RL78F_SO1_SO11 |
					RL78F_SO1_CKO10 |
					RL78F_SO1_CKO11 );

			/* SOóACKOo͂VAʐMɕύX */
			SOE1L = ( RL78F_SOE1L_SOE10 | RL78F_SOE1L_SOE11 );

#if UART1_PORT_REDUCTION == 0
			/* |[gE_NVݒ */
			PIOR4 &= (u1)(~RL78F_PIOR4_PIOR42);

			/* p|[gݒ(P12:TXD1, P11:RXD1) */
			P1 |= RL78F_P1_OR1;
			POM1 &= RL78F_POM1_AND1;
			PM1 = (u1)((PM1 & RL78F_PM1_AND1) | RL78F_PM1_OR1);
#else
			/* |[gE_NVݒ */
			PIOR4 |= RL78F_PIOR4_PIOR42;

			/* p|[gݒ(P74:TXD1, P75:TXD1) */
			P7 |= RL78F_P7_OR1;
			PM7 = (u1)((PM7 & RL78F_PM7_AND1) | RL78F_PM7_OR1);
#ifdef R5F10PPJ
			PMC7 &= RL78F_PMC7_AND1;
#endif /* R5F10PPJ */
#endif /* UART1_PORT_REDUCTION == 0 */

			/* M`lyюM`l̓Jn */
			SS1L = (RL78F_SS1L_SS10 | RL78F_SS1L_SS11);
		}
	}
}

/*
 * UART
 * MREADY`FbN
 *
 * READYłU1_TRUEԂ
 */
static u1 uart_tx_is_ready(u1 ch)
{
	u1 ret;
	u1 bff;

	/* DTCfB`FbN */
	ret = dtc_tx_is_ready(ch);

	/* Mobt@󂫏󋵂擾 */
	if (ch == 0)
	{
		bff = (u1)(SSR00L & RL78F_SSRL_BFF);
	}
	else
	{
		bff = (u1)(SSR10L & RL78F_SSRL_BFF);
	}

	/* dtcU1_TRUEBFF0ȂU1_TRUEԂ */
	if (U1_TRUE == ret)
	{
		if (0 != bff)
		{
			/* DTCREADYAMobt@BUSY */
			ret = U1_FALSE;
		}
	}

	return ret;
}

/*
 * UART
 * DTCMJn
 *
 * 荞݋֎~ԂŌĂ΂
 */
static void uart_tx_dtc(u1 ch, u1* buf, u1 len)
{
	/* JEgAbv */
	uart_count[ch].tx_cnt[0] += len;
	if (uart_count[ch].tx_cnt[0] < len)
	{
		/* オ肵Ă */
		uart_count[ch].tx_cnt[1]++;
	}

	/* `l */
	if (0 == ch)
	{
		/* 2oCgȏ̑MɂDTCKv */
		if (1 != len)
		{
			/* DTCUARTMJn */
			dtc_tx_start(0, &buf[1], len - 1);
		}

		/* ŏ̃f[^Zbg(DTCZbgAbvɍsƂKv) */
		SDR00L = buf[0];

		/* 荞݃}XN */
		MK0H &= (u1)(~RL78F_MK0H_STMK0);
	}
	else
	{
		/* 2oCgȏ̑MɂDTCKv */
		if (1 != len)
		{
			/* DTCUARTMJn */
			dtc_tx_start(1, &buf[1], len - 1);
		}

		/* ŏ̃f[^Zbg(DTCZbgAbvɍsƂKv) */
		SDR10L = buf[0];

		/* 荞݃}XN */
		MK1H &= (u1)(~RL78F_MK1H_STMK1);
	}
}

/*
 * UART
 * MJn
 *
 * Mobt@RAM̈܂ROM~[̈ɔzu邱
 */
u1 uart_tx_start(u1 ch, u1* buf, u1 len)
{
	u1 psw;
	u1 ready;
	u1 ret;

	/* ߂l0ŏ */
	ret = 0;

	/* OX`FbN */
	if (0 != len)
	{
		/* MREADY`FbN */
		ready = uart_tx_is_ready(ch);

		/* MREADYȂ瑗MJn */
		if (U1_TRUE == ready)
		{
			/* 荞݋֎~ */
			psw = cpu_di();

			/* DTCMJn */
			uart_tx_dtc(ch, buf, len);

			/* ߂llenƂ */
			ret = len;

			/* 荞ݕA */
			cpu_ei(psw);
		}
	}

	/* MJnłoCgԂ */
	return ret;
}

/*
 * UART
 * M荞
 *
 * DTC]ƁA̍ۂɐݒ肵ŏIf[^VtgWX^Ɉړۂ2񔭐
 */
void uart_tx_isr(u1 ch)
{
	u1 ready;

	/* JEgAbv */
	uart_count[ch].tx_isr++;

	/* MREADY`FbN */
	ready = uart_tx_is_ready(ch);

	/* MREADYȂ烍OƒʐMp[T֒ʒm */
	if (U1_TRUE == ready)
	{
		log_isr(ch);
	}
}

/*
 * UART
 * MJn
 *
 * 荞݋֎~ԂŌĂ΂
 */
void uart_rx_start(u1 ch)
{
	/* `l */
	if (0 == ch)
	{
		/* _uobt@tOݒ */
		uart_rx_flag &= (u1)(~UART_RX_CH0_BUF1);
		uart_rx_flag |= UART_RX_CH0_BUF0;

		/* DTCTTobt@ */
		uart_rx_dtcct[0] = UART_RX_DTC_LEN;

		/* `l0,obt@0ŎMX^[g */
		dtc_rx_start(0, &uart_rx_buffer[UART_RX_BUF0][0], UART_RX_DTC_LEN);

		/* 荞݃}XN */
		MK0H &= (u1)(~RL78F_MK0H_SRMK0);
	}
	else
	{
		/* _uobt@tOݒ */
		uart_rx_flag &= (u1)(~UART_RX_CH1_BUF1);
		uart_rx_flag |= UART_RX_CH1_BUF0;

		/* DTCTTobt@ */
		uart_rx_dtcct[1] = UART_RX_DTC_LEN;

		/* `l0,obt@0ŎMX^[g */
		dtc_rx_start(1, &uart_rx_buffer[UART_RX_BUF2][0], UART_RX_DTC_LEN);

		/* 荞݃}XN */
		MK1H &= (u1)(~RL78F_MK1H_SRMK1);
	}
}

/*
 * UART
 * MOobt@擾
 */
u1 uart_ring_get(u1 ch)
{
	return uart_ring_num[ch];
}

/*
 * UART
 * MOobt@擾
 */
u1 uart_ring_deque(u1 ch, u1* dst, u1 len1)
{
	u1  psw;
	u1  len2;
	u1  loop;
	u1* src;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* len1numő傫 */
	if (uart_ring_num[ch] < len1)
	{
		len1 = uart_ring_num[ch];
	}

	/* len2̔Ɛݒ */
	if ((uart_ring_readp[ch] + len1) > UART_RING_BUFFER_SIZE)
	{
		/* len2͕Kv */
		len2 = UART_RING_BUFFER_SIZE - (uart_ring_readp[ch] + len1);
		len1 -= len2;
	}
	else
	{
		/* len2͕sv */
		len2 = 0;
	}

	/* len1̔ */
	if (0 != len1)
	{
		/* \[X|C^ݒ */
		src = &uart_ring_buffer[ch][uart_ring_readp[ch]];

		/* f[^Rs[ */
		for (loop=0; loop < len1; loop++)
		{
			*dst++ = *src++;
		}

		/* readp̍XV */
		uart_ring_readp[ch] += len1;
		if (UART_RING_BUFFER_SIZE == uart_ring_readp[ch])
		{
			uart_ring_readp[ch] = 0;
		}

		/* num̍XV */
		uart_ring_num[ch] -= len1;
	}

	/* len2̔ */
	if (0 != len2)
	{
		/* f[^Rs[ */
		for (loop=0; loop < len2; loop++)
		{
			*dst++ = *src++;
		}

		/* readp̍XV */
		uart_ring_readp[ch] = len2;

		/* num̍XV */
		uart_ring_num[ch] -= len2;
	}

	/* 荞ݕA */
	cpu_ei(psw);

	return (u1)(len1 + len2);
}

/*
 * UART
 * MOobt@֑}
 */
static void uart_ring_enque(u1 ch, u1* src, u1 len1)
{
	u1  psw;
	u1  loop;
	u1  len2;
	u1* dst;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* JEgAbv */
	uart_count[ch].rx_cnt[0] += len1;
	if (uart_count[ch].rx_cnt[0] < len1)
	{
		/* オ肵Ă */
		uart_count[ch].rx_cnt[1]++;
	}

	/* len1̃I[o[t[ */
	if ((uart_ring_num[ch] + len1) > UART_RING_BUFFER_SIZE)
	{
		len1 = UART_RING_BUFFER_SIZE - uart_ring_num[ch];
	}

	/* len2̔Ɛݒ */
	if ((uart_ring_writep[ch] + len1) > UART_RING_BUFFER_SIZE)
	{
		/* len2͕Kv */
		len2 = UART_RING_BUFFER_SIZE - (uart_ring_writep[ch] + len1);
		len1 -= len2;
	}
	else
	{
		/* len2͕sv */
		len2 = 0;
	}

	/* len1Rs[ */
	dst = &uart_ring_buffer[ch][uart_ring_writep[ch]];
	for (loop=0; loop < len1; loop++)
	{
		*dst++ = *src++;
	}

	/* writep̍XV */
	uart_ring_writep[ch] += len1;
	if (UART_RING_BUFFER_SIZE == uart_ring_writep[ch])
	{
		uart_ring_writep[ch] = 0;
		dst = &uart_ring_buffer[ch][0];
	}

	/* num̍XV */
	uart_ring_num[ch] += len1;

	/* len2̔ */
	if (0 != len2)
	{
		/* len2Rs[ */
		for (loop=0; loop < len2; loop++)
		{
			*dst++ = *src++;
		}

		/* writep̍XV */
		uart_ring_writep[ch] = len2;

		/* num̍XV */
		uart_ring_num[ch] += len2;
	}

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * UART
 * MOobt@XV
 */
static void uart_ring_update(u1 ch)
{
	u1  psw;
	u1  flag;
	u1  now;
	u1  prev;
	u1  len;
	u1* ptr;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* DTC猻݂̏擾 */
	dtc_rx_get_dtcct(ch, &now);

	/* ODTCCT擾 */
	prev = uart_rx_dtcct[ch];

	/* ݂̏ɍXV */
	uart_rx_dtcct[ch] = now;

	/* _uobt@tO擾 */
	flag = uart_rx_flag;

	/* 荞ݕA */
	cpu_ei(psw);

	/* ݂ƑO񂪈قȂĂ΃Oobt@֑} */
	if (now != prev)
	{
		/* prev-nowɂAVɎMoCg𓾂 */
		len = (u1)(prev - now);

		/* flagAݑΏۂƂĂMobt@𓾂 */
		if (0 == ch)
		{
			if (0 != (flag & UART_RX_CH0_BUF0))
			{
				/* `l0, obt@0ŎM */
				ptr = uart_rx_buffer[UART_RX_BUF0];
			}
			else
			{
				/* `l0, obt@1ŎM */
				ptr = uart_rx_buffer[UART_RX_BUF1];
			}
		}
		else
		{
			if (0 != (flag & UART_RX_CH1_BUF0))
			{
				/* `l1, obt@0ŎM */
				ptr = uart_rx_buffer[UART_RX_BUF2];
			}
			else
			{
				/* `l1, obt@1ŎM */
				ptr = uart_rx_buffer[UART_RX_BUF3];
			}
		}

		/* Oobt@֑} */
		uart_ring_enque(ch, &ptr[UART_RX_DTC_LEN - prev], len);
	}
}

/*
 * UART
 * MG[`FbN
 */
static void uart_chk_error(u1 ch)
{
	u1 ssr;

	/* SSR01܂SSR11擾 */
	if (0 == ch)
	{
		ssr = SSR01;
	}
	else
	{
		ssr = SSR11;
	}

	/* G[3ނ݂̂Ƀ}XN */
	ssr &= (RL78F_SSRL_OVF | RL78F_SSRL_PEF | RL78F_SSRL_FEF);

	/* G[`FbN */
	if (0 != ssr)
	{
		/* I[o[EG[otO */
		if (0 != (ssr & RL78F_SSRL_OVF))
		{
			/* JEgAbv */
			uart_count[ch].ov_err++;
		}

		/* peBEG[otO */
		if (0 != (ssr & RL78F_SSRL_PEF))
		{
			/* JEgAbv */
			uart_count[ch].pe_err++;
		}

		/* t[~OEG[otO */
		if (0 != (ssr & RL78F_SSRL_FEF))
		{
			/* JEgAbv */
			uart_count[ch].fe_err++;
		}

		/* SIR01܂SIR11ɏ݁AG[NA */
		if (0 == ch)
		{
			SIR01 = ssr;
		}
		else
		{
			SIR11 = ssr;
		}
	}
}

/*
 * UART
 * ACh
 */
void uart_idle(u1 ch)
{
	/* MG[`FbN */
	uart_chk_error(ch);

	/* MOobt@XV */
	uart_ring_update(ch);
}

/*
 * UART
 * vf[^擾
 */
void uart_get_stat(u1 ch, uart_stat* buf)
{
	u1 psw;

	/* 荞݋֎~ */
	psw = cpu_di();

	/* \̂̈ꊇ */
	*buf = uart_count[ch];

	/* 荞ݕA */
	cpu_ei(psw);
}

/*
 * UART
 * M荞
 */
void uart_rx_isr(u1 ch)
{
	/* JEgAbv */
	uart_count[ch].rx_isr++;

	/* `l */
	if (0 == ch)
	{
		/* Mf[^Oobt@֑} */
		uart_ring_update(0);

		/* ǂ̃obt@ŎM肷 */
		if (0 != (uart_rx_flag & UART_RX_CH0_BUF0))
		{
			/* `l0,obt@1ŎMX^[g */
			dtc_rx_start(0, &uart_rx_buffer[UART_RX_BUF1][0], UART_RX_DTC_LEN);
		}
		else
		{
			/* `l0,obt@0ŎMX^[g */
			dtc_rx_start(0, &uart_rx_buffer[UART_RX_BUF0][0], UART_RX_DTC_LEN);
		}

		/* DTCTTobt@ */
		uart_rx_dtcct[0] = UART_RX_DTC_LEN;

		/* _uobt@tO𔽓] */
		uart_rx_flag ^= (u1)(UART_RX_CH0_BUF0 | UART_RX_CH0_BUF1);
	}
	else
	{
		/* Mf[^Oobt@֑} */
		uart_ring_update(1);

		/* ǂ̃obt@ŎM肷 */
		if (0 != (uart_rx_flag & UART_RX_CH1_BUF0))
		{
			/* `l1,obt@1ŎMX^[g */
			dtc_rx_start(1, &uart_rx_buffer[UART_RX_BUF3][0], UART_RX_DTC_LEN);
		}
		else
		{
		/* `l1,obt@0ŎMX^[g */
			dtc_rx_start(1, &uart_rx_buffer[UART_RX_BUF2][0], UART_RX_DTC_LEN);
		}

		/* DTCTTobt@ */
		uart_rx_dtcct[1] = UART_RX_DTC_LEN;

		/* _uobt@tO𔽓] */
		uart_rx_flag ^= (u1)(UART_RX_CH1_BUF0 | UART_RX_CH1_BUF1);
	}
}
