package project.batch;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.sql.Timestamp;

import common.db.dao.DaoUtil;

import core.config.Factory;

/**
 * バッチユーティリティ
 * @author Tadashi Nakayama
 */
public final class BatchUtil {

	/** バージョン */
	public static final String ITEM_VERSION = "Version";
	/** 作成者ID */
	public static final String ITEM_CREATE_USER_ID = "InsertId";
	/** 作成日時 */
	public static final String ITEM_CREATE_DATE_TIME = "InsertDatetime";
	/** 更新者ID */
	public static final String ITEM_UPDATE_USER_ID = "UpdateId";
	/** 更新日時 */
	public static final String ITEM_UPDATE_DATE_TIME = "UpdateDatetime";

	/**
	 * コンストラクタ
	 */
	private BatchUtil() {
		throw new AssertionError();
	}

	/**
	 * 作成者情報設定
	 * @param obj オブジェクト
	 * @param uid 作成者
	 * @param dateTime 作成日時
	 */
	public static void setCreateInfo(final Serializable obj,
			final String uid, final Timestamp dateTime) {
		setObject(obj, ITEM_CREATE_USER_ID, uid);
		setObject(obj, ITEM_CREATE_DATE_TIME, dateTime);
	}

	/**
	 * 更新者情報設定
	 * @param obj オブジェクト
	 * @param uid 更新者
	 * @param dateTime 更新日時
	 */
	public static void setUpdateInfo(final Serializable obj,
			final String uid, final Timestamp dateTime) {
		setObject(obj, ITEM_UPDATE_USER_ID, uid);
		setObject(obj, ITEM_UPDATE_DATE_TIME, dateTime);
	}

	/**
	 * カウントアップ
	 * @param obj 更新オブジェクト
	 * @return カウントアップ後値
	 */
	public static int countup(final Serializable obj) {
		final Number num = DaoUtil.getValue(obj, ITEM_VERSION);
		if (num.intValue() == 999999) {
			return 1;
		}
		return num.intValue() + 1;
	}

	/**
	 * バージョン設定
	 * @param obj 更新オブジェクト
	 * @param ver バージョン番号
	 */
	public static void setVersion(final Serializable obj, final int ver) {
		setObject(obj, ITEM_VERSION, Integer.valueOf(ver));
	}

	/**
	 * 値設定
	 *
	 * @param obj オブジェクト
	 * @param item 項目名
	 * @param val 値
	 */
	public static void setObject(final Object obj, final String item, final Object val) {
		final Class<?> cls = obj.getClass();
		final Method get = Factory.getMethod(cls, "get" + item);
		final Method set = Factory.getMethod(cls, "set" + item, get.getReturnType());
		Factory.invoke(obj, set, val);
	}
}
