package mirrg.simulation.cart.almandine.mods.vanilla.primaries;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;

import mirrg.simulation.cart.almandine.GameAlmandine;
import mirrg.simulation.cart.almandine.facroty.property.IDialogProperty;
import mirrg.simulation.cart.almandine.factory.Entity;
import mirrg.simulation.cart.almandine.factory.IllegalEntityIdException;
import mirrg.simulation.cart.almandine.factory.Primary;
import mirrg.simulation.cart.almandine.mods.vanilla.HRender;

public abstract class PrimaryConnection<E extends Entity> extends Primary
{

	@Deprecated
	public PrimaryConnection()
	{
		super();
	}

	public PrimaryConnection(GameAlmandine game, E begin, E end)
	{
		super(game);
		this.idBegin = begin.getId();
		this.idEnd = end.getId();
	}

	public int idBegin;
	public int idEnd;

	protected E getEntity(int id) throws IllegalEntityIdException
	{
		return game.factory.getEntityOrThrow(id, getClassEntity());
	}

	/**
	 * 内部座標における単純なatan2
	 */
	public double getAngle() throws IllegalEntityIdException
	{
		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		return Math.atan2(pointEnd.y - pointBegin.y, pointEnd.x - pointBegin.x);
	}

	protected abstract Class<E> getClassEntity();

	public boolean isContained(Rectangle rectangle) throws IllegalEntityIdException
	{
		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		return rectangle.contains(pointBegin) && rectangle.contains(pointEnd);
	}

	@Override
	public boolean isHover(int x, int y) throws IllegalEntityIdException
	{
		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		return new Line2D.Double(pointBegin, pointEnd).ptSegDist(x, y) < 5;
	}

	@Override
	public void drawHover(Graphics2D graphics) throws IllegalEntityIdException
	{

		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		//

		Stroke stroke = graphics.getStroke();
		graphics.setStroke(new BasicStroke(2));

		graphics.setColor(new Color(0x7bfc82));
		HRender.drawLine(graphics, pointBegin, pointEnd, 0, 4.5, 0, 1);

		graphics.setStroke(stroke);

	}

	@Override
	public void draw(Graphics2D graphics) throws IllegalEntityIdException
	{
		drawBody(graphics);
	}

	@Override
	public void drawOverlay(Graphics2D graphics) throws IllegalEntityIdException
	{
		if (selected) drawSelectMarker(graphics);
	}

	protected abstract void drawBody(Graphics2D graphics) throws IllegalEntityIdException;

	protected void drawSelectMarker(Graphics2D graphics) throws IllegalEntityIdException
	{

		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		//

		Stroke stroke = graphics.getStroke();
		graphics.setStroke(new BasicStroke(3));

		graphics.setColor(new Color(255, 128, 0));
		HRender.drawLine(graphics, pointBegin, pointEnd, 0, 0, 0, 1);

		graphics.setStroke(stroke);

	}

	@Override
	protected void addProperty(IDialogProperty dialogProperty)
	{
		super.addProperty(dialogProperty);

		dialogProperty.addPropertyInt("IdBegin", "id", () -> idBegin, idBegin -> {

			if (!game.factory.getEntity(idBegin, getClassEntity()).isPresent()) return false;

			this.idBegin = idBegin;
			return true;
		});
		dialogProperty.addPropertyInt("IdEnd", "id", () -> idEnd, idEnd -> {

			if (!game.factory.getEntity(idEnd, getClassEntity()).isPresent()) return false;

			this.idEnd = idEnd;
			return true;
		});
	}

	@Override
	public Point getPoint() throws IllegalEntityIdException
	{
		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();
		return new Point(
			(pointBegin.x + pointEnd.x) / 2,
			(pointBegin.y + pointEnd.y) / 2);
	}

	/**
	 * マージンされたコネクション領域における指定の割合の座標。
	 */
	public Point2D.Double getPoint(double position) throws IllegalEntityIdException
	{
		Point2D.Double[] pointMargined = getPointMargined();

		Point2D.Double begin = pointMargined[0];
		Point2D.Double end = pointMargined[1];

		return new Point2D.Double(
			begin.x + (end.x - begin.x) * position,
			begin.y + (end.y - begin.y) * position);
	}

	/**
	 * 座標がホバーしている前提で、座標がこのコネクションのどのあたりの位置に相当するかを計算する。
	 */
	public double getPosition(Point point) throws IllegalEntityIdException
	{
		return getPosition(point.x, point.y);
	}

	/**
	 * 座標がホバーしている前提で、座標がこのコネクションのどのあたりの位置に相当するかを計算する。
	 */
	public double getPosition(double x, double y) throws IllegalEntityIdException
	{
		Point2D.Double[] pointMargined = getPointMargined();

		// x, y: 始点から見た点の位置
		// x2, y2: 始点から見た終点の位置
		double x2 = pointMargined[1].x - pointMargined[0].x;
		double y2 = pointMargined[1].y - pointMargined[0].y;
		x -= pointMargined[0].x;
		y -= pointMargined[0].y;

		double anglePoint = Math.atan2(y, x);
		double angleConnection = Math.atan2(y2, x2);

		// anglePoint: コネクションの角度から見た点の角度
		double lengthPoint = Math.sqrt(x * x + y * y);
		double lengthConnection = Math.sqrt(x2 * x2 + y2 * y2);
		anglePoint -= angleConnection;

		double position = Math.cos(anglePoint) * lengthPoint / lengthConnection;

		if (position <= 0) return 0;
		if (position >= 1) return 1;
		return position;
	}

	protected void drawMargined(
		Graphics2D graphics,
		Point begin,
		Point end,
		double ratioBegin,
		double ratioEnd) throws IllegalEntityIdException
	{
		HRender.drawLine(graphics, begin, end, getMarginBegin(), getMarginEnd(), 0, 1);
	}

	protected abstract double getMarginBegin() throws IllegalEntityIdException;

	protected abstract double getMarginEnd() throws IllegalEntityIdException;

	protected Point2D.Double[] getPointMargined() throws IllegalEntityIdException
	{
		Point pointBegin = getEntity(idBegin).getPoint();
		Point pointEnd = getEntity(idEnd).getPoint();

		// 全長
		double length = pointBegin.distance(pointEnd);

		// 距離ベクトル
		Point pointDistance = new Point(pointEnd.x - pointBegin.x, pointEnd.y - pointBegin.y);

		// 始点のマージン長さ
		double marginBegin = getMarginBegin();

		// 終点のマージン長さ
		double marginEnd = getMarginEnd();

		// 全長におけるマージンされた始点の比率
		double ratioBeginMargined = marginBegin / length;

		// 全長におけるマージンされた終点の比率
		double ratioEndMargined = (length - marginEnd) / length;

		// マージンされた開始地点
		Point2D.Double pointBeginMargined = new Point2D.Double(
			pointBegin.x + pointDistance.x * ratioBeginMargined,
			pointBegin.y + pointDistance.y * ratioBeginMargined);

		// マージンされた終了地点
		Point2D.Double pointEndMargined = new Point2D.Double(
			pointBegin.x + pointDistance.x * ratioEndMargined,
			pointBegin.y + pointDistance.y * ratioEndMargined);

		return new Point2D.Double[] {
			pointBeginMargined,
			pointEndMargined,
		};
	}

	@Override
	public void tick(double deltaSecond) throws IllegalEntityIdException
	{
		// validate
		getEntity(idBegin).getPoint();
		getEntity(idEnd).getPoint();
	}

}
