package mirrg.simulation.cart.almandine;

import java.awt.Graphics2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.function.Supplier;

import mirrg.simulation.cart.almandine.factory.Factory;
import mirrg.swing.helium.logging.HLog;
import mirrg.swing.helium.property.ManagerProperty;
import mirrg.swing.helium.property.PropertyDouble;
import mirrg.swing.helium.property.PropertyInteger;
import mirrg.swing.helium.property.PropertyString;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import com.thoughtworks.xstream.converters.ConversionException;

/**
 * このクラスはXStream互換です。
 */
public class GameAlmandine
{

	public ManagerProperty managerProperty = new ManagerProperty();

	@XStreamOmitField
	private Supplier<PropertyInteger> propertyTickPerFrame;

	private Supplier<PropertyInteger> getOrCreatePropertyInteger(int defaultValue, String id, String name)
	{
		return managerProperty.getOrCreateProperty(PropertyInteger::new, PropertyInteger.class, defaultValue, id, name);
	}

	private Supplier<PropertyDouble> getOrCreatePropertyDouble(Double defaultValue, String id, String name)
	{
		return managerProperty.getOrCreateProperty(PropertyDouble::new, PropertyDouble.class, defaultValue, id, name);
	}

	private Supplier<PropertyString> getOrCreatePropertyString(String defaultValue, String id, String name)
	{
		return managerProperty.getOrCreateProperty(PropertyString::new, PropertyString.class, defaultValue, id, name);
	}

	public PropertyInteger getTickPerFrame()
	{
		if (propertyTickPerFrame == null) {
			propertyTickPerFrame = getOrCreatePropertyInteger(1, "panel.tickPerFrame", "1描画フレームあたりのTick数[t/f]");
		}
		return propertyTickPerFrame.get();
	}

	@XStreamOmitField
	private Supplier<PropertyDouble> propertySecondPerTick;

	public PropertyDouble getSecondPerTick()
	{
		if (propertySecondPerTick == null) {
			propertySecondPerTick = getOrCreatePropertyDouble(2D, "panel.secondPerTick", "1Tickあたりの進行時間[s/t]");
		}
		return propertySecondPerTick.get();
	}

	@XStreamOmitField
	private Supplier<PropertyInteger> propertyOffsetX;

	public PropertyInteger getOffsetX()
	{
		if (propertyOffsetX == null) {
			propertyOffsetX = getOrCreatePropertyInteger(0, "panel.offsetX", "画面中央の座標X[px]");
		}
		return propertyOffsetX.get();
	}

	@XStreamOmitField
	private Supplier<PropertyInteger> propertyOffsetY;

	public PropertyInteger getOffsetY()
	{
		if (propertyOffsetY == null) {
			propertyOffsetY = getOrCreatePropertyInteger(0, "panel.offsetY", "画面中央の座標Y[px]");
		}
		return propertyOffsetY.get();
	}

	@XStreamOmitField
	private Supplier<PropertyDouble> propertyFPS;

	public PropertyDouble getFPS()
	{
		if (propertyFPS == null) {
			propertyFPS = getOrCreatePropertyDouble(60D, "panel.fps", "描画FPS[frame/s]");
		}
		return propertyFPS.get();
	}

	@XStreamOmitField
	private Supplier<PropertyDouble> propertyTime;

	public PropertyDouble getTime()
	{
		if (propertyTime == null) {
			propertyTime = getOrCreatePropertyDouble(0D, "factory.time", "経過時間[s]");
		}
		return propertyTime.get();
	}

	@XStreamOmitField
	private Supplier<PropertyString> propertyIdDialogProperty;

	public PropertyString getIdDialogProperty()
	{
		if (propertyIdDialogProperty == null) {
			propertyIdDialogProperty = getOrCreatePropertyString("null:null", "panel.idDialogProperty", "プロパティダイアログプロバイダID");
		}
		return propertyIdDialogProperty.get();
	}

	/////////////////////////////////////////////

	public Factory factory = new Factory();

	public void tick()
	{

		for (int i = 0; i < getTickPerFrame().value; i++) {

			factory.tick(getSecondPerTick().value);
			factory.processEditPrimaries();

			getTime().value += getSecondPerTick().value;
		}

		// 時間が流れなくてもプライマリを編集可能
		if (getTickPerFrame().value == 0) factory.processEditPrimaries();

	}

	public void render(Graphics2D graphics, IFrameGameAlmandine frameGameAlmandine)
	{
		factory.render(graphics, frameGameAlmandine);
	}

	///////////////////////////////// file /////////////////////////////////

	public boolean save(File file)
	{

		// バックアップ
		if (file.exists()) {
			Path pathBackup = new File(file.getAbsolutePath() + ".orig").toPath();
			try {
				Files.copy(
					file.toPath(),
					pathBackup,
					StandardCopyOption.REPLACE_EXISTING);
				HLog.fine("古い設定データをバックアップしました: " + pathBackup);
			} catch (IOException e) {
				HLog.processException(e, "古い設定データのバックアップに失敗しました: " + pathBackup, false);
			}
		}

		// 設定ファイル書き込み
		try {
			XStream xStream = new XStream();
			xStream.autodetectAnnotations(true);
			xStream.toXML(this, new FileOutputStream(file));
		} catch (FileNotFoundException e2) {
			HLog.processException(e2);
			return false;
		}

		HLog.fine("設定データを保存しました: " + file.getPath());
		return true;
	}

	public static GameAlmandine load(File file)
	{

		// 設定ファイル読み込み
		Object data;
		if (file.exists()) {
			try {
				data = new XStream().fromXML(new FileInputStream(file));
			} catch (FileNotFoundException e) {
				HLog.processExceptionUnexpected(e);
				return null;
			} catch (ConversionException e) {
				HLog.processException(e, "設定ファイルが破損しており読み込めませんでした！: " + file.getPath(), true);
				return null;
			}
		} else {
			HLog.error("設定ファイルが見つかりませんでした！: " + file.getPath());
			return null;
		}

		// 設定適用
		if (!(data instanceof GameAlmandine)) {
			HLog.error("設定ファイルが破損しています！: " + file.getPath());
			return null;
		}
		GameAlmandine game = (GameAlmandine) data;

		// ファクトリ初期化
		game.factory.onLoad(game);

		return game;
	}

}
