<?php
/**
 * DtoClassLoader.
 * @package magic.core
 * @subpackage internal.loader
 */
/**
 * DTOクラスのローダーです.
 * <p>
 * DTOクラスの使用は必須ではありません(推奨します)。<br/>
 * アプリケーション側でDTOクラスを使用しないのなら、
 * これは単純に無視されます。
 * </p>
 * @package magic.core
 * @subpackage internal.loader
 * @author T.Okumura
 * @version 1.0.0
 * @final
 */
final class DtoClassLoader {
    /**
     * コンストラクタ.
     */
    private function __construct() {
    }
    /**
     * DTOクラスをロードします.
     * <p>
     * アプリケーションでDTOクラスを使用しない場合はコールされません。
     * </p>
     * @param string $className ロードするクラス名
     * @param NameConfigure $name 命名規則設定クラス
     * @param PathConfigure $path パス設定クラス
     * @return Dto Dtoクラスを継承したクラス
     * @throws MagicException クラスがロードできなかった時
     * @static
     */
    public static function load($className, NameConfigure $name, PathConfigure $path) {
        $classPath = $path->getDtoPath();
        $classFile = $name->getPhpFileNamingType()->apply($className) . '.php';
        if (!@include_once $classPath . $classFile) {
            throw new MagicException(MessageBehavior::getMessage('SYS001', $classPath . $classFile));
        }
        $className = $name->getClassNamingType()->apply($className);
        if (!class_exists($className, FALSE)) {
            throw new MagicException(MessageBehavior::getMessage('SYS004', $className));
        }
        $class = new $className();
        if ($class instanceof Dto) {
            return $class;
        }
        throw new MagicException(MessageBehavior::getMessage('SYS002', array('Dto', $className)));
    }
    /**
     * DTOクラスのセッターメソッドを取得します.
     * @param Dto $dto DTOクラス
     * @return array セッターメソッドの配列
     * @static
     */
    public static function getSetterMethods(Dto $dto) {
        $result = array();
        $methods = get_class_methods($dto);
        foreach ($methods as $method) {
            if (StringUtil::startWith($method, 'set')) {
                $result[] = $method;
            }
        }
        return $result;
    }
    /**
     * DTOクラスのゲッターメソッドを取得します.
     * <p>
     * ゲッターメソッドは、getまたはisで始まる必要があります。
     * </p>
     * @param Dto $dto DTOクラス
     * @return array ゲッターメソッドの配列
     * @static
     */
    public static function getGetterMethods(Dto $dto) {
        $result = array();
        $methods = get_class_methods($dto);
        foreach ($methods as $method) {
            if (StringUtil::startWith($method, 'get') || StringUtil::startWith($method, 'is')) {
                $result[] = $method;
            }
        }
        return $result;
    }
}
// EOF.