 /*
 * org.openmicroscopy.shoola.agents.editor.view.AutosaveRecovery 
 *
 *------------------------------------------------------------------------------
 *  Copyright (C) 2006-2009 University of Dundee. All rights reserved.
 *
 *
 * 	This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *------------------------------------------------------------------------------
 */
package org.openmicroscopy.shoola.agents.editor.view;

//Java imports

import java.io.File;
import java.io.FilenameFilter;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

//Third-party libraries

//Application-internal dependencies

import org.openmicroscopy.shoola.agents.editor.EditorAgent;
import org.openmicroscopy.shoola.util.filter.file.EditorFileFilter;
import org.openmicroscopy.shoola.util.ui.MessageBox;

/** 
 * This class handles the recovery of Files generated by the auto-save
 * functionality. 
 * The {@link #checkForRecoveredFiles()} method should be called when the
 * Editor is activated for use, and if there are any files in the 
 * auto-save folder the user will be asked if they want to open them etc. 
 *
 * @author  William Moore &nbsp;&nbsp;&nbsp;&nbsp;
 * <a href="mailto:will@lifesci.dundee.ac.uk">will@lifesci.dundee.ac.uk</a>
 * @version 3.0
 * <small>
 * (<b>Internal version:</b> $Revision: $Date: $)
 * </small>
 * @since 3.0-Beta4
 */
public class AutosaveRecovery {
	
	/**
	 * Inner class used to open the recovered files on a separate thread. 
	 * Opening several files simultaneously causes concurrency problems when
	 * they are all activated at once. 
	 * 
	 * @author will
	 *
	 */
	private class FileOpener 
		implements Runnable 
	{
		private File file;
		
		/**
		 * Creates an instance.
		 * @param f		The local file to open. 
		 */
		FileOpener(File f)
		{
			file = f;
		}
		
		/**
		 * Implemented as specified by the {@link Runnable} interface.
		 * Opens the local file, sets the edited status to 'true' and deletes
		 * the file from the auto-save folder. 
		 */
		public void run() {
			Editor e = EditorAgent.openLocalFile(file, false);
			if (e != null) {
				e.setEdited(true);	// so that save button active etc. 
			}
			file.delete();
		}
	}
	
	/** 
	 * Method for checking for auto-save files for recovery.
	 */
	public void checkForRecoveredFiles()
	{
		String path = EditorAgent.getEditorAutosave();
		File folder = new File(path);
		
		// filter for Editor files (avoids hidden files etc). 
		File[] recoveredFiles = folder.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				if (name.endsWith(EditorFileFilter.CPE_XML))
					return true;
				return false;
			}
		});

		int fileCount = 0;
		if (recoveredFiles != null) {
			fileCount = recoveredFiles.length;
		}

		// if there are any files found...
		if (fileCount > 0) {
			
			String message = fileCount + " Editor file" + 
				(fileCount > 1 ? "s were " : " was ") + "recovered." +
				"\nThe application may not have shut down properly last time" +
				" it was used." +
				"\nDo you want to open the recovered files or delete them?";
			
			JFrame f = EditorAgent.getRegistry().getTaskBar().getFrame();
			MessageBox msg = new MessageBox(f, "Files Recovered", message);
			msg.addCancelButton();
			msg.setYesText("Open Files");
			msg.setNoText("Delete Files");
			msg.setCancelText("Ignore");
			
			int option = msg.centerMsgBox();
			if (option == MessageBox.YES_OPTION) {
				// open files (and delete)
				File file;
				for (int i=0; i<fileCount; i++) {
					file = recoveredFiles[i];
					
					// invoke later because opening several files at once 
					// causes problems with concurrent activation? 
					SwingUtilities.invokeLater(new FileOpener(file));
				}
			} else if (option == MessageBox.NO_OPTION) {
				// delete files
				File file;
				for (int i=0; i<fileCount; i++) {
					file = recoveredFiles[i];
					file.delete();
				}
			}
		}
	}

}
