﻿package CSVAsHash;

use strict;
use warnings;

use utf8;
use Encode;
use Data::Dumper;
use Carp;
use Text::CSV_XS;
use HTML::TreeBuilder;
#use feature ':5.10';

use Util;
use HashWithKeyCheck;

=pod
csv_enc:
    エンコード名
    CSVのエンコードを指定
=cut
sub new {

    my ($class,%arg_opt) = @_;  
	my $self;
    
    my %opt = (csv_header => 1,csv_sep_char => ',',csv_enc=> 'cp932');
    
    %opt = (%opt, %arg_opt);
    $self=\%opt;
        
    $self->{csv} = Text::CSV_XS->new ({ 
        binary      => 1,
        sep_char    => $self->{csv_sep_char}
    });
    
    return bless $self, $class;
}

########################
#プロパティ

=pod
str	","区切りの列名
=cut
sub set_col_names_from_str{
	my ($self,$header) = @_;
	
	my @col_names = split(/,/,$header);
	col_names($self,\@col_names);
	
}

sub col_names{
  my $self = shift;
  if( @_ ){ $self->{col_names} = shift }
  return $self->{col_names};
}

#########################
#オブジェクトメソッド

#一行分のhashを取得、値は空白
sub get_blank_row{
    my ($self) = @_;
    
    my $key="";
    my %data=();

    my $col_names = $self->{col_names};
    $col_names||croak "header is not set\n";
    for (my $i=0;$i<scalar(@$col_names);$i++){
        $data{$col_names->[$i]} = "";
    }
    my %ret;
    tie %ret,'HashWithKeyCheck',\%data;
    return \%ret;
}

#一行分のarrayをhashに変換
sub cols2hash{
    my ($self,$cols) = @_;
    
    my $key="";
    my %data=();

    my $col_names = $self->{col_names};
    $col_names||croak "header is not set\n";
    for (my $i=0;$i<scalar(@$col_names);$i++){
        $data{$col_names->[$i]} = $cols->[$i];
    }
    my %ret;
    tie %ret,'HashWithKeyCheck',\%data;

    return \%ret;
}

#一行分のhashをarrayに変換
sub hash2cols{
    my ($self,$data) = @_;
    
    my $key="";
    my @cols=();
    my $col_names = $self->{col_names};
    $col_names||croak "header is not set\n";
    for (my $i=0;$i<scalar(@$col_names);$i++){
        $cols[$i]=$data->{$col_names->[$i]};
    }
    return \@cols;
}

#CSVをhashのhashに読み込み
sub csv2hoh{
    my ($self,$file,$key_cols,%arg_opt) = @_;
    
    my %opt = ("action_when_key_deplicated"   => "nothing");
    %opt = (%opt,%arg_opt);
    
    my %lines=();
    my $mode = sprintf('<:encoding(%s):crlf',$self->{csv_enc});
    my $fi= IO::File->new( $file, $mode)||croak "$!:$file";
    #$fi->binmode('>:encoding(CP932):crlf');
    my $lno;
    $lno = 1;
    my $cnt =0;
    my @key_cols = split(/,/,$key_cols);
    while (my $cols = $self->{csv}->getline ($fi)){
        if ($lno < $self->{csv_header}){
            $lno++;
            next;
        }

        if ($lno == $self->{csv_header}){
            $self->{col_names}=$cols;
            $lno++;
            next;
        }
        
        my $cols_hash = cols2hash($self,$cols);
        my $key = concate_cols($cols_hash,\@key_cols);
        if (defined($lines{$key}) ){
        	if ($opt{action_when_key_deplicated} eq 'die') {
        		croak "$key:diplicated";
        	}elsif ($opt{action_when_key_deplicated} eq 'warn') {
                    #carp "$key:diplicated";        		
        	}
        }
        $lines{$key} = $cols_hash;
        $cnt++;
                
        $lno++;
    }
    $fi->close();
    
    return \%lines;
}

#CSVをhashのarrayに読み込み
sub csv2aoh{
    my ($self,$file) = @_;
    
    my @lines=();
    my $mode = sprintf('<:encoding(%s):crlf',$self->{csv_enc});
    my $fi= IO::File->new( $file, $mode)||croak "$!:$file";    
    my $lno;
    $lno = 1;
    my $cnt =0;
    while (my $cols = $self->{csv}->getline ($fi)){
        if ($lno < $self->{csv_header}){
            $lno++;
            next;
        }

        if ($lno == $self->{csv_header}){
            $self->{col_names}=$cols;
            $lno++;
            next;
        }
        
        my $cols_hash = cols2hash($self,$cols);
        push @lines, $cols_hash;
        $cnt++;
        
        $lno++;
    }
    $fi->close();
    
    return \@lines;
}

=pod
CSVから一行分のhashを読み込み

file
    CSVファイルパス
=cut
sub func_fetch_line_from_csv{
    my ($self,$file) = @_;
    my $mode = sprintf('<:encoding(%s):crlf',$self->{csv_enc});
    my $fi= IO::File->new( $file, $mode)||croak "$!:$file";
    my $lno=1;
    my $cnt =0;
    
    return sub{
        my $cols;
        if ($lno<=$self->{csv_header}) {
            while ($lno <= $self->{csv_header}){
                #ヘッダ前まで読み飛ばし
                $cols = $self->{csv}->getline ($fi);
                $lno++;
            }
            $self->{col_names}=$cols;
            $lno++;
        }
        
        $cols = $self->{csv}->getline ($fi);
        if (!$cols) {
            $fi->close();
            return undef;
        }
        my $line = cols2hash($self,$cols);
        $cnt++;
        $lno++;
        return $line;
    }
}

#hashのarray→CSV
sub aoh2csv{
    my ($self,$aoh,$file) = @_;
    
    my $mode = sprintf('>:encoding(%s)',$self->{csv_enc});
    my $fo= IO::File->new( $file, $mode)||croak $!;
    if ($self->{csv_header}!=0) {    
        #ヘッダ出力
        my $col_names=$self->{col_names};
        $col_names||die "header is not read yet\n";
        $self->{csv}->combine(@$col_names);
        $fo->printf("%s\n",$self->{csv}->string);
    }
    
    for (my $i=0;$i<scalar(@$aoh);$i++){
        my $line = hash2cols($self,$aoh->[$i]);
        $self->{csv}->combine(@$line);
        $fo->printf("%s\n",$self->{csv}->string);
    }
    $fo->close();
}

#hashのhash→CSV
sub hoh2csv{
    my ($self,$hoh,$file) = @_;
    
    my $mode = sprintf('>:encoding(%s)',$self->{csv_enc});
    my $fo= IO::File->new( $file, $mode)||croak "$!:$file";
    
    if ($self->{csv_header}!=0) {
        #ヘッダ出力
        my $col_names=$self->{col_names};
        $col_names||die "header is not read yet\n";
        $self->{csv}->combine(@$col_names);
        $fo->printf("%s\n",$self->{csv}->string);
    }
    
    my @keys = keys (%$hoh);
    for (my $i=0;$i<scalar(@keys);$i++){
        my $key = $keys[$i];
        my $line = hash2cols($self,$hoh->{$key});
        
        $self->{csv}->combine(@$line);
        $fo->printf("%s\n",$self->{csv}->string);
    }
    $fo->close();
}

###############################
#パッケージメソッド

###############################
#CSV1行分のhashの操作

=pod
指定列名の値を"."で結合する

cols
    一行分のhash
key_cols
    キーの列名のarrayref
=cut
sub concate_cols{
    my ($cols,$key_cols) = @_;
    
    my @vals=();
    for (my $i=0;$i<scalar(@$key_cols);$i++){
        push @vals,$cols->{$key_cols->[$i]};
    }
    
    return join(".",@vals);
}

###############################
#aoh操作

=pod
aohから指定列の値を"."で結合した文字列の配列を取得

arg_col_names
    列名を","区切りで指定
$arg_opt{uniq}
    1:  重複は除かれる(デフォルト）
    0:  重複を除かない
=cut
sub get_cols_values{
    my ($aoh,$arg_col_names,%arg_opt) = @_;
    
    my %opt=(uniq=>1);
    %opt=(%opt,%arg_opt);
    
    my @col_names = split(/,/,$arg_col_names);
    my @arr=();
    for (my $lno=0;$lno<scalar(@$aoh);$lno++) {
        my $line = $aoh->[$lno];
        push @arr,concate_cols($line,\@col_names);
    }
    
    if ($opt{uniq} ==1) {
        return Util::uniq_array(\@arr);
    } else{
        return \@arr;
    }
}

=pod
aohから指定列の値を"."で結合した文字列の配列を取得し、その中で重複しているものの配列を返す

arg_col_names
    列名を","区切りで指定
=cut
sub get_cols_values_duplicated{
    my ($aoh,$arg_col_names) = @_;
    
    my $arr=get_cols_values($aoh,$arg_col_names,uniq=>0);
    $arr=Util::get_duplicated_value($arr);
    
    return $arr;
}

=pod
aohからキーで削除

key_hash
	削除条件
	列名=>値
reverse
   =>0:条件に一致するものを削除
   =>1:条件に一致しないものを削除
戻り値
	削除した件数
=cut
sub delete_aoh{
    my ($aoh,$key_hash,%arg_opt) = @_;
    
    my %opt=(reverse=>0);
    %opt=(%opt,%arg_opt);
    
    my $cnt = 0;
    for (my $i=scalar(@$aoh)-1;$i>=0;$i--){
    	my $line=$aoh->[$i];
		my $bDelete=check_line($line,$key_hash);
		
		if ($opt{reverse}==1){
			$bDelete =!$bDelete; 
		}
		
		if ($bDelete==1){
			splice @$aoh,$i,1;
			$cnt++;				
		}
    }
    return $cnt;
}
=pod
一行分のhashが条件に一致するかどうかを返す

line
	一行分のhash
key_hash
	条件
	列名=>値
=cut
sub check_line{
	my ($line,$key_hash)=@_;

	my $bHit=1;
	for my $key (keys(%$key_hash)){
		if ($line->{$key} ne $key_hash->{$key}){
			$bHit=0;
			last;
		}
	}
	return $bHit;
}


=pod
aohを正規表現でフィルターしaohで返す

col_name   判定する列名
patterns   正規表現の配列
reverse
   =>0:条件に一致するものを出力
   =>1:条件に一致しないものを出力
html_text
	=>1:判定対象列をHTMLと仮定し、HTML中のテキストを判定対象とする
=cut
sub filter_aoh{
    my ($aoh,$col_name,$patterns,%arg_opt) = @_;
    
    my %opt=(reverse=>0,html_text=>0);
    %opt=(%opt,%arg_opt);
    

    my @arr=();
    for (my $i=0;$i<scalar(@$aoh);$i++){
		my $text=$aoh->[$i]->{$col_name};
		if ($opt{html_text}==1){
			my $tree = HTML::TreeBuilder->new; 
			$tree->parse($text);
			$tree->eof;
			$text = $tree->as_text();    #HTML中のテキストを抜き出す
			$tree->delete;
		}
        my $bOut = Util::is_match_patterns($text,$patterns);
        if ($opt{reverse}==1){
            $bOut=!$bOut;
        }
        if ($bOut) {
            push @arr,$aoh->[$i];
        }
    }
    
    return \@arr;
}


=pod
aohからキーで削除

key_hash
	削除条件
	列名=>値
reverse
   =>0:条件に一致するものを出力
   =>1:条件に一致しないものを出力
戻り値
	削除した件数
=cut
sub filter_aoh_by_value{
    my ($aoh,$key_hash,%arg_opt) = @_;
    
    my %opt=(reverse=>0);
    %opt=(%opt,%arg_opt);
    
    my @arr = ();
    for (my $i=scalar(@$aoh)-1;$i>=0;$i--){
    	my $line=$aoh->[$i];
		my $bHit=check_line($line,$key_hash);
		
		if ($opt{reverse}==1){
			$bHit =!$bHit; 
		}
		
		if ($bHit==1){
            push @arr,$line;
		}
    }
    return \@arr;
}

=pod
aohをソート
arg_key_cols
    ソートキー:列名を","区切りで指定
opt{dir}
    desc/asc
=cut
sub sort_aoh{
    my ($aoh,$arg_key_cols,%opt) = @_;
    
    my $hoh=aoh2hoh($aoh,$arg_key_cols);
    
    return Util::sort_hash($hoh,%opt);
}

=pod
aohからhohに変換
hashのキーは、指定列の値を"."で連結したものに行番号を付加し重複に対応する

arg_key_cols
    キー:列名を","区切りで指定

=cut
sub aoh2hoh{
    my ($aoh,$arg_key_cols) = @_;
    
    my @key_cols = split(/,/,$arg_key_cols);
    my %hoh = ();
    for (my $lno=0;$lno<scalar(@$aoh);$lno++) {
        my $line = $aoh->[$lno];
        my $key=concate_cols($line,\@key_cols);
        $key = sprintf("%s%08d",$key,$lno);
        $hoh{$key}=$line;
    }
    
    return \%hoh;
}



=pod
aohから指定列の値が一致するものを選択

aoh_in
    入力aoh
arg_in_col_names
    入力aohの指定列名を","区切りで指定
aoh_ref
    参照aoh
ref_col_names
    参照aohの指定列名を","区切りで指定。省略時はin_col_namesと同じ
reverse
   0:条件に一致するものを出力
   1:条件に一致しないものを出力
exact
    1:完全一致
    0:部分一致(aoh_refの一部に合致すれば出力)
=cut
sub select_aoh{
    my ($aoh_in,$arg_in_col_names,$aoh_ref,%arg_opt) = @_;
    
    my %opt=(ref_col_names=>$arg_in_col_names,reverse=>0,exact=>1);
    %opt=(%opt,%arg_opt);
    
    my @in_col_names = split(/,/,$arg_in_col_names);
    my @ref_col_names = split(/,/,$opt{ref_col_names});
    my $ref_values = get_cols_values($aoh_ref,$opt{ref_col_names});
    my @arr=();
    for (my $i=0;$i<scalar(@$aoh_in);$i++){
        my $bOut = 0;
        my $line=$aoh_in->[$i];
        my $val=concate_cols($line,\@in_col_names);
        if (Util::is_exist_in_array($ref_values,$val,exact=>$opt{exact})) {
            $bOut = 1;
        }
        if ($opt{reverse}==1) {
            $bOut = !$bOut;
        }
        if ($bOut) {
            push @arr,$aoh_in->[$i];
        }
    }
    
    return \@arr;
}

=pod
aohから指定列を抜き出す

aoh_in
    入力aoh
arg_col_names
    入力aohの指定列名を","区切りで指定
=cut
sub select_col_aoh{
    my ($aoh_in,$arg_col_names) = @_;
    
    my @col_names = split(/,/,$arg_col_names);
    my @arr=();
    for (my $i=0;$i<scalar(@$aoh_in);$i++){
        my $line=$aoh_in->[$i];
        my %new_line=();
        for (my $j=0;$j<scalar(@col_names);$j++){
            $new_line{$col_names[$j]}=$line->{$col_names[$j]};
        }
        
        push @arr,\%new_line;
    }
    
    return \@arr;
}

=pod
aohから指定列の値が一致するものを更新

aoh_in
    入力aoh
arg_in_col_names
    入力aohの指定列名を","区切りで指定
hoh_ref
    参照hoh
    キーは入力aohの指定列に対応する列であること
rules
    更新する場合のルールのhash
    参照hohの列名→入力aohの列名
=cut
sub update_aoh{
    my ($aoh_in,$arg_in_col_names,$ref_hoh,$rules) = @_;
    
    my @in_col_names = split(/,/,$arg_in_col_names);
    for (my $i=0;$i<scalar(@$aoh_in);$i++){
        my $line=$aoh_in->[$i];
        my $val=concate_cols($line,\@in_col_names);
        if (defined($ref_hoh->{$val})) {
            my $ref_line = $ref_hoh->{$val};
            while (my ($ref_col_name, $in_col_name) = each(%$rules)){
                $line->{$in_col_name} = $ref_line->{$ref_col_name};
            }
        }
    }
}

=pod
aohから異なるヘッダのaohにコピー

aoh
    入力aoh
out_csv
   CSVAsHashオブジェクト、ヘッダが指定されていること
rules
    in→outの対応列名のhashref
=cut
sub conv_aoh{
    my ($aoh,$out_csv,$rules) = @_;
    
    my $out_col_names   = $out_csv->col_names();
    $out_col_names || croak ("header is not set");
    my @arr=();
    for (my $i=0;$i<scalar(@$aoh);$i++){
        my $line=$aoh->[$i];
        my $out_line=$out_csv->get_blank_row;
        
        while (my ($in_col_name, $out_col_name) = each(%$rules)){
            $out_line->{$out_col_name} = $line->{$in_col_name};
        }
        
        push @arr,$out_line;
    }
    
    return \@arr;
}

1;
